// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   protocols/jd2/JobOutputter.hh
/// @brief  header file for JobOutputter class, part of August 2008 job distributor as planned at RosettaCon08
/// @author Steven Lewis smlewi@gmail.com


#ifndef INCLUDED_protocols_jd2_JobOutputter_HH
#define INCLUDED_protocols_jd2_JobOutputter_HH

//unit headers
#include <protocols/jd2/JobOutputter.hh>
#include <protocols/jd2/Job.hh>

//project headers
#include <core/pose/Pose.fwd.hh>
#include <core/util/Tracer.hh>
#include <protocols/evaluation/PoseEvaluator.hh>

//utility headers
#include <utility/pointer/ReferenceCount.hh>

//C++ headers
#include <string>

namespace protocols {
namespace jd2 {

static core::util::Tracer job_outputter_tracer("protocols.jobdist.jobs.hh");

///@details the JobOutputter class is responsible for dealing with output, as well as determining what jobs have already been output and what sort of name is associated with a job.  Derived classes will be responsible for output such as PDBS, PDBS.tar.gz and silent files.
class JobOutputter : public utility::pointer::ReferenceCount
{
public:

	//constructor -- reads cmd-line to initialize evaluators
	JobOutputter();

	virtual ~JobOutputter();

	//////////////////////////////creating output functions/////////////////////////////////////////

	///@brief this function takes a string and writes it to disk (separately from Tracer output).
	///use some sort of extention option system - default .dat?  .data?
	virtual
	void file( JobCOP job, std::string const & data ) = 0;

	///@brief this function outputs the final result of a job.
	virtual
	void final_pose( JobCOP job, core::pose::Pose const & pose ) = 0;

	///@brief this function is intended for saving mid-protocol poses; for example the final centroid structure in a combined centroid/fullatom protocol.
	virtual
	void other_pose( JobCOP job, core::pose::Pose const & pose, std::string const & tag ) = 0;

	///@brief optionally pass a starting (reference) pose to a JobOutputter for later comparison purposes and/or as interface for initializing evaluators
	virtual
	void starting_pose( core::pose::Pose const & );

	/////////////////////////////////state of output functions/////////////////////////////////

	///@brief this function is not used for output, but it belongs here since it needs to check the same output locations as the class normally writes to.  This class checks wherever output goes to see if the job's expected output already exists (on disk or whatever).  This is the most basic form of checkpointing.
	virtual
	bool job_has_completed( JobCOP job ) = 0;

	///@brief this is the master function for determining the unique output identifier for a job - should this be defined in the base class?
	virtual
	std::string output_name( JobCOP job ) = 0;

	virtual
	std::string filename( JobCOP ) const {
		return "unknown_file";
	}

	///////////////////////////////// evaluator interface ////////////////////////////////////////////
public:
  void add_evaluation( evaluation::PoseEvaluatorOP );

  void set_evaluators( evaluation::PoseEvaluators const& );

  void clear_evaluators();

  evaluation::PoseEvaluators const& evaluators() const;


protected:
  void evaluate( core::pose::Pose &pose, std::string tag, core::io::silent::SilentStruct &pss) const;

private:

  evaluation::PoseEvaluators evaluators_;
	//////////////////////////////// end evaluator interface /////////////////////////////////////////

protected:
	///@brief this function provides the extended name, not just the output name.  e.g output_name returns 1UBQ_0001, this returns 1UBQ_0001.pdb
	//not necessary in the interface class - not all derived classes CAN implement this
	//virtual
	//std::string extended_name( JobCOP job ) = 0;

	///@brief this function generates the affixed, numbered name for the job as prefix + input + suffix + number (affixes from options system).  This function is deliberately not virtual, this should be a common mechanism; your JobOutputter can not call it if it would like.
	std::string affixed_numbered_name( JobCOP job );
}; // JobOutputter

} // namespace jd2
} // namespace protocols

#endif //INCLUDED_protocols_jd2_JobOutputter_HH
