// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
// :noTabs=false:tabSize=4:indentSize=4:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   protocols/jd2/SilentFileJobOutputter.cc
/// @brief
/// @author Oliver Lange

// MPI headers
#ifdef USEMPI
#include <mpi.h> //keep this first
#endif

///Unit headers
#include <protocols/jd2/SilentFileJobOutputter.hh>

///Project headers
#include <protocols/jd2/Job.hh>
#include <core/io/silent/SilentFileData.hh>
#include <core/io/silent/SilentStructFactory.hh>
#include <core/pose/Pose.hh>

///Utility headers
#include <utility/file/FileName.hh>
#include <core/options/option.hh>
#include <core/util/Tracer.hh>
#include <core/util/prof.hh>

#include <utility/file/file_sys_util.hh>

// option key includes
#include <core/options/keys/out.OptionKeys.gen.hh>
#include <core/options/keys/run.OptionKeys.gen.hh>
#include <core/options/keys/jd2.OptionKeys.gen.hh>

#include <string>
#include <algorithm>
#include <functional>

static core::util::Tracer tr("protocols.jd2.JobOutputter");

namespace protocols {
namespace jd2 {

SilentFileJobOutputter::SilentFileJobOutputter() {
	set_defaults();
	read_done_jobs();
}

void SilentFileJobOutputter::set_defaults() {
	using namespace core::options;
	using namespace OptionKeys;

	silent_file_ = option[ out::file::silent ]();
	if ( silent_file_.relative() ) {
		silent_file_.path( option[ out::path::all ]().path() + "/" + silent_file_.path() );
		//FileName takes care of platform-specific path seperator, i.e.,  "/" or "\" ...
	}
	tr.Debug << "SilentFileJobOutputter setup for file " << silent_file_ << std::endl;
#ifdef USEMPI
	int mpi_rank;
	MPI_Comm_rank (MPI_COMM_WORLD, &mpi_rank);/* get current process id */
	std::string name = silent_file_;
	// attach mpi rank to out files
	size_t lastslash = name.find_last_of("/\\");
	size_t lastdot   = name.find_last_of('.');
	silent_file_ = name;
#endif

	// dd_parser should use binary silent files as a default. but score-only
	// silent files are also an option
	// this is really stupid. What knowledge do you about the user's intentions
	// that trumps the user's input via the command-line. A better idea would be
	// to exclude certain SilentStruct types, throw a warning or error message,
	// or do something more intelligent like that.
	if ( option[ core::options::OptionKeys::jd2::dd_parser ]() ) {
		if ( option[ out::file::silent_struct_type ]() != "score")
			option[ out::file::silent_struct_type ].value( "binary" );
	}

	bWriteIntermediateFiles_ = (
		option[ run::intermediate_scorefiles ]() ||
		option[ run::intermediate_structures ]()
	);

	bWriteIntermediateStructures_ = option[ run::intermediate_structures ]();
}

void SilentFileJobOutputter::read_done_jobs() {
	core::io::silent::SilentFileData sfd;
	if ( utility::file::file_exists( silent_file_ ) ) {
		silent_file_tags_ = sfd.read_tags_fast( silent_file_ );
	}
}

void SilentFileJobOutputter::final_pose(
	JobCOP job, core::pose::Pose const & pose
) {
	dump_pose( silent_file_, job, pose,  false /* bWriteScoresOnly */);
	dump_pose( scorefile_name(), job, pose, true );
}

/// @brief this function is intended for saving mid-protocol poses; for example
/// the final centroid structure in a combined centroid/fullatom protocol.
/// --->these go to file silent_filename+tag
void SilentFileJobOutputter::other_pose(
	JobCOP job,
	core::pose::Pose const & pose,
	std::string const & tag
) {
	utility::file::FileName filename( silent_file_ );
	filename.base( silent_file_.base() +"_"+ tag );
	if ( bWriteIntermediateFiles_ ) {
		dump_pose( filename, job, pose, !bWriteIntermediateStructures_ );
	}
}

void SilentFileJobOutputter::dump_pose(
	utility::file::FileName const & filename,
	JobCOP job,
	core::pose::Pose const & pose_in,
	bool bWriteScoreOnly
) {
	PROF_START( core::util::JD2_SILENT_OUTPUTTER );
	core::io::silent::SilentFileData sfd;

	using core::io::silent::SilentStructFactory;
	core::io::silent::SilentStructOP pss;
	if ( bWriteScoreOnly ) {
		pss = core::io::silent::SilentStructFactory::get_silent_struct("score");
	} else {
		pss = core::io::silent::SilentStructFactory::get_silent_struct_out();
	}

	std::string tag( output_name( job ) );
	pss->fill_struct( pose_in, tag );

	for( Job::StringStringPairs::const_iterator jobdata = job->output_string_string_pairs_begin();
	     jobdata != job->output_string_string_pairs_end();
			 ++jobdata
	) {
		pss->add_string_value(jobdata->first,  jobdata->second );
	}

	for( Job::StringRealPairs::const_iterator jobdata = job->output_string_real_pairs_begin();
	     jobdata != job->output_string_real_pairs_end();
			 ++jobdata
	) {
		pss->add_energy(jobdata->first, jobdata->second, 1.0 );
	}

	core::pose::Pose pose( pose_in );
	evaluate( pose, tag, *pss );
	sfd.write_silent_struct( *pss, filename );
	PROF_STOP( core::util::JD2_SILENT_OUTPUTTER );
}

// This is necessary because some protocols append prefixes to the tags and
// thus a simple string comparison will not recognise that S_1234_4 and
// C_S_1234_4 are the same
class CompareTags: public std::unary_function<std::string, bool > {
public:
	CompareTags( const std::string & querytag ): querytag_(querytag) {}

	bool operator () ( const std::string & compare_tag ) const {
		// Strings match if all the characters of the shorter string match all of the last characters of the other.
		core::Size offset1 = compare_tag.find("S_");
		if ( offset1 == std::string::npos ) offset1 = 0;
		core::Size offset2 = querytag_.find("S_");
		if ( offset2 == std::string::npos ) offset2 = 0;

		return ( compare_tag.substr(offset1) == querytag_.substr(offset2) );
	}
private:
	const std::string querytag_;
};


/////////////////////////////////state of output functions/////////////////////////////////
bool SilentFileJobOutputter::job_has_completed( JobCOP job ) {
	using namespace core::options;
	using namespace OptionKeys;

	// did we complete the job later ?
	if ( job->completed() ) {
		return true;
	}

	// was the job completed beforehand ( in the silent file before the app even
	// started ) ?
	if ( option[ run::multiple_processes_writing_to_one_directory ].value() ) {
		read_done_jobs(); // refresh silent_file_tags_ for parallel processes
	}
	CompareTags predicate(output_name( job ));
	if ( find_if (silent_file_tags_.begin(), silent_file_tags_.end(), predicate) != silent_file_tags_.end() ) {
		// job was completed before the runs was started (is present in outfile)
		return true;
	}

	return false;
}

/// @details
/// SilentFile tags should preserve the FULL NAME such that we don't end up with
/// duplicate tags. This has been a problem on BOINC till now.
std::string SilentFileJobOutputter::output_name( JobCOP job ) {
	return affixed_numbered_name( job );
}

} //jd2
} //protocols
