// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   protocols/jd2/Job.hh
/// @brief  header file for ThreadingJob classes, part of August 2008 job distributor as planned at RosettaCon08.  This file is responsible for three ideas: "inner" jobs, "outer" jobs (with which the job distributor works) and job container (currently just typdefed in the .fwd.hh)
/// @author Steven Lewis smlewi@gmail.com

#ifndef INCLUDED_protocols_jd2_ThreadingJob_HH
#define INCLUDED_protocols_jd2_ThreadingJob_HH

//unit headers
#include <protocols/jd2/ThreadingJob.fwd.hh>
#include <protocols/jd2/Job.hh>
#include <core/sequence/SequenceAlignment.hh>

//project headers
#include <core/pose/Pose.fwd.hh>

#include <protocols/jd2/ThreadingJobInputter.fwd.hh> //for friendship
#include <protocols/jd2/Parser.fwd.hh> //for friendship

#include <protocols/comparative_modeling/util.hh>

#include <core/io/silent/SilentFileData.hh>

//utility headers
#include <utility/pointer/ReferenceCount.hh>
#include <core/types.hh>
//#include <utility/vector1.hh>

//C++ headers
#include <string>
#include <list>

namespace protocols {
namespace jd2 {

/* /////////////////////////////
given that InnerThreadingJob contains data, setters, getters, but no meaningful functions, would it be better to make InnerThreadingJob
a struct with public data and then let ThreadingJob manage the data?  This would simplify function calls (only calls to ThreadingJob's
member functions, no job->inner_job()-> stuff)
yeah, I don't particular dig the inner-job outer job stuff. but that is how the jd2 design works.

or what about having InnerThreadingJob be a subclass of protocols::jobdist::BasicThreadingJob?
*/

///@details The InnerThreadingJob class is responsible for knowing input requirements for a given job - how many nstruct, and what the input is.  InnerThreadingJobs are relatively heavy; there is no need to duplicate a series of InnerThreadingJobs for each index into nstruct.  The companion ThreadingJob class handles the nstruct index and has a pointer to an InnerThreadingJob (which is shared across many ThreadingJobs).  InnerThreadingJob also holds a PoseOP to maintain the unmodified input pose for that job.
class ThreadingJob : public InnerJob {
public:
	///@brief ctor.  Note that it takes only the input tag and max nstruct,
	/// pose instantiation is deferred until the pose is needed
	ThreadingJob(
							 core::pose::PoseCOP template_pdb,
							 core::sequence::SequenceAlignmentCOP alignment,
							 std::string const & input_tag,
							 core::Size nstruct_max
	) :
		InnerJob( template_pdb, input_tag, nstruct_max ),
		alignment_( alignment )
	{	}

	///@brief the alignment for this Job
	core::sequence::SequenceAlignment const& alignment() const {
		return *alignment_;
	}

	///@brief convenience : alignment id
	std::string alignment_id() const {
		return alignment().alignment_id();
	}

	///@brief returns the "standard" loop definition (as conservative as possible)
	protocols::loops::Loops loops( core::Size nres ) const {
		return comparative_modeling::loops_from_alignment( nres, alignment(), 0 /*no use min_loop_size */);
	}
	//NOTE:
	//get_pose() will give you the parent_pdb and not the starting model, the starting model is created from the ThreadingJobInputter
	// i have done it like this, since we can have multiple parent decoys for a given alignment read in via a silent file.
	// thus, to keep track which job should work on which parent-pdb + alignment I put them here.
	// alternative: have get_pose() refer to the starting_model
	// InnerJob --> contains a sufficiently uniq id ( or pointer ) to fish out the parent_decoy when starting the job.
private:
	core::sequence::SequenceAlignmentCOP alignment_;
};

} // namespace jd2
} // namespace protocols

#endif //INCLUDED_protocols_jd2_ThreadingJob_HH
