// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// wtf is all of this incorrect? what's the point of having this if it's
/// totally wrong?
/// @file   protocols/jd2/GenericJobInputter.cc
/// @brief  August 2008 job distributor as planned at RosettaCon08 - Base class GenericJobInputter
/// @author Oliver Lange

///Unit headers
#include <protocols/jd2/ThreadingJobInputter.hh>
#include <protocols/jd2/ThreadingJob.hh>

///Project headers
#include <core/io/pdb/pose_io.hh>
#include <core/pose/Pose.hh>
#include <core/pose/PDBInfo.hh>

#include <protocols/comparative_modeling/util.hh>
#include <protocols/comparative_modeling/ThreadingMover.hh>
#include <protocols/loops/LoopClass.hh>
#include <core/sequence/util.hh>

#include <core/io/silent/SilentFileData.hh>

#include <core/chemical/ChemicalManager.hh>
#include <core/chemical/util.hh>

///Utility headers
#include <core/util/Tracer.hh>
#include <core/options/option.hh>
#include <core/options/util.hh>
#include <utility/vector1.hh>
#include <utility/file/FileName.hh>

///C++ headers
#include <string>

// option key includes

#include <core/options/keys/cm.OptionKeys.gen.hh>
#include <core/options/keys/in.OptionKeys.gen.hh>
#include <core/options/keys/out.OptionKeys.gen.hh>



static core::util::Tracer tr("protocols.jd2.ThreadingJobInputter");

namespace protocols {
namespace jd2 {

using namespace core;
using namespace pose;
using namespace core::options;
using namespace OptionKeys;

protocols::jd2::ThreadingJobInputter::ThreadingJobInputter() :
	input_source_( JobInputterInputSource::NONE )
{
 	tr.Debug << "Instantiate ThreadingJobInputter" << std::endl;

	/// read alignments from command-line
	alignments_ = sequence::read_aln( option[ cm::aln_format ](), option[ in::file::alignment ]()[1] );

	/// get template-pdbs from files
	if ( option[ in::file::template_pdb ].user() ) {
		FileList template_pdb_filenames = option[ in::file::template_pdb ]();
		tr.Trace << template_pdb_filenames[ 1 ] << std::endl;
		typedef utility::vector1< pose::PoseOP > PoseOPvec;
		PoseOPvec poses = io::pdb::poseOPs_from_pdbs( template_pdb_filenames );

		/// put template-pdbs into map --- use filename as key --- this is used to match pdb and alignment
		for ( PoseOPvec::const_iterator it = poses.begin(); it != poses.end(); ++it ) {
			utility::file::FileName fn( (*it)->pdb_info()->name() );
			std::string const base_fn( static_cast< std::string > (fn.base()) );
			std::string const match( ObjexxFCL::uppercased( base_fn.substr(0,4) ) );
			tr.Trace << "add template " << match << std::endl;
			template_poses_[ match ].push_back( *it );
		}

		input_source_ = JobInputterInputSource::PDB_FILE;

	} else if ( option[ in::file::template_silent ].user() ) { // get template-pdbs from silent-file
		io::silent::SilentFileData sfd;
		sfd.read_file( option[ in::file::template_silent ]() );
		for ( io::silent::SilentFileData::iterator it = sfd.begin(); it != sfd.end(); ++it ) {
			PoseOP pose = new Pose;
			it->fill_pose( *pose );
			std::string const match( ObjexxFCL::uppercased( it->decoy_tag().substr(2,4) ) );
			tr.Trace << "add template " << match << std::endl;
			template_poses_[ match ].push_back( pose );
			// protocols::jobdist::BasicJob = protocols::jd2::InnerJob
			// note that we are not really using the second and third fields in this
			// implementation
		}

		input_source_ = JobInputterInputSource::SILENT_FILE;

	} else { //no -in:file:template_xxx option
		//this is not a user error, since the Factory shouldn't have created a ThreadingInputter without these flags
		runtime_assert( "ThreadingJobInputter needs parent-pdbs either as in:file:template_pdb or as in:file:template_silent ");
	}
}

/// @details This function will first see if the pose already exists in the Job.
/// If not, it will read it into the pose reference, and hand a COP cloned from
/// that pose to the Job. If the pose pre-exists it just copies the COP's pose
/// into it.
void protocols::jd2::ThreadingJobInputter::pose_from_job( pose::Pose& pose, JobOP job){
	tr.Debug << "ThreadingJobInputter::pose_from_job" << std::endl;

	///cast to ThreadingJob ... to acces alignment and template pdb
	ThreadingJobCOP tjob = dynamic_cast< ThreadingJob const* const> ( job->inner_job().get() );

	pose.clear();
	std::string sequence;
	if ( option[ in::file::fasta ].user() ) {
		sequence = core::sequence::read_fasta_file( option[ in::file::fasta ]()[1] )[1]->sequence();
	} else {
		sequence = tjob->alignment().sequence( 1 )->sequence();
	}

	chemical::make_pose_from_sequence(
		pose,
		sequence,
		*( chemical::ChemicalManager::get_instance()->residue_type_set( chemical::FA_STANDARD ))
	);


	Real const alignment_coverage( tjob->alignment().length() - tjob->alignment().gapped_positions() );
	Real const alignment_identities( (core::Real)tjob->alignment().identities() / pose.total_residue() );
	Real const alignment_perc ( alignment_coverage / pose.total_residue() );



	// Add the alignment length, perc coverage and total length

	job->add_string_real_pair( "aln_len", alignment_coverage  );
	job->add_string_real_pair( "aln_perc", alignment_perc );
	job->add_string_real_pair( "aln_ident", alignment_identities );
	job->add_string_real_pair( "nres", pose.total_residue() );


	///thread to get starting model
	comparative_modeling::ThreadingMover mover( tjob->alignment(), *(tjob->get_pose()) );
	mover.build_loops( false );
	mover.randomize_loop_coords( true );
	mover.repack_query( false );
	mover.apply( pose );

	if( option[ OptionKeys::cm::start_models_only ]() ||
			tr.Debug.visible() )
	{

		std::string const alignment_id( tjob->alignment().alignment_id() );
		std::string const template_id( alignment_id.substr(0,4));
		pose.dump_pdb( alignment_id + ".pdb");
		tjob->get_pose()->dump_pdb( alignment_id + ".parent.pdb");
		//loops::Loops loops( tjob->loops( pose.total_residue() ) );
		//tr.Trace << "Standard loop definition for model " << tjob->alignment_id() << " " << loops << std::endl;
	}

}

/// @details this function determines what jobs exist from -s/-l
void protocols::jd2::ThreadingJobInputter::fill_jobs( Jobs & jobs ){
	tr.Debug << "ThreadingJobInputter::fill_jobs" << std::endl;

	jobs.clear(); //should already be empty anyway

	//read command line
	Size const nstruct( get_nstruct() );
	Real filter_threshold = -1;


	if( option[ cm::aln_length_filter_quantile ].user() ){
		Real quantile = option[ cm::aln_length_filter_quantile ]();

		// make list of lengths
		//create jobs for each alignment
		std::vector <int> length_list;
		for ( Alignments::const_iterator align_it = alignments_.begin(); align_it != alignments_.end(); ++align_it ) {
			length_list.push_back(  align_it->length() - align_it->gapped_positions() );
			tr << "Len " <<  align_it->length() - align_it->gapped_positions() << std::endl;
		}

		std::vector<int>::iterator  i = length_list.begin();
		std::vector<int>::size_type m =(size_t)( length_list.size() * quantile );

		std::nth_element(i, i + m, length_list.end());

		filter_threshold = length_list.at(m);

		tr << "Quantile filter threshold = " << filter_threshold << std::endl;
	}

	//create jobs for each alignment
	for ( Alignments::const_iterator align_it = alignments_.begin(); align_it != alignments_.end(); ++align_it ) {

		// alignment id
		std::string const alignment_id( align_it->alignment_id() );
		std::string const template_id( alignment_id.substr(0,4));
		tr.Debug << "creating job for alignment " << alignment_id << " on template " << template_id << std::endl;

		Real const alignment_coverage( align_it->length() - align_it->gapped_positions() );

		if( option[ cm::aln_length_filter ].user() ){
			filter_threshold = option[ cm::aln_length_filter ]();
		}

		if( ( filter_threshold > 0 ) && ( alignment_coverage < 	filter_threshold ) ){
			tr << "Skipping alignment " << alignment_id << ": length = " << int( alignment_coverage )
			               << "  threshold = " << int( filter_threshold ) << std::endl;
			continue;
		}

		// find matching template pdb
		PoseMap::const_iterator iter = template_poses_.find( template_id );

		if ( iter != template_poses_.end() ) {

			// found template
			PoseOPs template_poses( iter->second );

			for ( PoseOPs::const_iterator it = template_poses.begin(); it != template_poses.end(); ++it ) {
				// create inner job

				ThreadingJobOP ijob( new ThreadingJob(
										*it,
						        align_it->clone(),
										"S_"+alignment_id,
										nstruct )
				);

				// make nstruct outer jobs
				for( Size index = 1; index <= nstruct; ++index) {
					jobs.push_back( JobOP( new Job( ijob, index ) ) );
					jobs.back()->add_string_string_pair( "aln_id", alignment_id );
				}//loop over nstruct
			}
		} else { // report error if template pdb not found
			utility_exit_with_message( "ERROR: no template_pdb provided for alignment " +  alignment_id );
		}
	} //for alignments
}//fill_jobs

/// @brief Return the type of input source that the ThreadingJobInputter is currently
///  using for template structures.
JobInputterInputSource::Enum ThreadingJobInputter::input_source() const {
	return input_source_;
}

}//jd2
}//protocols
