// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is part of the Rosetta software suite and is made available under license.
// The Rosetta software is developed by the contributing members of the Rosetta Commons consortium.
// (C) 199x-2009 Rosetta Commons participating institutions and developers.
// For more information, see http://www.rosettacommons.org/.

/// @file   protocols/jd2/MPIFileBufJobDistributor.cc
/// @brief  implementation of MPIFileBufJobDistributor
/// @author Oliver Lange olange@u.washington.edu

// Unit headers
#include <protocols/jd2/archive/ArchiveBase.hh>
#include <protocols/jd2/archive/ArchiveManager.hh>

#include <core/io/silent/SilentFileData.hh>
#include <core/io/silent/SilentStructFactory.hh>

#include <core/conformation/Conformation.hh>
#include <core/pose/Pose.hh>
#include <core/scoring/Energies.hh>
#include <core/scoring/ScoreFunction.hh>
#include <core/scoring/ScoreFunctionFactory.hh>
#include <core/io/raw_data/DisulfideFile.hh>
#include <core/scoring/ResidualDipolarCoupling.hh>

#include <core/pose/Pose.hh>

#include <core/util/datacache/BasicDataCache.hh>
#include <core/util/prof.hh>
#include <core/util/Tracer.hh>

// Utility headers
#include <utility/exit.hh>
#include <utility/excn/Exceptions.hh>
#include <utility/file/file_sys_util.hh>

#include <utility/io/ozstream.hh>
#include <utility/io/izstream.hh>
#include <ObjexxFCL/formatted.o.hh>

#include <core/options/option.hh>
#include <core/options/keys/abinitio.OptionKeys.gen.hh>
#include <core/options/keys/in.OptionKeys.gen.hh>

static core::util::Tracer tr("protocols.jd2.Archive");

// Utility headers
#include <core/options/option_macros.hh>

OPT_1GRP_KEY( Boolean, archive, evaluate_only_on_slaves )
bool protocols::jd2::archive::ArchiveBase::options_registered_( false );

using namespace core::options;
using namespace OptionKeys;
//Mike: when you want to remove these Macros... leave them at least here as comment - since they provide documentation
void protocols::jd2::archive::ArchiveBase::register_options() {
	if ( !options_registered_ ) {
		options_registered_ = true;
		NEW_OPT( archive::evaluate_only_on_slaves,"do not re-evaluate decoys when they are read into archvie (e.g. on BlueGene)", false );
	}
}


namespace protocols {
namespace jd2 {
namespace archive {
// using namespace core::options;
// using namespace OptionKeys;
using namespace core;

std::string const ArchiveBase::TAG_IN_FILE( "tag_in_file" );
std::string const ArchiveBase::SOURCE_FILE( "source_file" );

ArchiveBase::ArchiveBase( ArchiveManagerAP ptr ) :
	AbstractArchiveBase( ptr ),
	accepts_since_last_batch_( 0 ),
	total_accepts_( 0 ),
	proposed_since_last_batch_( 0 ),
	total_proposed_( 0 ),
	b_evaluate_incoming_decoys_( !option[ OptionKeys::archive::evaluate_only_on_slaves ]() ), ///yields bottleneck on BG
	rdc_data_( NULL )
{}

void ArchiveBase::count_structure( jd2::archive::Batch const&, bool accepted ) {
	accepts_since_last_batch_ += accepted ? 1 : 0;
	++proposed_since_last_batch_;
}

void ArchiveBase::read_structures( core::io::silent::SilentFileData& sfd, Batch const& batch ) {
	using namespace core;
	using namespace io::silent;
	using namespace pose;
	tr.Debug << "read structures returned for " << batch.batch() << std::endl;
	Size ct( batch.decoys_returned() );
	Size accepted_ct( 0 );
	for ( SilentFileData::iterator it=sfd.begin(), eit=sfd.end(); it!=eit; ++it ) {
		Pose pose;
		std::string tag = it->decoy_tag();
		it->set_decoy_tag( batch.batch()+"_"+ObjexxFCL::lead_zero_string_of( ++ct, 6 ) );

		//note this does nothing but return *it, if b_evaluate_incoming_decoys_ is false
		core::io::silent::SilentStructOP pss = evaluate( *it );

		pss->add_comment( TAG_IN_FILE, tag );
		pss->add_comment( SOURCE_FILE, batch.silent_out() );
		bool accept = add_structure( pss, *it );
		count_structure( batch, accept );
		accepted_ct += accept ? 1 : 0;
	}

	tr.Debug << "...done reading --- " << accepted_ct << " structures were accepted into the archive" << std::endl;
}

bool ArchiveBase::add_structure( core::io::silent::SilentStructOP decoy, core::io::silent::SilentStructOP ) {
	decoys_.push_back( decoy ); //of course this can't remain as simple as this.
	++accepts_since_last_batch_;
	return true;
}

core::io::silent::SilentStructOP
ArchiveBase::evaluate( core::io::silent::SilentStructOP iss ) const {
	if ( !evaluate_local() ) {
		return iss;
	}

	core::io::silent::SilentStructOP pss = iss->clone();

	PROF_START( core::util::ARCHIVE_FILL_POSE );
	pose::Pose pose;
	pss->fill_pose( pose ); //has to reread RDC file for each pose!
	pose.data().clear();
	pose.energies().clear();
	if ( core::options::option[ core::options::OptionKeys::in::file::rdc ].user() ) {
		if ( !rdc_data_ ) rdc_data_ = new core::scoring::ResidualDipolarCoupling;
		core::scoring::store_RDC_in_pose( rdc_data_, pose );
	}
	PROF_STOP( core::util::ARCHIVE_FILL_POSE );
	return evaluate( pss, pose );
}

core::io::silent::SilentStructOP
ArchiveBase::evaluate( core::io::silent::SilentStructOP iss, core::pose::Pose& pose ) const {

	if ( core::options::option[ core::options::OptionKeys::in::fix_disulf ].user() ) {
			utility::vector1< std::pair<Size, Size> > disulfides;
			core::io::raw_data::DisulfideFile ds_file( core::options::option[ core::options::OptionKeys::in::fix_disulf ]() );
			ds_file.disulfides( disulfides, pose);
			pose.conformation().fix_disulfides( disulfides );
	}

	PROF_START( core::util::ARCHIVE_SCORE_POSE );
	score( pose );
	//	core::io::silent::SilentStructOP pss = core::io::silent::SilentStructFactory::get_silent_struct_out();
	//	iss->fill_struct( pose, tag );
	iss->energies_from_pose( pose );
	PROF_STOP( core::util::ARCHIVE_SCORE_POSE );
	//	iss->set_all_comments(
	return iss;
}

void ArchiveBase::save_to_file( std::string suffix ) {
	std::string const dirname( name() + suffix );
	std::string const filename ( dirname + "/decoys.out" );
	std::string const backup_filename ( dirname + "/decoys.out.backup" );
	std::string const tmp_filename ( dirname + "/tmp_decoys.out" );
	using namespace core::io::silent;

	//don't write empty file
	if ( !decoys().size() ) return;

	utility::file::create_directory( dirname );

	SilentFileData sfd;

	//handle output myself... so it keeps the order of decoys.
	utility::io::ozstream output( tmp_filename );
	if ( decoys_.begin() != decoys_.end() ) (*decoys_.begin())->print_header( output );

	for ( SilentStructs::const_iterator it = decoys_.begin(); it != decoys_.end(); ++it ) {
		//		sfd.add_structure( *it ); //only add OP to sfd
		//	sfd.write_silent_struct( *it, tmp_filename );
		sfd.write_silent_struct( **it, output );
	}
	//	sfd.write_all( tmp_filename );

	//rename to final
	//delete old file
	if ( utility::file::file_exists( filename ) ) {
		rename( filename.c_str(), backup_filename.c_str() );
	}
	rename( tmp_filename.c_str(), filename.c_str() );

	utility::io::ozstream status( dirname+"/STATUS" );
	save_status( status );
}

void ArchiveBase::restore_from_file() {
	std::string const& dirname( name() );
	std::string const filename ( dirname + "/decoys.out" );
	using namespace core::io::silent;
	decoys_.clear();
	if ( utility::file::file_exists( filename ) ) {
		SilentFileData sfd;
		if ( !sfd.read_file( filename ) ) throw ( utility::excn::EXCN_BadInput( "problem reading silent file"+filename ) );
		for ( SilentFileData::iterator it=sfd.begin(), eit=sfd.end(); it!=eit; ++it ) {
			decoys_.push_back( *it );
		}
	}

	utility::io::izstream status( dirname+"/STATUS" );
	if ( status.good() ) {
		restore_status( status );
	} else {
		tr.Info << name() << ": no archive status found... " << std::endl;
	}
}

void ArchiveBase::restore_status( std::istream& is ) {
	std::string line;
	getline( is, line );
	is >> total_accepts_ >> accepts_since_last_batch_ >> total_proposed_ >> proposed_since_last_batch_;
}

void ArchiveBase::save_status( std::ostream& os ) const {
	using namespace ObjexxFCL::fmt;
	os << "total_accepts accepts_since_last_batch total_proposed proposed_since_last_batch\n"
		 << RJ( 14, total_accepts_ ) << RJ( 25, accepts_since_last_batch_ )
		 << RJ( 15, total_proposed_ ) << RJ( 30, proposed_since_last_batch_) << std::endl;
}

void DebugArchive::generate_batch() {
	using namespace core::io::silent;
	++ct_batches_;
	tr.Debug << "generate batch number " << ct_batches_ << std::endl;
	if ( ct_batches_ <= 3 || decoys().size()==0 ) {

		//always call start_new_batch to generate a new batch
		Batch& batch( manager().start_new_batch() );

		utility::io::ozstream flags( batch.flag_file() );
		if ( ct_batches_ == 1 && make_mistake_ )	flags << "-abinitio::sskip_stages 1 2" << std::endl;
		flags << "-abinitio::skip_convergence_check" << std::endl;
		flags.close();

		//always finish your batch-generation with "finalize_batch"
		try {
			manager().finalize_batch( batch );
		} catch ( EXCN_Archive& excn ) {
			--ct_batches_;
			make_mistake_ = false; //don't make this mistake again
		}

	} else {
		SilentStructOPs start_decoys;
		std::copy( decoys().begin(), decoys().end(), std::back_inserter( start_decoys ) );

		Batch& batch( manager().start_new_batch( start_decoys ) );

		utility::io::ozstream broker( batch.broker_file() );
		broker << "USE_INPUT_POSE\n\nCLAIMER StartStructClaimer\nEND_CLAIMER\n" << std::endl;
		broker.close();

		manager().finalize_batch( batch );
	}
}

void DebugArchive::score( core::pose::Pose& pose ) const {
	runtime_assert( !pose.is_fullatom() );
	runtime_assert( cen_score_ );
	(*cen_score_)( pose );
}

DebugArchive::DebugArchive( ArchiveManagerAP ptr ) :
	ArchiveBase( ptr ),
	ct_batches_( 0 ),
	cen_score_( NULL ),
	make_mistake_( true )
{
	using namespace core::options;
	using namespace OptionKeys;
	using namespace core::scoring;
	if ( !cen_score_ ) cen_score_ = ScoreFunctionFactory::create_score_function( "score3", option[ OptionKeys::abinitio::stage4_patch ] );
}

void DebugArchive::save_status( std::ostream& out ) const {
	ArchiveBase::save_status( out );
	out << "nr_batches_generated: "<< ct_batches_ << std::endl;
}

void DebugArchive::restore_status( std::istream& in ) {
	ArchiveBase::restore_status( in );
	if ( in.good() ) {
		std::string tag;
		in >> tag >> ct_batches_;
	}
}

bool DebugArchive::add_structure( core::io::silent::SilentStructOP decoy ) {
	if ( decoys().size() < 100 ) {
		decoys().push_back( decoy ); //of course this can't remain as simple as this.
	} else {
		decoys().front() = decoy;
	}
	return true;
}

}//archive
}//jd2
}//protocols
