// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is part of the Rosetta software suite and is made available under license.
// The Rosetta software is developed by the contributing members of the Rosetta Commons consortium.
// (C) 199x-2009 Rosetta Commons participating institutions and developers.
// For more information, see http://www.rosettacommons.org/.

/// @file   protocols/jd2/MPIFileBufJobDistributor.cc
/// @brief  implementation of MPIFileBufJobDistributor
/// @author Oliver Lange olange@u.washington.edu


// MPI headers
#ifdef USEMPI
#include <mpi.h> //keep this first
#endif

// Unit headers
#include <protocols/jd2/archive/ArchiveManager.hh>
#include <protocols/jd2/archive/MPIArchiveJobDistributor.hh>
#include <protocols/jd2/archive/ArchiveBase.hh>

// Package headers
#include <protocols/jd2/MpiFileBuffer.hh>

// to test setup-file
#include <protocols/topology_broker/TopologyBroker.hh>
#include <protocols/topology_broker/util.hh>

//for factory
#include <protocols/abinitio/IterativeAbrelax.hh>

// Utility headers
#include <core/util/Tracer.hh>
#include <core/options/option.hh>
#include <utility/exit.hh>

#include <core/io/silent/SilentFileData.hh>

#include <ObjexxFCL/string.functions.hh>
#include <utility/file/file_sys_util.hh>

// Option headers
#include <core/options/keys/out.OptionKeys.gen.hh>
#include <core/options/keys/in.OptionKeys.gen.hh>
#include <core/options/keys/run.OptionKeys.gen.hh>
#include <core/options/keys/broker.OptionKeys.gen.hh>
#include <core/options/keys/abinitio.OptionKeys.gen.hh>

//#include <core/options/keys/jd2.OptionKeys.gen.hh>
#include <core/options/option.cc.gen.hh>

// C++ headers
#include <string>
#include <ctime>
#include <sstream>
#include <iterator>

//Debug headers
#include <protocols/abinitio/AbrelaxMover.hh>
#include <fstream> //testing
#include <utility/io/izstream.hh>

static core::util::Tracer tr("protocols.jd2.Archive");

namespace protocols {
namespace jd2 {
namespace archive {

using namespace core::options;
using namespace OptionKeys;
using namespace core;

std::string
Batch::batch() const {
	return "batch_" + ObjexxFCL::lead_zero_string_of( id(), 6 );
}

std::string Batch::dir() const {
	return batch() + "/";
}

std::string Batch::silent_out() const {
	return batch() + "/decoys.out";
}

std::string Batch::silent_in() const {
	//	if ( has_silent_in() )
	return batch() + "/decoys.in";
	//	else
	//		return "";
}

std::string Batch::score_file() const {
	return batch() + "/score.fsc";
}

std::string Batch::flag_file() const {
	return batch() + "/flags";
}

std::string Batch::broker_file() const {
	return batch() + "/setup.tpb";
}

void Batch::show( std::ostream& out, bool single_line ) const {
	std::string eol( single_line ? " " : "\n" );
	out << "ID " << id() << eol
			<< "INPUT " << ( has_silent_in() ? "yes" : "no" ) << eol
			<< "NSTRUCT " << nstruct() << eol
			<< "RETURNED " << decoys_returned() << eol
			<< "FINISHED " << ( has_finished() ? "yes" : "no" ) << eol;
}

std::ostream& operator<< (std::ostream& out, Batch const& batch ) {
	batch.show( out, true );
	return out;
}

void Batch::write_info_file() const {
	utility::io::ozstream out( dir() + "BATCH_INFO" );
	tr.Debug << "write batch info " << dir() << "BATCH_INFO" << std::endl;
	show( out, false /*not single_line*/ );
}

void Batch::read_info_file() {
	core::Size this_id = id(); //to detec errors
	std::string this_batch = batch(); //for error report
	utility::io::izstream in( dir() + "BATCH_INFO" );
	if ( !in.good() ) throw( EXCN_Archive( "cannot find " + dir() + "BATCH_INFO" ) );
	in >> *this;
	if ( this_id != id() ) {
		throw( EXCN_Archive("Inconsistency detected when reading BATCH_INFO for "+ this_batch+" ID in BATCH_INFO is " + batch() ) );
	}
}

//instead of goto statements:   I think goto would be clearer... but there are coding guidlines to adhere...
void report_tag_error( Batch& batch, std::string const& expected_tag, std::string const& tag ) {
	throw( EXCN_Archive( "Error reading batch information for batch: "+batch.batch()+" expected_tag: "+expected_tag+ " found " + tag) );
}

void report_value_error( Batch& batch, std::string const& tag ) {
	throw( EXCN_Archive( "Error reading batch information for batch: "+batch.batch()+" wrong value for tag: "+tag ) );
}

std::istream& operator >> (std::istream& in, Batch &batch ) {
	std::string tag;
	std::string expected_tag;

	in >> tag;
	expected_tag = "ID";
	if ( tag == expected_tag ) {
		in >> batch.batch_id_;
		if ( !in.good() ) report_value_error( batch, tag );
	} else report_tag_error( batch, expected_tag, tag );;

	in >> tag;
	expected_tag = "INPUT";
	if ( tag == expected_tag ) {
		std::string yesno;
		in >> yesno;
		if ( !in.good() ) report_value_error( batch, tag );
		if ( yesno == "yes" ) batch.has_silent_in_ = true;
		else if ( yesno == "no" ) batch.has_silent_in_ = false;
		else report_value_error( batch, tag );
	} else report_tag_error( batch, expected_tag, tag );;

	in >> tag;
	expected_tag = "NSTRUCT";
	if ( tag == expected_tag ) {
		in >> batch.nstruct_;
		if ( !in.good() ) report_value_error( batch, tag );
	} else report_tag_error( batch, expected_tag, tag );;

	in >> tag;
	expected_tag = "RETURNED";
	if ( tag == expected_tag ) {
		in >> batch.decoys_returned_to_archive_;
		if ( !in.good() ) report_value_error( batch, tag );
	} else report_tag_error( batch, expected_tag, tag );;

	in >> tag;
	expected_tag = "FINISHED";
	if ( tag == expected_tag ) {
		std::string yesno;
		in >> yesno;
		if ( !in.good() ) report_value_error( batch, tag );
		if ( yesno == "yes" ) batch.has_finished_ = true;
		else if ( yesno == "no" ) batch.has_finished_ = false;
	} else report_tag_error( batch, expected_tag, tag );;
	return in;
}

#ifndef WIN32

///@details constructor.  Notice it calls the parent class!  It also builds some internal variables for determining
///which processor it is in MPI land.
ArchiveManager::ArchiveManager( core::Size archive_rank, core::Size jd_master_rank, core::Size file_buf_rank ) :
  archive_rank_( archive_rank ),
	jd_master_rank_( jd_master_rank ),
	file_buf_rank_( file_buf_rank ),
	unfinished_batches_( 0 ),
	save_archive_time_interval_( 5 )
{ }

///@brief dummy for master/slave version

void
ArchiveManager::go()
{
	tr.Debug << "starting ArchiveManager ..." << archive_rank_ << " " << jd_master_rank_ << " " << file_buf_rank_ << std::endl;
	theArchive_ = new abinitio::IterativeAbrelax( this );
	try {
		restore_archive();
		read_existing_batches();
	} catch ( utility::excn::EXCN_Base& excn ) {
		send_stop_to_jobdistributor();
		throw;
	}
	//	if ( batches_.size() == 0 ) theArchive_->generate_batch();
	sleep( 5 ); //give JobDistributor time to start up...
#ifdef USEMPI
	MPI_Status status;
	MPI_Request request;
#endif
	bool stop( false );
	bool print_status( true );
	while ( !stop || unfinished_batches_ ) {
		if ( print_status ) {
			tr.Debug << "probing for message in ArchiveManager" << std::endl;
			tr.Debug << "STATUS: " << (stop ? "STOP send: " : "" ) << "  ------ unfinished_batches: " << unfinished_batches_ << std::endl;
			print_status = false;
		}
		//is there a message ?
		int flag( -1 );
#ifdef USEMPI
		//no idea why... but 4 request seems to be the magical number... to receive the correct answer ... WEIRD
		MPI_Iprobe( jd_master_rank_, MPI_ARCHIVE_TAG, MPI_COMM_WORLD, &flag, &status );
		MPI_Iprobe( jd_master_rank_, MPI_ARCHIVE_TAG, MPI_COMM_WORLD, &flag, &status );
		MPI_Iprobe( jd_master_rank_, MPI_ARCHIVE_TAG, MPI_COMM_WORLD, &flag, &status );
		MPI_Iprobe( jd_master_rank_, MPI_ARCHIVE_TAG, MPI_COMM_WORLD, &flag, &status );
		MPI_Iprobe( jd_master_rank_, MPI_ARCHIVE_TAG, MPI_COMM_WORLD, &flag, &status );
#endif
		// 	if ( !flag ) { //nothing ...
		// 			//tell JobDistributor, that we are ready to receive message
		// 			int buf[ 4 ];
		// 			buf[ 0 ] = NOTIFICATION_QUERY;
		// 			MPI_Send( &buf, 1, MPI_INT, jd_master_rank_, MPI_JOB_DIST_TAG, MPI_COMM_WORLD );
		// 			sleep( 1 );
		// 			//check if there is something this time...
		// 			MPI_Iprobe( jd_master_rank_, MPI_ARCHIVE_TAG, MPI_COMM_WORLD, &flag, &status );
		// 		}
		try {
			//if there is a message -- go get it.
			int buf[ 6 ]={0,0,0,0,0,0};
			if ( flag ) {
#ifdef USEMPI
				int merrno = MPI_Recv( &buf, 6, MPI_INT, jd_master_rank_, MPI_ARCHIVE_TAG, MPI_COMM_WORLD, &status );
				if ( merrno != MPI_SUCCESS ) tr.Error << "MPI_Recv error " << std::endl;
#endif
			}	else { //nothing received
				idle();
				continue;
			}
			print_status = true;
			// here if we got a message
			Size const msg_tag( buf[ 0 ]);
			tr.Debug << "received message in ArchiveManager " << msg_tag << std::endl;

			switch( msg_tag ) {
			case JOB_COMPLETION: {
				Size const batch_id( buf[ 1 ] );
				bool const final( buf[ 2 ] == 1 );
				Size const bad( buf[ 3 ] );
				Size const good( buf[ 4 ] );
				Size const total( buf[ 5 ] ); //total nr of jobs
				tr.Debug << "ArchiveManager received JOB_COMPLETION " << batch_id << " " << bad << " " << good << " " << total << std::endl;
				jobs_completed_[ batch_id ] = CompletionMessage( batch_id, final, bad, good, total );
				break;
			}
			case QUEUE_EMPTY:	{
				Size const batch_id( buf[ 1 ] );

				//we ignore QUEUE_EMPTY if we know that a new batch has been submitted after issuing of this signal (i.e., the batch-number
				// coming with the message would be smaller than the currently highest batch number... however, there might be invalid batches...
				// find last valid and unfinished batch in list:
				if ( batches_.size() ) {
					Size max_working_batch_id( batches_.size() );

					while ( max_working_batch_id > 0 && ( !batches_[ max_working_batch_id ].valid() || batches_[ max_working_batch_id ].has_finished() ) )
						--max_working_batch_id;
					if ( batch_id < max_working_batch_id ) {
						tr.Debug << "ArchiveManager ignored outdated QUEUE_EMPTY with batch_id " << batch_id << " -- already submitted " << batches_.size() << std::endl;
						break;
					}
				}
				//any job-completions we should work thru before generating a new batch?
				PROF_START( core::util::ARCHIVE_CRITICAL_JOBSCOMPLETE );
				while ( jobs_completed_.size() ) {
					jobs_completed(); //get thru these before making job decisions
					theArchive_->idle();
				}
				PROF_STOP( core::util::ARCHIVE_CRITICAL_JOBSCOMPLETE );

				PROF_START( core::util::ARCHIVE_GEN_BATCH );
				//this is a valid QUEUE_EMPTY request: do something about it
				tr.Debug << "ArchiveManager received QUEUE_EMPTY" << std::endl;
				if ( !theArchive_->finished() ) {
					//if !finished Archive should always generate a batch...
					//but let's make sure by monitoring, since it would be bad if we hang in the communication...
					Size ct( batches_.size() );//monitor number of batches
					theArchive_->generate_batch();
					if ( ct == batches_.size() ) { //if generate_batch didn't create anything --- we still owe Jobdistributor a signal
						send_stop_to_jobdistributor(); //send stop
						stop = true;
					}
				} else {
					tr.Debug << "archive is finished ... spinning down" << std::endl;
					send_stop_to_jobdistributor();
					stop = true;
				}
				PROF_STOP( core::util::ARCHIVE_GEN_BATCH );
				core::util::prof_show();
				break;
			}
			default:
				utility_exit_with_message( "unknown msg in ArchiveManager " + ObjexxFCL::string_of( msg_tag ) );
			}
		} catch ( utility::excn::EXCN_Base &excn ) {
			tr.Error << "[ERROR] " << excn.msg() << std::endl;
			tr.Error << "spinning down" << std::endl;
			send_stop_to_jobdistributor();
			stop = true;
		}
	} //while loop
	save_archive();
	tr.Info << "ArchiveManager finished !!!" << std::endl;
}

void
ArchiveManager::idle() {

	{ //save archive
		static time_t last_save( time(NULL) );
		time_t now( time( NULL ) );
		Size const elapsedtime( now - last_save );
		if ( elapsedtime > save_archive_time_interval_ ) {
			save_archive();
			last_save = now;
		}
	}

	//	tr.Debug << "idle..." << std::endl;
	if ( jobs_completed_.size() ) {
		PROF_START( core::util::ARCHIVE_JOBSCOMPLETE );
		jobs_completed();
		PROF_STOP( core::util::ARCHIVE_JOBSCOMPLETE );
		return;
	};

	//	if ( !theArchive_->finished() && theArchive_->ready_for_batch() ) {
		//		theArchive_->generate_batch();
	//	} else {
		time_t before( time(NULL) );
		theArchive_->idle();
		time_t after( time( NULL ) );
		if ( after-before > 1 ) tr.Debug << "spend " << after-before << " seconds in archives idle method... " << std::endl;
		//sleep some more if idle didn't use much time
		if ( after-before < 5 ) sleep( (5 - ( after - before )) );
		//	}
}

void
ArchiveManager::jobs_completed() {// core::Size batch_id, bool final, core::Size bad ) {
	runtime_assert( jobs_completed_.begin() != jobs_completed_.end() );
	CompletionMessage msg = jobs_completed_.begin()->second;
	jobs_completed_.erase( jobs_completed_.begin() );
	Size batch_id( msg.batch_id );
	bool final( msg.final );
	Size bad( msg.bad );
	Batch const& batch( batches_[ batch_id ] );
	tr.Debug << "jobs_completed for " << batch.batch() << "..." << "already "
					 << batch.decoys_returned() << " decoys known" << std::endl;
	runtime_assert( batch.id() == batch_id );
	WriteOut_MpiFileBuffer file_buf( file_buf_rank_ );
	if ( bad ) {
		///there were some bad-jobs --- we might be at the end of this run... hard to tell
	}
	PROF_START( core::util::ARCHIVE_BLOCK_FILE );
	if ( !final ) {
		tr.Debug << "not final ... block file" << std::endl;
		//careful if file isn't in FileBuf anymore... create it just for blocking ? don't block... ?
		file_buf.block_file( ".//"+batch.silent_out() ); //destructor will release file automatically
	} else {
		tr.Debug << "final ... close file " << std::endl;
		file_buf.close_file( ".//"+batch.silent_out() ); //that is not very nice, but file-buf isn't very smart with filenames...
		file_buf.close_file( ".//"+batch.score_file() );
	}
	PROF_STOP( core::util::ARCHIVE_BLOCK_FILE );
	//sleep( 5 );
	PROF_START( core::util::ARCHIVE_READ_DECOYS );
	tr.Debug << "read file " << batch.silent_out() << std::endl;
	utility::io::izstream testin( batch.silent_out() );
	tr.Debug << "stream is " << ( testin.good() ? "good " : "bad" ) << std::endl;
	if ( !testin.good() ) { //this happens sometimes... usually it needs a little bit of waiting and then it works -- NFS lag ?
		//let's look at this later...
		jobs_completed_[ batch_id ] = msg;
		sleep( 5 );
		return;
	}

	using namespace core::io::silent;
	SilentFileData sfd;
	utility::vector1< std::string > tags_in_file;

	//this keeps order as in file... important since we skip already known tags by just keeping their number
	sfd.read_tags_fast( batch.silent_out(), tags_in_file );

	if ( !final ) {
		tr.Debug << "...and release file" << std::endl;
		file_buf.release_file( ".//"+batch.silent_out() );
	}

	tr.Debug << "found " << tags_in_file.size() << " decoys in " << batch.silent_out() << std::endl;
	utility::vector1< std::string >::iterator iter = tags_in_file.begin();
	for ( Size ct = 1;
				iter != tags_in_file.end() && ct <= batch.decoys_returned(); ++iter, ++ct ) { }; //just skipping...
	utility::vector1< std::string > tags_to_read;

	std::copy( iter, tags_in_file.end(), std::back_inserter( tags_to_read ) );
	if ( tags_to_read.size() ) {
		sfd.read_file( batch.silent_out(), tags_to_read );
		PROF_STOP( core::util::ARCHIVE_READ_DECOYS );
		tr.Debug << "add " << tags_to_read.size() << " structures to archive " << std::endl;

		PROF_START( core::util::ARCHIVE_EVAL_DECOYS );
		//read structures and add to archive
		theArchive_->read_structures( sfd, batch );
		PROF_STOP( core::util::ARCHIVE_EVAL_DECOYS );
	} else {
		tr.Info << "no more decoys to read from file " << batch.silent_out() << std::endl;
		PROF_STOP( core::util::ARCHIVE_READ_DECOYS );
	}

	{ //now update our batch information and save to disck
		Batch& batch( batches_[ batch_id ] );
		batch.set_decoys_returned( tags_in_file.size() );
		if ( final ) {
			batch.mark_as_finished();
			--unfinished_batches_;
		}
		batch.write_info_file();
	}
	PROF_START( core::util::SAVE_ARCHIVE );
	if ( jobs_completed_.size() == 0 ) save_archive();
	PROF_STOP( core::util::SAVE_ARCHIVE );
}

void
ArchiveManager::queue_batch( Batch const& batch ) {
	tr.Debug << "queue new batch into MPIArchiveJobDistributor " << batch.flag_file() << std::endl;
	Size const size( 3 );
	int buf[ size ];
	buf[ 0 ] = ADD_BATCH;
	buf[ 1 ] = batch.id();
	buf[ 2 ] = batch.nstruct();
#ifdef USEMPI
	MPI_Send( &buf, size, MPI_INT, jd_master_rank_, MPI_JOB_DIST_TAG, MPI_COMM_WORLD );
	//need to have MPI_JOB_DIST_TAG... since it goes into main msg-loop of JobDist

	//send size of string
	std::string strbuf( batch.flag_file() );
	buf[ 0 ] = strbuf.size();
	buf[ 1 ] = batch.id();
	MPI_Send( buf, 2, MPI_INT, jd_master_rank_, MPI_JOB_DIST_TAG, MPI_COMM_WORLD );
	//send string
	MPI_Send(const_cast<char*> ( strbuf.data() ), strbuf.size(), MPI_CHAR, jd_master_rank_, MPI_JOB_DIST_TAG, MPI_COMM_WORLD );
#else
	protocols::jd2::JobDistributor::get_instance()->add_batch( batch.flag_file() );
#endif

}

void ArchiveManager::send_stop_to_jobdistributor() {
	//we do this by sending empty batch.
	tr.Debug << "send STOP signal to JobDistributor " << std::endl;
	Batch stop_batch( 0 );
	queue_batch( stop_batch );
}

void
ArchiveManager::read_existing_batches() {
	using utility::file::file_exists;
	using namespace core::options::OptionKeys;

	//possible options:
	//reads all structures from batches as if they were newly coming in
	bool b_reread_all_structures( option[ OptionKeys::archive::reread_all_structures ]() /* default false */ );

	//don't know how to probe directory... take counter...
	core::Size id( 1 );
	Batch aBatch( id );
	batches_.clear();
	while ( file_exists( aBatch.flag_file() ) ) {
		Batch& new_batch( start_new_batch() );
		runtime_assert( new_batch.id() == id );
		tr.Info << "found existing batch " << new_batch.batch() << std::endl;
		try {
			finalize_batch( new_batch, true /*reread */ );
			tr.Debug << new_batch << std::endl;
		} catch ( EXCN_Archive& excn ) {
			//last started batch must have problems... ignore it
			tr.Warning << "[ WARNING ] "+new_batch.batch()+" is errorneous: " + excn.msg() << std::endl;
			tr.Warning << "[ WARNING ] ignoring this batch..." << std::endl;
			//fill batch list if exception state has left us without it
			if ( batches_.size() < id ) batches_.push_back( Batch( id ) );
			batches_.back().mark_as_invalid();
		}
		if ( b_reread_all_structures ) {
			if ( batches_[ id ].decoys_returned() ) {
				jobs_completed_[ id ] =
					CompletionMessage( id, batches_[id ].has_finished(), 0, batches_[ id ].decoys_returned(), batches_[ id ].nstruct() );
			}
			batches_[ id ].set_decoys_returned( 0 );
			if ( batches_[ id ].has_finished() ) { //if it is not finished we will later queue the batch and the counter will be incremented.
				unfinished_batches_ += 1; //increment because we will decrement when we work on the CompletionMessage
			}
		}
		aBatch = Batch( ++id );
	}
}

Batch&
ArchiveManager::start_new_batch() {
	core::io::silent::SilentStructOPs empty;
	return start_new_batch( empty );
}

Batch&
ArchiveManager::start_new_batch( core::io::silent::SilentStructOPs const& start_decoys ) {
	using utility::file::file_exists;

	core::Size batch_id( batches_.size() + 1 );
	tr.Debug << "start new batch " << batch_id << std::endl;
	batches_.push_back( Batch( batch_id ) );
	Batch &new_batch( batches_.back() );

	new_batch.set_id( batch_id );
	//make directory:
	utility::file::create_directory( new_batch.dir() );
	if ( start_decoys.size() ) {
		new_batch.set_has_silent_in();
		core::io::silent::SilentFileData sfd;
		for ( core::io::silent::SilentStructOPs::const_iterator
						it = start_decoys.begin(); it != start_decoys.end(); ++it ) {
			sfd.add_structure( **it );
		}
		sfd.write_all( new_batch.silent_in() );
	}
	new_batch.user_options().add_built_in_options();
	add_all_rosetta_options( new_batch.user_options() );

	//copy the system broker setup
	if ( !file_exists( new_batch.broker_file() ) && option[ OptionKeys::broker::setup ].user() ) {
		utility::io::ozstream batch_broker( new_batch.broker_file() );
		utility::io::izstream system_broker( option[ OptionKeys::broker::setup ]() );
		std::string line;
		while ( getline( system_broker, line ) ) batch_broker << line << std::endl;
	}

	new_batch.nstruct() = core::options::option[ core::options::OptionKeys::out::nstruct ];
	return batches_.back();
}

void report_batch_inconsistency( Batch& new_batch, std::string const &tag ) {
	throw( EXCN_Archive( "inconsistency detected when re-reading "+new_batch.batch()+" for " + tag) );
}

void
ArchiveManager::finalize_batch( Batch& new_batch, bool reread ) {
	using utility::file::file_exists;
	using namespace core::options::OptionKeys;
	tr.Debug << "finalize_batch " << new_batch << std::endl;
	if ( !reread) new_batch.set_decoys_returned( 0 );

	if ( !utility::file::file_exists( new_batch.broker_file() ) ) {
		utility::io::ozstream broker( new_batch.broker_file() );
		broker << "# NO CLAIMERS PRESENT" << std::endl;
		broker.close();
	}

	if( file_exists( new_batch.flag_file() ) ) {
		tr.Debug << "checking aBatch.flag_file()... " << std::endl;
		utility::options::OptionCollection batch_opts;
		batch_opts.add_built_in_options();
		add_all_rosetta_options( batch_opts );
		try {
			tr.Debug << "load options from file" << std::endl;
			batch_opts.load_options_from_file_exception( new_batch.flag_file() );
		} catch ( utility::excn::EXCN_Msg_Exception &excn ) {
			tr.Error << "[ERROR] problems with flags in " << new_batch.flag_file() << " aborting... " << std::endl;
			// excn.show( tr.Error );
			batches_.pop_back();
			throw ( EXCN_Archive( new_batch.flag_file() + " contains errors: " + excn.msg() ) );
		}
		if ( !reread )  {
			//access all archive controlled options... so they are not in the "user_flags" anymore
			if ( batch_opts[ in::file::silent ].user() )
				tr.Warning << "option -in:file:silent will be overwritten by ArchiveMaster"
									 << " -- control directly via class Batch" << std::endl;
			if ( batch_opts[ out::nstruct ].user() )
				tr.Warning << "option -nstruct will be overwritten by ArchiveMaster "
									 << "-- control directly via class Batch" << std::endl;
			if ( batch_opts[ run::intermediate_structures ].user() )
				tr.Warning << "option -run::intermediate_structures will be overwritten by ArchiveMaster "
									 << "-- control directly via class Batch" << std::endl;
			if ( batch_opts[ out::file::silent ].user() )
				tr.Warning << "option -out:file:silent will be overwritten by ArchiveMaster "
									 << "-- control directly via class Batch" << std::endl;
			if ( batch_opts[ broker::setup ].user() )
				tr.Warning << "option -broker:setup will be overwritten by ArchiveMaster "
									 << "-- control directly via class Batch" << std::endl;
			if ( batch_opts[ out::file::scorefile ].user() )
				tr.Warning << "option -out:file:scorefile will be overwritten by ArchiveMaster "
									 << "-- control directly via class Batch" << std::endl;
		}

		bool has_silent( batch_opts[ in::file::silent ].user() );
		core::Size nstruct( batch_opts[ out::nstruct ]() );
		bool intermeds( batch_opts[ run::intermediate_structures ]() );
		std::string silent_out( batch_opts[ out::file::silent ]() );
		std::string broker( batch_opts[ broker::setup ]() );
		std::string score_file( batch_opts[ out::file::scorefile ]() );

		// now the other options are "inaccessed options" and can be dumped to a stream
		std::stringstream user_flags;
		batch_opts.show_inaccessed_user_options( user_flags );
		tr.Debug << "user_options: \n" << user_flags.str() << std::endl;

		// and can be added to the batch-options
		new_batch.user_options().load_options_from_stream( user_flags, "USER_FLAGS" );
		if ( reread ) {
			new_batch.read_info_file();
			new_batch.set_intermediate_structs( intermeds ); //this is not read from BATCH_INFO

			// for all other values we just double-check consistency
			if ( new_batch.nstruct() != nstruct ) report_batch_inconsistency( new_batch, "NSTRUCT" );
			if ( new_batch.has_silent_in() !=  has_silent ) report_batch_inconsistency( new_batch, "INPUT" );
			if ( silent_out != new_batch.silent_out() ) report_batch_inconsistency( new_batch, "OUTPUT" );
			if ( broker != new_batch.broker_file() ) report_batch_inconsistency( new_batch, "BROKER_FILE" );
			//TODO: determine how many decoys have been returned to archive...
		}
	}

	//now write the final flag-file
	utility::io::ozstream flag_out( new_batch.flag_file() );
	new_batch.user_options().show_user( flag_out );
	flag_out << "\n\n#Archive controlled flags" << std::endl;
	flag_out << "-out:file:silent " << new_batch.silent_out() << std::endl;
	if ( new_batch.has_silent_in() ) flag_out << "-in:file:silent " << new_batch.silent_in() << std::endl;

	flag_out << "-out:nstruct " << new_batch.nstruct() << std::endl;
	flag_out << "-out:file:scorefile " << new_batch.score_file() << std::endl;
	flag_out << "-broker:setup " << new_batch.broker_file() << std::endl;
	if ( new_batch.intermediate_structs() ) flag_out << "-run:intermediate_structures" << std::endl;
	if ( !reread ) {
		try {
			test_broker_settings( new_batch );
		} catch ( utility::excn::EXCN_Msg_Exception &excn ) {
			tr.Error << "[ERROR] problems with broker setup in " << new_batch.broker_file() << " aborting... " << std::endl;
			// excn.show( tr.Error );
			batches_.pop_back();
			throw ( EXCN_Archive( new_batch.broker_file() + " contains errors: " + excn.msg() ) );
		}
	}

	if ( !reread ) {
		new_batch.write_info_file();
	}

	if ( !new_batch.has_finished() && theArchive_->still_interested( new_batch ) ) {
		tr.Debug << "queue " << new_batch.batch() << " " << new_batch.flag_file() << std::endl;
		queue_batch( new_batch );
		unfinished_batches_ += 1;
	} else {
		new_batch.mark_as_finished();
	}

	tr.Debug << "\n" << std::endl;

}

void
ArchiveManager::test_broker_settings( Batch const& batch ) {
	tr.Debug << "test broker settings...." << std::endl;
	OptionCollection vanilla_options( option );
  option.load_options_from_file( batch.flag_file() );
	try {
		topology_broker::TopologyBrokerOP topology_broker = new topology_broker::TopologyBroker();
		topology_broker::add_claims_from_file( *topology_broker, option[ OptionKeys::broker::setup ]() );
	} catch ( utility::excn::EXCN_Exception &excn ) {  // clean up options and rethrow
		option = vanilla_options;
		throw; //rethrow exception
	}
	option = vanilla_options;
}

void
ArchiveManager::save_archive() {
	theArchive_->save_to_file();
}


void
ArchiveManager::restore_archive() {
	theArchive_->restore_from_file();
}

#endif //ndef WIN32

}//archive
}//jd2
}//protocols


