// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is part of the Rosetta software suite and is made available under license.
// The Rosetta software is developed by the contributing members of the Rosetta Commons consortium.
// (C) 199x-2009 Rosetta Commons participating institutions and developers.
// For more information, see http://www.rosettacommons.org/.

/// @file   protocols/jd2/MPIWorkPoolJobDistributor.hh
/// @brief  header for MPIWorkPoolJobDistributor - intended for continuous resamplig jobs  that spawn new jobs based on a pool/archive of
///         structures
/// @author Oliver Lange olange@u.washington.edu

#ifndef INCLUDED_protocols_jd2_archive_ArchiveManager_HH
#define INCLUDED_protocols_jd2_archive_ArchiveManager_HH

// Unit headers
#include <protocols/jd2/archive/ArchiveManager.fwd.hh>
#include <protocols/jd2/archive/ArchiveBase.hh>
#include <protocols/jd2/archive/MPIArchiveJobDistributor.hh>

// Package headers
#include <protocols/jd2/JobDistributor.hh>
#include <protocols/jd2/Job.fwd.hh>
#include <protocols/jd2/JobDistributorFactory.hh>
#include <core/io/silent/silent.fwd.hh>


#include <protocols/moves/Mover.fwd.hh>
#include <utility/options/OptionCollection.hh>
#include <utility/excn/Exceptions.hh>

// Utility headers
#include <core/types.hh>

// C++ headers
#include <string>

namespace protocols {
namespace jd2 {
namespace archive {

class EXCN_Archive : public utility::excn::EXCN_Msg_Exception {
public:
	EXCN_Archive( std::string const& msg ) : EXCN_Msg_Exception( msg ) {};
};


class Batch {
	public:

	Batch( utility::options::OptionCollection const& options, bool intermediate_structs, bool has_silent_in, core::Size nstruct )
		: nstruct_( nstruct ),
			intermediate_structs_( intermediate_structs ),
			has_silent_in_( has_silent_in ),
			has_finished_( false ),
			invalid_( false ),
			options_( options ),
			decoys_returned_to_archive_( 0 )
	{};

	Batch( core::Size id ) : batch_id_( id ),
													 nstruct_( 0 ),
													 intermediate_structs_( false ),
													 has_silent_in_( false ),
													 has_finished_( false ),
													 invalid_( false ),
													 decoys_returned_to_archive_( 0 )

	{};


	std::string silent_out() const;
	std::string silent_in() const;
	std::string score_file() const;
	std::string flag_file() const;
	std::string batch() const;
	std::string dir() const;
	std::string broker_file() const;

	utility::options::OptionCollection const& user_options() const { return options_; };
	utility::options::OptionCollection& user_options() { return options_; };

	///Getters
	bool has_silent_in() const { return has_silent_in_; }

	bool intermediate_structs() const { return intermediate_structs_; };

	bool has_finished() const { return has_finished_; }

	core::Size& nstruct() { return nstruct_; }

	core::Size nstruct() const { return nstruct_; }

	core::Size id() const { return batch_id_; }

	core::Size decoys_returned() const { return decoys_returned_to_archive_; }

	///Setters
	void set_has_silent_in( bool setting = true ) {
		has_silent_in_ = setting;
	}

	void set_intermediate_structs( bool setting = true ) {
		intermediate_structs_ = setting;
	}

	void mark_as_finished() {
		has_finished_ = true;
	}

	void mark_as_invalid() {
		invalid_ = true;
	}

	bool valid() {
		return !invalid_;
	}

	void set_id( core::Size id ) {
		batch_id_ = id;
	}

	void set_decoys_returned( core::Size setting ) {
		decoys_returned_to_archive_ = setting;
	}

	//input-output: batch_id, nstruct, has_finished, decoys_returned...
	void show( std::ostream&, bool single_line = false ) const;
	friend std::ostream& operator<< (std::ostream&, Batch const& );
	friend std::istream& operator>> (std::istream&, Batch & );

	void write_info_file() const;
	void read_info_file();
private:
	core::Size batch_id_;
	core::Size nstruct_;
	bool intermediate_structs_;
	bool has_silent_in_;
	bool has_finished_;
	bool invalid_;
	utility::options::OptionCollection options_;
	core::Size decoys_returned_to_archive_;
};


#ifndef WIN32

class ArchiveManager {
public:
  ///@brief ctor is protected; singleton pattern
  ArchiveManager( core::Size archive_rank, core::Size jd_master_rank, core::Size file_buf_rank );
	virtual ~ArchiveManager() {}; //virtual destructor because we have virtual functions
public:

  void go();

	Batch& start_new_batch( core::io::silent::SilentStructOPs const& start_decoys );
	Batch& start_new_batch();
	//now write flags to batch.flag_file()
	//put claimers into batch.broker_file()
	//put stuff into directory batch.dir()

	void finalize_batch( Batch&, bool reread = false );
		 //this will read options to check
		 //it will read the broker_file and check
		 //register and queue the option...
	core::Size last_batch_id() const {
		return batches_.size();
	}


protected:
	///@brief triggered in slave if new batch_ID comes in.
	virtual void batch_underflow() {};

// 	///@brief return true if message was understood
// 	virtual bool process_message( int msg_tag, int slave_rank, int slave_job_id );

	void idle();

	void jobs_completed();// core::Size batch_id, bool final, core::Size bad );
	void queue_batch( Batch const& batch );
	void read_existing_batches();
	void register_batch( Batch new_batch );
	void send_stop_to_jobdistributor();
	void save_archive();
	void restore_archive();

	void test_broker_settings( Batch const& batch );

  friend class JobDistributorFactory; //ctor access
private:

	utility::vector1< Batch > batches_;

	core::Size const archive_rank_;
	core::Size const jd_master_rank_;
	core::Size const file_buf_rank_;

	AbstractArchiveBaseOP theArchive_;
	typedef MPIArchiveJobDistributor::CompletionMessage CompletionMessage;
	typedef	std::map< core::Size, CompletionMessage > CompletionMessages;
	CompletionMessages jobs_completed_;
	core::Size unfinished_batches_;

	///@brief specify seconds between automatic saves to filesystem
	core::Size save_archive_time_interval_;
};



//compile on WINDOWS :makes dummy class
#else

// DUMY CLASS SO that windows compiles. I dont know why it doesnt compile this class,
// it just hangs on ArchiveManager.cc and times out. Since the Archive is not used on BOINC anyway, this is all currently disabled.
// Contact Oliver Lange if you want to run MPI Archive style jobs on WIndows machines.
class ArchiveManager
{
public:
	ArchiveManager( core::Size archive_rank, core::Size jd_master_rank, core::Size file_buf_rank ) {};
	virtual ~ArchiveManager() {}; //virtual destructor because we have virtual functions
	void go(){};

	Batch& start_new_batch( core::io::silent::SilentStructOPs const& start_decoys ){
		// This code will break. dont use it. It's just to satisfy the compiler!
		Batch* newbatch = new Batch( 1 ); return *newbatch; };

	Batch& start_new_batch(){
		// This code will break. dont use it. It's just to satisfy the compiler!
		Batch* newbatch = new Batch( 1 ); return *newbatch; };

	void finalize_batch( Batch&, bool reread = false ) {};

	core::Size last_batch_id() const {
		return 0;
	}
protected:
  friend class JobDistributorFactory; //ctor access
private:
};

#endif //WIN32

}//archive
}//jd2
}//protocols

#endif //INCLUDED_protocols_jd2_ArchiveManager_HH
