// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is part of the Rosetta software suite and is made available under license.
// The Rosetta software is developed by the contributing members of the Rosetta Commons consortium.
// (C) 199x-2009 Rosetta Commons participating institutions and developers.
// For more information, see http://www.rosettacommons.org/.

/// @file   protocols/jd2/MPIFileBufJobDistributor.cc
/// @brief  implementation of MPIFileBufJobDistributor
/// @author Oliver Lange olange@u.washington.edu

// Unit headers
#include <protocols/jd2/archive/EvaluatedArchive.hh>
#include <protocols/jd2/archive/ArchiveManager.hh>

#include <core/io/silent/SilentFileData.hh>
#include <core/io/silent/SilentStructFactory.hh>

#include <core/scoring/ScoreFunction.hh>
#include <core/scoring/ScoreFunctionFactory.hh>

#include <core/pose/Pose.hh>

#include <protocols/evaluation/ConstraintEvaluator.hh>
#include <protocols/evaluation/util.hh>

#include <utility/exit.hh>
#include <utility/excn/Exceptions.hh>
#include <utility/file/file_sys_util.hh>
#include <core/util/prof.hh>

//for DebugArchive
#include <utility/io/ozstream.hh>
#include <utility/io/izstream.hh>

#include <core/options/option.hh>
//#include <core/options/keys/abinitio.OptionKeys.gen.hh>

#include <core/options/keys/run.OptionKeys.gen.hh>

//for setup_default_evaluators
#include <core/options/keys/constraints.OptionKeys.gen.hh>

static core::util::Tracer tr("protocols.jd2.Archive");

namespace protocols {
namespace jd2 {
namespace archive {
// using namespace core::options;
// using namespace OptionKeys;
using namespace core;
using namespace core::io::silent;


bool EvaluatedArchive::add_structure( core::io::silent::SilentStructOP decoy, core::io::silent::SilentStructOP ) {
	if ( decoys().size() < nstruct() ) { //take all decoys until full
		decoys().push_back( decoy );
		return true;
	} else { //now we are more selective
		Real new_decoy_score ( select_score(decoy) );
		SilentStructs::iterator iss = decoys().begin();
		while ( iss != decoys().end() && new_decoy_score >= select_score( *iss ) ) ++iss;
		if ( iss != decoys().end() ) {
			decoys().insert( iss, decoy );
			decoys().pop_back();
			return true;
		}
	}
	return false;
}

void EvaluatedArchive::read_structures( core::io::silent::SilentFileData& sfd, Batch const& batch ) {
	tr.Info << "structures are scored with the following weights: " << std::endl;
	for ( WeightMap::const_iterator it=select_weights_.begin(); it != select_weights_.end(); ++it ) {
		std::string const& name( it->first );
		core::Real const& weight( it->second );
		tr.Info << name << " " << weight << std::endl;
	}
	Parent::read_structures( sfd, batch );
}

core::io::silent::SilentStructOP
EvaluatedArchive::evaluate( core::io::silent::SilentStructOP iss, core::pose::Pose& pose ) const {
	Parent::evaluate( iss, pose );
	PROF_START( core::util::ARCHIVE_EVALUATORS );
	tr.Trace << "evaluate decoy " << iss->decoy_tag() << std::endl;
 	for ( EvaluatorMap::const_iterator it=evaluators_.begin(), eit=evaluators_.end();
				it!=eit; ++it ) {
		it->second->apply( pose, iss->decoy_tag(), *iss );
	}
	PROF_STOP( core::util::ARCHIVE_EVALUATORS );
	return iss;
}

void EvaluatedArchive::score( pose::Pose& pose ) const {
	runtime_assert( scorefxn_ );
	(*scorefxn_)( pose );
}

class SortPredicate {
public:
	SortPredicate( EvaluatedArchive& arc ) : arc_( arc ) {};
	bool operator() (SilentStructOP const& pss1, SilentStructOP const& pss2 ) {
		return arc_.select_score( pss1 ) < arc_.select_score( pss2 );
	}
	EvaluatedArchive& arc_;
};

void EvaluatedArchive::rescore() {
	tr.Debug << "rescore " << name() << " decoys " << std::endl;
	for ( SilentStructs::iterator iss = decoys().begin(); iss != decoys().end(); ++iss ) {
		*iss = Parent::evaluate( *iss );
	}
	scores_are_clean_ = true; //do this first, since SortPredicate asks for the select_score

	if ( tr.Trace.visible() ) {
		SilentFileData sfd;
		//handle output myself... so it keeps the order of decoys.
		utility::io::ozstream output( "freshly_rescored.out" );
		if ( decoys().begin() != decoys().end() ) (*decoys().begin())->print_header( output );

		for ( SilentStructs::const_iterator it = decoys().begin(); it != decoys().end(); ++it ) {
			sfd.write_silent_struct( **it, output, true );
		}
	}

	tr.Debug << "sort freshly scored decoys " << std::endl;
	decoys().sort( SortPredicate( *this ) );
	tr.Debug << "...done rescoring and sorting " << std::endl;
	if ( tr.Trace.visible() ) {
		for ( SilentStructs::const_iterator it = decoys().begin(); it != decoys().end(); ++it ) {
			tr.Trace << select_score( *it ) << " " << (*it)->decoy_tag() << std::endl;
		}
	}
}

Real EvaluatedArchive::select_score( SilentStructOP pss ) {
	if ( !scores_are_clean_ && evaluate_local() ) rescore();
	if ( pss->has_energy( "_archive_select_score_" ) ) {
		return pss->get_energy( "_archive_select_score_" );
	}
	Real sum( 0.0 );
	for ( WeightMap::const_iterator it=select_weights_.begin(); it != select_weights_.end(); ++it ) {
		std::string const& name( it->first );
		Real const& weight( it->second );
		if ( weight == 0.0 ) continue;
		if ( name == "nrjump_weighted_chainbreaks" ) { //special energy term
			//			runtime_assert( pss->has_energy( "nrjumps" ) );
			core::Real const nrjump( pss->get_energy ( "nrjumps" ) );
			if ( nrjump == 0 ) continue;
			//		runtime_assert( pss->has_energy( "overlap_chainbreak" ) );
			//		runtime_assert( pss->has_energy( "linear_chainbreak" ) );
			core::Real const cb( pss->get_energy( "linear_chainbreak" ) + pss->get_energy( "overlap_chainbreak" ) );
			sum += weight * ( cb/std::pow(nrjump,1.5) );
		} else {
			if ( !pss->has_energy( name ) ) {
				throw EXCN_Archive( "energy name "+name+" not found in returned decoys -- run with rescoring in archive to avoid this or fix your batches" );
			}
			sum += weight * pss->get_energy( name );
		}
	}
	pss->add_energy( "_archive_select_score_", sum );
	return sum;
}

void EvaluatedArchive::add_evaluation( evaluation::PoseEvaluatorOP eval, Real weight ) {
	for ( Size i=1; i<= eval->size(); i++ ) {
		std::string const& column( eval->name( i ) );
		select_weights_[ column ] = weight;
		evaluators_[ column ] = eval;
	}
	scores_are_clean_ = false;
}

void EvaluatedArchive::set_weight( std::string const& column, core::Real weight ) {
	//	runtime_assert( has_evaluator( column ) ); or part of score!
	select_weights_[ column ] = weight;
}

bool EvaluatedArchive::has_evaluator( std::string const& column ) {
	EvaluatorMap::const_iterator iter = evaluators_.find( column );
	return iter != evaluators_.end();
}

void EvaluatedArchive::set_scorefxn( core::scoring::ScoreFunctionOP scorefxn ) {
	scorefxn_ = scorefxn;
	scores_are_clean_ = false;
}

void EvaluatedArchive::setup_default_evaluators() {
	using namespace core::options;
	using namespace OptionKeys;
	using namespace scoring::constraints;
	if ( evaluate_local() && option[ constraints::cst_file ].user() ) {
		std::string filename( option[ constraints::cst_file ]()[ 1 ] );
		evaluation::PoseEvaluatorOP ev_cst ( new evaluation::ConstraintEvaluator( "cmdline", filename ) );
		add_evaluation( ev_cst, option[ constraints::cst_weight ] );
		//The ConstraintEvaluator creates two columns: cmdline_cst and cmdline_viol...
		set_weight( "cmdline_viol", 0.0 );
	}
	set_weight( "score", 1.0 );

	evaluation::MetaPoseEvaluator cmdline_evals;
	evaluation::read_common_evaluator_options( cmdline_evals );
	for ( evaluation::MetaPoseEvaluator::EvaluatorList::const_iterator it = cmdline_evals.evaluators().begin();
				it != cmdline_evals.evaluators().end(); ++it ) {
 		add_evaluation( *it );
	}
}

}//archive
}//jd2
}//protocols
