// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is part of the Rosetta software suite and is made available under license.
// The Rosetta software is developed by the contributing members of the Rosetta Commons consortium.
// (C) 199x-2009 Rosetta Commons participating institutions and developers.
// For more information, see http://www.rosettacommons.org/.

/// @file   protocols/jd2/MPIWorkPoolJobDistributor.hh
/// @brief  header for MPIWorkPoolJobDistributor - intended for continuous resamplig jobs  that spawn new jobs based on a pool/archive of
///         structures
/// @author Oliver Lange olange@u.washington.edu

#ifndef INCLUDED_protocols_jd2_archive_EvaluatedArchive_HH
#define INCLUDED_protocols_jd2_archive_EvaluatedArchive_HH

// Unit headers
#include <protocols/jd2/archive/ArchiveManager.fwd.hh>
#include <protocols/jd2/archive/ArchiveBase.hh>
//#include <protocols/jd2/archive/EvaluatedArchive.fwd.hh>


// Package headers
#include <core/io/silent/silent.fwd.hh>
#include <core/pose/Pose.fwd.hh>
#include <core/scoring/ScoreFunction.hh>

#include <protocols/evaluation/PoseEvaluator.hh>

// Utility headers
#include <core/types.hh>
#include <utility/pointer/ReferenceCount.hh>
#include <utility/exit.hh>

// C++ headers
#include <string>
#include <list>

namespace protocols {
namespace jd2 {
namespace archive {
//class ArchiveManager;



///@brief Tags used to tag messeges sent by MPI functions used to decide whether a slave is requesting a new job id or
///flagging as job as being a bad input

///@details This job distributor is meant for running jobs where the machine you are using has a large number of
///processors, the number of jobs is much greater than the number of processors, or the runtimes of the individual jobs
///could vary greatly. It dedicates the head node (whichever processor gets processor rank #0) to handling job requests
///from the slave nodes (all nonzero ranks). Unlike the MPIWorkPartitionJobDistributor, this JD will not work at all
///without MPI and the implementations of all but the interface functions have been put inside of ifdef directives.
///Generally each function has a master and slave version, and the interface functions call one or the other depending
///on processor rank.

class EvaluatedArchive : public ArchiveBase {
	typedef ArchiveBase Parent;
public:
	EvaluatedArchive( ArchiveManagerAP ptr ) : ArchiveBase( ptr ), scorefxn_( NULL ) { setup_default_evaluators(); };

	virtual bool add_structure( core::io::silent::SilentStructOP, core::io::silent::SilentStructOP );
	virtual core::io::silent::SilentStructOP evaluate( core::io::silent::SilentStructOP, core::pose::Pose& pose ) const;
	virtual void score( core::pose::Pose& pose ) const; //apply appropriate score function to pose

	// how many structures should be in archive .. varies from decoys().size() in startup phase.
	core::Size nstruct() const { return nstruct_; };

	// how many structures should be in archive .. varies from decoys().size() in startup phase.
	void set_nstruct( core::Size set ) { nstruct_ = set; };

	// compute score according to select_weights --- this can contain any evaluator columns
	// assumes that all energy entries are up-to-date
	core::Real select_score( core::io::silent::SilentStructOP );

	//save Evaluator state ?
	//	virtual void save_to_file( std::string const& dirname );
	// virtual void restore_from_file( std::string const& dirname );

	///@brief set common evaluators: eg. ConstraintEvaluator if -cst_file is present
	void setup_default_evaluators();


	///@brief only overloaded this to add some verbosity each time we read structures
	virtual void read_structures( core::io::silent::SilentFileData&, Batch const& batch );


protected:

	void add_evaluation( evaluation::PoseEvaluatorOP, core::Real weight = 0.0 );
	bool has_evaluator( std::string const& column );
	void set_weight( std::string const& column, core::Real weight );
	void set_scorefxn( core::scoring::ScoreFunctionOP scorefxn_ );
	// void set_evaluators( evaluation::PoseEvaluators const& );
	//	void remove_evaluation( );
	//  void clear_evaluators();
private:

	void rescore();
	//	void sort();

	//a map from a column name to a weight
	typedef std::map< std::string, core::Real > WeightMap;
	typedef std::map< std::string, evaluation::PoseEvaluatorOP > EvaluatorMap;

	core::scoring::ScoreFunctionOP scorefxn_;
	WeightMap select_weights_;
	EvaluatorMap evaluators_;
	bool scores_are_clean_; //false after add_evaluation or change of scorefxn_
	core::Size nstruct_; //how many structures maintained in archive
};


}//archive
}//jd2
}//protocols


#endif //INCLUDED_protocols_jd2_Archive_HH
