// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   protocols/ligand_docking/MultiResidueLigandDock.cc
///
/// @brief
/// @Gordon Lemmon

#include <protocols/ligand_docking/MultiResidueLigandDock.hh>


#include <protocols/moves/Mover.hh>
#include <protocols/moves/MinMover.hh>
#include <protocols/moves/RigidBodyMover.hh>
#include <protocols/geometry/RB_geometry.hh>
#include <protocols/jd2/Job.hh>

#include <core/kinematics/FoldTree.hh>
#include <core/conformation/Residue.hh>

#include <core/optimization/MinimizerOptions.hh>
#include <core/options/option.hh>

#include <core/scoring/Energies.hh>
#include <core/scoring/rms_util.hh>
#include <core/scoring/ScoreFunction.hh>
#include <core/scoring/constraints/RotamerConstraint.hh>

#include <core/types.hh>
#include <core/util/prof.hh>
#include <core/scoring/ScoreType.hh>

#include <core/util/Tracer.hh>

#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray1D.io.hh>
#include <ObjexxFCL/string.functions.hh>

#include <ctime>
#include <fstream>
#include <sstream>



// option key includes

#include <core/options/keys/docking.OptionKeys.gen.hh>
#include <core/options/keys/in.OptionKeys.gen.hh>
#include <core/io/pdb/pose_io.hh>
#include <protocols/jd2/JobDistributor.hh>

namespace protocols {
namespace ligand_docking {


MultiResidueLigandDock::MultiResidueLigandDock(): Mover("MultiResidueLigandDock"), command_map_(){
	Mover::type( "MultiResidueLigandDock" );
	if( core::options::option[ core::options::OptionKeys::in::file::native].user()){ /// TODO shouldn't this be handled by the base class?
		std::string pdb_name(core::options::option[core::options::OptionKeys::in::file::native]().name());
		core::pose::PoseOP input_pose= new core::pose::Pose();
		core::io::pdb::pose_from_pdb(*input_pose, pdb_name);
		set_input_pose(input_pose);
	}
}


MultiResidueLigandDock::MultiResidueLigandDock(MultiResidueLigandDock const & /*that*/): Mover()
{
	utility_exit_with_message("copy c-tor not allowed!");
}


MultiResidueLigandDock::~MultiResidueLigandDock() {}


void
MultiResidueLigandDock::apply( core::pose::Pose & pose )
{

	core::pose::Pose const  copy = (get_native_pose()==0) ? pose : *(get_native_pose());

	core::util::prof_reset();

	core::scoring::constraints::load_unboundrot(pose); // adds scoring bonuses for the "unbound" rotamers, if any

	if( core::options::option[ core::options::OptionKeys::docking::ligand::rescore])return;

	// If we change the fold tree during the course of this run,
	// we should restore it afterwards for scoring and output!
	core::kinematics::FoldTree fold_tree_copy = pose.fold_tree();

	command_map_.parse_default_ligand_option_file(pose);
	command_map_[ligand_options::random_conformer]->apply(); // now only "pose" parameter is needed, actually
	command_map_[ligand_options::start_from]->apply();
	command_map_[ligand_options::translate]->apply();
	multi_residue_ligand_dock_tracer<< "finished translate, time to rotate ligand"<< std::endl;
	command_map_[ligand_options::rotate]->apply();
	multi_residue_ligand_dock_tracer<< "finished rotate, time to minimize ligand"<< std::endl;
	command_map_[ligand_options::minimize_ligand]->apply();
	multi_residue_ligand_dock_tracer<< "finished minimize ligand, time to slide together"<< std::endl;
	command_map_[ligand_options::slide_together]->apply();
	multi_residue_ligand_dock_tracer<< "finished slide together, time to minimize backbone"<< std::endl;
	command_map_[ligand_options::minimize_backbone]->apply();
	multi_residue_ligand_dock_tracer<< "finished minimize backbone, time to tether_ligand"<< std::endl;
	command_map_[ligand_options::tether_ligand]->apply();
	multi_residue_ligand_dock_tracer<< "finished tether_ligand, time to soft_rep"<< std::endl;
	command_map_[ligand_options::soft_rep]->apply();// setup the score_fxn for use with protocol
	multi_residue_ligand_dock_tracer<< "finished soft_rep, time to apply protocol"<< std::endl;
	command_map_[ligand_options::protocol]->apply();

	// Remove the ligand tether.
	command_map_.release_constraints();
	// Do the final, "tight" minimization of all DOFs
	// Set up move map for minimizing.
	// Have to do this after initial perturb so we get the "right" interface defn.
	protocols::moves::MinMoverOP const dfpMinTightTol = command_map_.get_final_min_mover();
	dfpMinTightTol->min_options()->nblist_auto_update(true);
	dfpMinTightTol->apply(pose);

	// If we change the fold tree during the course of this run, we should restore it.
	pose.fold_tree( fold_tree_copy );
	core::util::prof_show(); ///TODO delete this
	add_scores_to_job(pose);
	append_ligand_docking_scores(copy, pose);
}

void MultiResidueLigandDock::add_scores_to_job(
	core::pose::Pose & pose
)
{
	using namespace core::scoring;
	ScoreFunctionCOP sfxn = command_map_.get_score_function();

	core::Real const tot_score = sfxn->score( pose );

	// Which score terms to use
	typedef utility::vector1<ScoreType> ScoreTypeVec;
	ScoreTypeVec score_types;
	for(int i = 1; i <= n_score_types; ++i) {
		ScoreType ii = ScoreType(i);
		if ( sfxn->has_nonzero_weight(ii) ) score_types.push_back(ii);
	}

	protocols::jd2::JobOP job( protocols::jd2::JobDistributor::get_instance()->current_job() );
	for(ScoreTypeVec::iterator ii = score_types.begin(), end_ii = score_types.end(); ii != end_ii; ++ii) {
		job->add_string_real_pair(name_from_score_type(*ii), sfxn->get_weight(*ii) * pose.energies().total_energies()[ *ii ]);
	}
	job->add_string_real_pair(name_from_score_type(core::scoring::total_score), tot_score);
}

/// @brief For multiple ligands, append ligand docking scores for each ligand
void
MultiResidueLigandDock::append_ligand_docking_scores(core::pose::Pose const & before, core::pose::Pose const & after) const
{
	protocols::jd2::JobOP job( protocols::jd2::JobDistributor::get_instance()->current_job() );
	std::set< core::Size > const ligand_jump_ids = command_map_.get_docking_ligands(after);
	for(
			std::set<core::Size>::const_iterator index(ligand_jump_ids.begin()),
			end(ligand_jump_ids.end());
			index != end;
			++index
	){
		multi_residue_ligand_dock_tracer.Debug << "appending ligand: "<< *index << std::endl;
		append_ligand_docking_scores(*index, job, before, after);
	}
}

/// @brief Scores to be output that aren't normal scorefunction terms.
void
MultiResidueLigandDock::append_ligand_docking_scores(
	int const jump_id,
	protocols::jd2::JobOP job,
	core::pose::Pose const & before,
	core::pose::Pose const & after
) const {
	append_interface_deltas(jump_id, job, after, command_map_.get_score_function());
	append_ligand_travel(jump_id, job, before, after);
	append_radius_of_gyration(jump_id, job, before);
	append_ligand_RMSD(jump_id, job, before, after);
}

} // namespace ligand_docking
} // namespace protocols
