// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   protocols/ligand_docking/ligand_functions.cc
///
/// @brief
/// @author Ian W. Davis


#include <protocols/ligand_docking/ligand_functions.hh>

#include <core/conformation/Residue.hh>
#include <core/id/AtomID.hh>
#include <core/options/option.hh>
#include <core/options/keys/packing.OptionKeys.gen.hh>
#include <core/scoring/constraints/AmbiguousConstraint.hh>
#include <core/scoring/constraints/Constraint.hh>
#include <core/scoring/constraints/ConstraintSet.hh>
#include <core/scoring/constraints/DihedralConstraint.hh>
#include <core/scoring/constraints/HarmonicFunc.hh>
#include <core/util/Tracer.hh>

#include <protocols/ligand_docking/grid_functions.hh>

#include <numeric/angle.functions.hh>
#include <numeric/conversions.hh>

namespace protocols {
namespace ligand_docking {

static core::util::Tracer TR("protocols.ligand_docking.ligand_functions");

/// @brief Helper function.
core::scoring::constraints::ConstraintOP
torsion_constraints_from_mean_sd(
	core::Size rsd_no,
	core::Size chino,
	core::chemical::ResidueType const & rsd_type,
	utility::vector1< std::pair< core::Real, core::Real > > const & mean_sd_degrees
)
{
	using namespace core;
	using namespace core::scoring::constraints;
	using core::chemical::AtomIndices;
	using core::id::AtomID;

	TR << "Rsd " << rsd_no << " chi " << chino << " =";
	ConstraintCOPs csts;
	for(Size j = 1; j <= mean_sd_degrees.size(); ++j) {
		TR << " " << mean_sd_degrees[j].first;
		// input is in degrees, but dihedral constraints deal in radians
		Real const chi_radians = numeric::conversions::radians( mean_sd_degrees[j].first );
		Real const stddev_radians = numeric::conversions::radians( mean_sd_degrees[j].second );
		FuncOP restr_func = new CircularHarmonicFunc( chi_radians, stddev_radians );
		AtomIndices chi_idx = rsd_type.chi_atoms(chino);
		ConstraintCOP constraint = new DihedralConstraint(
			AtomID(chi_idx[1], rsd_no),
			AtomID(chi_idx[2], rsd_no),
			AtomID(chi_idx[3], rsd_no),
			AtomID(chi_idx[4], rsd_no),
			restr_func
		);
		csts.push_back( constraint );
	}
	TR << std::endl;
	ConstraintOP cst = new AmbiguousConstraint( csts );
	return cst;
}


core::scoring::constraints::ConstraintOP
torsion_constraints_from_rotamers(
	core::Size rsd_no,
	core::Size chino,
	utility::vector1< core::conformation::ResidueCOP > const & rsds,
	core::Real stddev_degrees
)
{
	using namespace core;
	chemical::ResidueType const & rsd_type = rsds[1]->type();

	Real const tol_d = stddev_degrees / 10.0; // within this range, considered to be the same minimum
	utility::vector1< std::pair< Real, Real> > minima_d;
	for(Size i = 1; i <= rsds.size(); ++i) {
		//runtime_assert( rsds[i].type().name() == rsdtype.name() );
		Real const chi_d = rsds[i]->chi(chino);
		bool found = false;
		for(Size j = 1; j <= minima_d.size(); ++j) {
			Real const min = minima_d[j].first;
			if(std::abs(min - numeric::nearest_angle_degrees(chi_d, min)) < tol_d) {
				found = true;
				break;
			}
		}
		if( !found ) {
			minima_d.push_back( std::make_pair(chi_d, stddev_degrees) );
		}
	}

	return torsion_constraints_from_mean_sd(rsd_no, chino, rsd_type, minima_d);
}


core::scoring::constraints::ConstraintOP
torsion_constraints_from_chi_rotamers(
	core::Size rsd_no,
	core::Size chino,
	core::chemical::ResidueType const & rsdtype
)
{
	return torsion_constraints_from_mean_sd( rsd_no, chino, rsdtype, rsdtype.chi_rotamers(chino) );
}


void
get_ligand_torsion_constraints(
	core::pose::Pose & pose,
	core::Size rsd_no,
	core::Real stddev_degrees,
	utility::vector1< core::scoring::constraints::ConstraintOP > & csts_out
)
{
	using namespace core;
	using namespace core::options;
	using core::chemical::ResidueType;

	ResidueType const & rsdtype = pose.residue_type(rsd_no);
	for(Size i = 1; i <= rsdtype.nchi(); ++i) {
		if( rsdtype.chi_rotamers(i).size() == 0 ) {
			utility::vector1< core::conformation::ResidueOP > rotamers;
			rotamers_for_trials(pose, rsd_no, rotamers);
			if( rotamers.empty() || option[ OptionKeys::packing::use_input_sc ]() ) rotamers.push_back( pose.residue(rsd_no).clone() );
			csts_out.push_back( torsion_constraints_from_rotamers(rsd_no, i, rotamers, stddev_degrees) );
		} else {
			csts_out.push_back( torsion_constraints_from_chi_rotamers(rsd_no, i, rsdtype) );
		}
	}
}


void
constrain_ligand_torsions(
	core::pose::Pose & pose,
	core::Real stddev_degrees
)
{
	using namespace core;
	using namespace core::scoring::constraints;
	ConstraintSetOP new_constraint_set = pose.constraint_set()->clone();
	for(Size rsdno = 1; rsdno <= pose.total_residue(); ++rsdno) {
		if( pose.residue_type(rsdno).is_polymer() ) continue;
		utility::vector1< ConstraintOP > csts;
		get_ligand_torsion_constraints(pose, rsdno, stddev_degrees, csts);
		for(Size cstno = 1; cstno <= csts.size(); ++cstno) {
			//csts[cstno]->show(TR);
			new_constraint_set->add_constraint( csts[cstno] );
		}
	}
	pose.constraint_set( new_constraint_set );
}


} // namespace ligand_docking
} // namespace protocols
