// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   core/pack/task/ResfileReader.cc
/// @brief  implementation of resfile reader and its command classes
/// @author Gordon Lemmon (glemmon@gmail.com)
///
// Unit Headers
#include <protocols/ligand_docking/ligand_options/InterfaceBuilder.hh>
#include <protocols/ligand_docking/ligand_options/interface_distance_functions.hh>
#include <core/util/Tracer.hh>
#include <core/conformation/Conformation.hh>
#include <core/types.hh>
#include <protocols/ligand_docking/ligand_options/chain_functions.hh>

#include <utility/assert.hh>

namespace protocols {
namespace ligand_docking {
namespace ligand_options {

InterfaceBuilder::InterfaceBuilder(
		core::pose::Pose const & pose,
		std::set<core::Size> const & ligand_chains,
		core::Real cutoff,
		bool add_nbr_radius,
		bool all_atom_mode,
		core::Size extension_window
):
	pose_(pose),
	ligand_chains_(ligand_chains),
	cutoff_(cutoff),
	add_nbr_radius_(add_nbr_radius),
	all_atom_mode_(all_atom_mode),
	extension_window_(extension_window)
{}

InterfaceBuilder::InterfaceBuilder(
		core::pose::Pose const & pose,
		std::map<core::Size, core::Real> const & ligand_chains,
		core::Real cutoff,
		bool add_nbr_radius,
		bool all_atom_mode,
		core::Size extension_window
):
	pose_(pose),
	cutoff_(cutoff),
	add_nbr_radius_(add_nbr_radius),
	all_atom_mode_(all_atom_mode),
	extension_window_(extension_window)
{
	std::map<core::Size, core::Real>::const_iterator ligand_chain= ligand_chains.begin();
	for(; ligand_chain != ligand_chains.end(); ++ligand_chain){
		core::Size chain_id= ligand_chain->first;
		ligand_chains_.insert(chain_id);
	}
}

Interface InterfaceBuilder::build() const{
	Interface interface( pose_.total_residue(), InterfaceInfo() ); // init all positions to false
	find_interface_residues(interface);
	if (extension_window_ > 0) enforce_minimum_length(interface);
	interface_builder_tracer.Debug<< "built interface: "<< interface<< std::endl;
	return interface;
}

void InterfaceBuilder::find_interface_residues(Interface & interface) const{

	find_ligand_residues(interface);
	// set up protein residues to minimize based on distance to ligand residues we are minimizing
	utility::vector1<core::Size> ligand_residues= interface.get_interface_residues();
	find_protein_residues(interface, ligand_residues);
}

void InterfaceBuilder::find_ligand_residues(
		Interface & interface
)const{
	std::set<core::Size>::const_iterator ligand_chain= ligand_chains_.begin();
	for(; ligand_chain != ligand_chains_.end(); ++ligand_chain){
		core::Size residue_id= pose_.conformation().chain_begin(*ligand_chain);
		core::Size end= pose_.conformation().chain_end(*ligand_chain);
		for(; residue_id <= end; ++residue_id){
			interface[residue_id].type=  InterfaceInfo::is_interface;
			interface[residue_id].chain= *ligand_chain;
		}
	}
	interface_builder_tracer.Debug << "ligands found: " << interface << std::endl;
}

// First call find_ligand_residues
void InterfaceBuilder::find_protein_residues(
		Interface & interface,
		utility::vector1<core::Size> const ligand_residues
)const{
	for(core::Size i = 1, i_end = pose_.total_residue(); i <= i_end; ++i) {
		if(pose_.residue(i).is_protein() ){
			utility::vector1<core::Size>::const_iterator ligand_residue_iterator= ligand_residues.begin();
			for(; ligand_residue_iterator != ligand_residues.end(); ++ligand_residue_iterator){
				set_interface_residue(interface, i, *ligand_residue_iterator);
			}
		}
	}
}

void InterfaceBuilder::set_interface_residue(
		Interface & interface,
		core::Size const potential_interface_residue_id,
		core::Size const ligand_interface_residue_id
)const{
	core::conformation::Residue const & potential_interface_residue = pose_.residue(potential_interface_residue_id);
	core::conformation::Residue const & ligand_residue = pose_.residue(ligand_interface_residue_id);

	if( is_interface_residue(potential_interface_residue, ligand_residue)){
		core::Size const ligand_chain= interface[ligand_interface_residue_id].chain;
		interface[potential_interface_residue_id].type= InterfaceInfo::is_interface;
		interface[potential_interface_residue_id].chain= ligand_chain;
	}
}

bool InterfaceBuilder::is_interface_residue(
		core::conformation::Residue const & potential_interface_residue,
		core::conformation::Residue const & ligand_interface_residue
)const {
	core::Size const potential_interface_neighbor_atom_id= potential_interface_residue.nbr_atom();
	core::Vector const potential_interface_vector= potential_interface_residue.xyz(potential_interface_neighbor_atom_id);

	double cutoff = add_nbr_radius_ ?
			potential_interface_residue.nbr_radius() + cutoff_ :
			cutoff_;

	if (all_atom_mode_){
		return check_all_ligand_atoms(ligand_interface_residue, potential_interface_vector, cutoff);
	}
	else{
		return check_neighbor_ligand_atom(ligand_interface_residue, potential_interface_vector, cutoff);
	}
}

void InterfaceBuilder::enforce_minimum_length(Interface & interface) const{
	for(core::Size chain_id=1; chain_id <= pose_.conformation().num_chains(); ++chain_id){
		core::Size start= pose_.conformation().chain_begin(chain_id);
		core::Size end= pose_.conformation().chain_end(chain_id);
		if(pose_.residue(start).is_polymer()){ // because ligand residues don't know about terminus
			interface.enforce_minimum_length(start, end, extension_window_);
		}
	}
}

} //namespace ligand_options
} //namespace ligand_docking
} //namespace protocols
