// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   protocols/ligand_docking/ligand_options/LigandOptionMap.cc
/// @brief  implementation of resfile reader and its command classes
/// @author Gordon Lemmon (glemmon@gmail.com), adapted from the ResfileReader code
/// by Steven Lewis (smlewi@unc.edu) and Andrew Leaver-Fay

// Unit Headers
#include <protocols/ligand_docking/ligand_options/LigandOptionMap.hh>
#include <protocols/ligand_docking/ligand_options/Mutate_same_name3.hh>
#include <protocols/ligand_docking/ligand_options/ProtocolOption.hh>
#include <protocols/ligand_docking/ligand_options/Translate.hh>
#include <protocols/ligand_docking/ligand_options/Rotate.hh>
#include <protocols/ligand_docking/ligand_options/Minimize_backbone.hh>
#include <protocols/ligand_docking/ligand_options/Minimize_ligand.hh>
#include <protocols/ligand_docking/ligand_options/Tether_ligand.hh>
#include <protocols/ligand_docking/ligand_options/Start_from.hh>
#include <protocols/ligand_docking/ligand_options/Random_conformer.hh>
#include <protocols/ligand_docking/ligand_options/Soft_rep.hh>
#include <protocols/ligand_docking/ligand_options/Slide_together.hh>
#include <protocols/ligand_docking/ligand_options/chain_functions.hh>
#include <protocols/geometry/RB_geometry.hh>
#include <protocols/moves/RigidBodyMover.hh>
#include <protocols/moves/MinMover.hh>
#include <protocols/ligand_docking/RandomConformerMover.hh>
#include <protocols/ligand_docking/UnconstrainedTorsionsMover.hh>

// Package Headers
#include <core/pose/Pose.hh>
#include<core/pose/PDBInfo.hh>

// Project Headers
#include <core/chemical/ResidueType.hh>
#include <core/options/option.hh>

// Utility Headers
#include <utility/io/izstream.hh>
#include <utility/exit.hh>
#include <utility/assert.hh> //ASSERT_ONLY makes release build happy
#include <utility/string_util.hh>
#include <core/util/Tracer.hh>

using core::util::T;
using core::util::Error;
using core::util::Warning;

//STL headers
#include <string>
//#include <iostream> //need this for debugging
#include <fstream>
#include <sstream> //automatic checking string to int conversion
#include <cctype> //for split_lines to handle '\t' tab characters
#include <stdlib.h>
#include <iostream>
#include <set>


// option key includes

#include <core/options/keys/docking.OptionKeys.gen.hh>


namespace protocols {
namespace ligand_docking {

namespace ligand_options {

LigandOptionMap::LigandOptionMap(){
	populate_option_map();
}

LigandOptionMap::LigandOptionMap(
		core::pose::Pose & pose,
		std::string const & options_file
){
	populate_option_map();
	parse_ligand_option_file(pose, options_file);
}

LigandOptionMap::LigandOptionMap(
		core::pose::Pose & pose
){
	parse_default_ligand_option_file(pose);
}

void
LigandOptionMap::populate_option_map(){
	option_map_["protocol"]=protocol;
	option_map_["soft_rep"]=soft_rep;
	option_map_["random_conformer"]=random_conformer;
	option_map_["translate"]=translate;
	option_map_["rotate"]=rotate;
	option_map_["mutate_same_name3"]=mutate_same_name3;
	option_map_["minimize_ligand"]=minimize_ligand;
	option_map_["minimize_backbone"]=minimize_backbone;
	option_map_["tether_ligand"]=tether_ligand;
	option_map_["start_from"]=start_from;
	option_map_["slide_together"]=slide_together;
}

///@details this creates a map linking the parsed strings from the LigandOptionsFile
///to the command objects.  NEW COMMANDS MUST BE ADDED HERE, HARD CODED
///note that it uses the command object name() method, not hard coded strings
///(of course, the name() method has hard coded strings...)
void
LigandOptionMap::add_commands(core::pose::Pose & pose){
	// These ones stand alone
	(*this)[ rotate] = new Rotate(pose);
	(*this)[ translate] = new Translate(pose);
	(*this)[ start_from] = new Start_from(pose);
	(*this)[ slide_together] = new Slide_together(pose);
	(*this)[ tether_ligand] = new Tether_ligand(pose);

	Minimize_backbone* mb= new Minimize_backbone(pose);
	(*this)[ minimize_backbone] = mb;

	// These 3 are needed by others
	Minimize_ligand* ml= new Minimize_ligand(pose);
	(*this)[ minimize_ligand] = ml;

	Mutate_same_name3* msn = new Mutate_same_name3(pose);
	(*this)[ mutate_same_name3] = msn;

	Soft_rep* sr= new Soft_rep(pose);
	(*this)[ soft_rep] = sr;

	(*this)[ random_conformer] = new Random_conformer(pose, ml->get_const_ligand_torsion_restraints_reference());

	(*this)[protocol]= new Protocol(
			pose,
			sr->get_score_fxn(),
			ml->get_const_ligands_reference(),
			ml->get_const_ligand_torsion_restraints_reference(),
			mb->get_const_ligands_reference(),
			msn->get_const_ligands_to_mutate_reference()
	);
}

void LigandOptionMap::release_constraints(){
	LigandOptionCommand* base_type= (*this)[tether_ligand]();
	Tether_ligand *t= dynamic_cast<Tether_ligand*>(base_type);
	option_map_tracer.Debug<< "now releasing_constraints" << std::endl;
	t->release();
}

///@brief utility function for ligandOptionFile reader (checks for a leading # signaling a comment)
void
LigandOptionMap::remove_comments( utility::vector1< std::string > & tokens)
{
	utility::vector1<std::string>::iterator index;
	for( index= tokens.begin(); index != tokens.end(); ++index){
		if((*index)[0]=='#'){
			tokens.erase(index, tokens.end());
			return;
		}
	}
}

///@details LigandOption parser applies a LigandOptionFile filename to a pose
///each line of the ligand option file is broken into whitespace-delimited tokens
///whenever it reads a comment token, it ignores the rest of the line
///commands read before a "LIGAND" token are stored for application
///later as defaults
void
LigandOptionMap::parse_default_ligand_option_file( core::pose::Pose & pose)
{
	std::string const optionsFile= core::options::option[ core::options::OptionKeys::docking::ligand::option_file].value();
	parse_ligand_option_file(pose, optionsFile);
}

void
LigandOptionMap::parse_ligand_option_file( core::pose::Pose & pose, std::string const filename)
{
	std::string optionsFile;
	utility::io::izstream file( filename );
	if (!file) {
		Error() << "File:" << filename << " not found!\n";
		utility_exit_with_message( "Cannot open file " + filename );
	} else {
		//T() << "read file: " << filename << "\n";
	}
	utility::slurp( file, optionsFile );
	parse_options_file_string( pose, optionsFile );

}

void
LigandOptionMap::parse_options_file_string( core::pose::Pose & pose, std::string const & options_file_string )
{
	std::istringstream ligand_collector(options_file_string);
	add_commands(pose);
	core::Size current_ligand=0;

	while(ligand_collector){
		utility::vector1< std::string > tokens( tokenize_line( ligand_collector ) );
		// ignore blank lines
		remove_comments(tokens);
		if ( tokens.size()==0 ) continue;
		std::string command_token = tokens[1];
		if ( command_token != "LIGAND") continue;
		current_ligand=get_chain_id_from_chain(tokens[2], pose);
		option_map_tracer.Debug<< "inserting ligand "<<tokens[2] << std::endl;
		ligands_to_dock_.insert(current_ligand);
	}
	current_ligand=0;

	core::Size lineno = 0;
	std::istringstream options_file(options_file_string);
	while ( options_file ) {
		utility::vector1< std::string > tokens( tokenize_line( options_file ) );
		++lineno;

		// ignore blank lines
		remove_comments(tokens);
		if ( tokens.size()==0 ) continue;
		std::string command_token = tokens[1];

		if(command_token=="DEFAULT"){
			current_ligand=0;
			option_map_tracer.Debug<< "now parsing default options"<< std::endl;
		}
		else if(command_token=="LIGAND"){
			if ( tokens.size()!=2 ){
				utility_exit_with_message("LIGAND must be followed by chain_id");
			};
			current_ligand=get_chain_id_from_chain(tokens[2], pose);
			option_map_tracer.Debug<< "now parsing chain "<< current_ligand <<" options"<< std::endl;
		}
		else{
			///TODO have option_map_ inherit from std::map and overwrite find to quit.
			std::map<std::string, option>::const_iterator const option_pair= option_map_.find(command_token);
			if( option_pair == option_map_.end()){
				Error() << "LigandOptionFile ERROR: line: " << lineno << " option not found: " << command_token << std::endl;
				utility_exit();
			}
			///TODO overwrite std::map find in the LigandOptionMap to throw an exception or quit.
			std::map<option, LigandOptionCommandOP>::const_iterator const command_pair= this->find(option_pair->second);
			if(command_pair == this->end()){
				Error() << "LigandOptionFile ERROR: line: " << lineno << " command not found: " << option_pair->second << std::endl;
				utility_exit();
			}
			LigandOptionCommandOP command = command_pair->second;
			if ( !command ) {
				Error() << "LigandOptionFile ERROR: line: " << lineno << " command not found: " << command_token << std::endl;
				utility_exit();
			}
			option_map_tracer.Debug << command_token << ' '<< current_ligand << std::endl;
			current_ligand==0 ?
				command->option( tokens, ligands_to_dock_):
				command->option( tokens, current_ligand);
		}
	}
}

const protocols::moves::MinMoverOP
LigandOptionMap::get_final_min_mover() {
	std::string min_type= "dfpmin_armijo_nonmonotone_atol";
	core::Real tolerance= 0.02;
	bool use_nb_list= true;

	core::scoring::ScoreFunctionOP const & score_function= get_final_score_fxn();// change score fxn and retrieve
	option_map_tracer.Debug<< "final sc_fxn: "<< *score_function<< std::endl;
	core::kinematics::MoveMapOP const movemap= get_final_move_map();
	return new protocols::moves::MinMover( movemap, score_function, min_type, tolerance, use_nb_list);
}

const core::scoring::ScoreFunctionOP &
LigandOptionMap::get_final_score_fxn(){
	LigandOptionCommand* base_option= (*this)[soft_rep]();
	Soft_rep *s= dynamic_cast<Soft_rep*>(base_option);
	s->restore_to_hard_rep();
	return s->get_score_fxn();
}

const core::scoring::ScoreFunctionOP & ///TODO make this COP
LigandOptionMap::get_score_function()const{
	// can't use [] because does not return const
	const_iterator iter= this->find(soft_rep);
	LigandOptionCommandOP const base_option= iter->second;
	Soft_rep *s= dynamic_cast<Soft_rep*>(base_option());//get raw pointer
	return s->get_score_fxn();
}

const std::set<core::Size> LigandOptionMap::get_docking_ligands(const core::pose::Pose & pose) const{

	return get_jump_ids_from_chain_ids(ligands_to_dock_, pose);
}

const core::kinematics::MoveMapOP
LigandOptionMap::get_final_move_map() const{
	// can't use [] because does not return const
	const_iterator iter= this->find(protocol);
	LigandOptionCommandOP const base_option= iter->second;
	Protocol *p= dynamic_cast<Protocol*>(base_option());
	option_map_tracer.Debug<< "final_movemap:"<< std::endl;
	return p->make_movemap(true /* allow backbone to minimize according to user preference*/);
}

///@detail given an input stream, return a vector of space delimited words
utility::vector1< std::string >
LigandOptionMap::tokenize_line( std::istream & inputstream )
{

	utility::vector1< std::string > tokens;
	std::string input_line;
	std::getline( inputstream, input_line );

	unsigned int llength = input_line.size();
	unsigned int processing = 0;
	unsigned int token_start_char = 0;

	while ( processing < llength ) {
		if ( std::isspace( input_line[ processing ] ) ) { // == ' ') {
			if ( !( std::isspace(input_line[ token_start_char ] ) ) ) { // != ' ' ) {
				std::string token = input_line.substr( token_start_char, processing - token_start_char);
				tokens.push_back( token );
			}
			++processing;
			token_start_char = processing;
		} else {
			++processing;
		}
	}
	if ( processing != token_start_char ) { // last token on the line
		std::string token = input_line.substr( token_start_char, processing - token_start_char + 1 );
		tokens.push_back( token );
	}

	return tokens;
}

} //namespace ligand_options
} //namespace ligand_docking
} //namespace protocols
