// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   core/pack/task/ResfileReader.cc
/// @brief  implementation of resfile reader and its command classes
/// @author Gordon Lemmon (glemmon@gmail.com), adapted from the ResfileReader code
/// by Steven Lewis (smlewi@unc.edu) and Andrew Leaver-Fay

// Unit Headers
#include <protocols/ligand_docking/ligand_options/Minimize_backbone.hh>
#include <protocols/ligand_docking/ligand_options/InterfaceBuilder.hh>
#include <protocols/ligand_docking/ligand_options/Interface.hh>

#include <core/scoring/constraints/CoordinateConstraint.hh>
#include <core/scoring/constraints/HarmonicFunc.hh>

#include <core/chemical/util.hh>

// Utility Headers
#include <numeric/random/random.hh>
#include <utility/io/izstream.hh>
#include <utility/exit.hh>
#include <utility/assert.hh> //ASSERT_ONLY makes release build happy
#include <utility/string_util.hh>
#include <core/util/Tracer.hh>
#include <core/types.hh>

using core::util::T;
using core::util::Error;
using core::util::Warning;

//STL headers
#include <algorithm> // for std::min

namespace protocols {
namespace ligand_docking {
namespace ligand_options {

Minimize_backbone::Minimize_backbone(
		core::pose::Pose & pose
) :	LigandOptionCommand(pose), LigandSpecificCommand(pose), DefaultCommand(pose){}

void Minimize_backbone::option(
		utility::vector1< std::string> const & tokens,
		core::Size const & ligand
) {
	if (tokens.size() != 2) {
		utility_exit_with_message("'minimize_ligand' takes one argument (Real)");
	}
	core::Real stdev_angstroms = strtod(tokens[2].c_str(), NULL);

	minimize_backbone_tracer << "adding ligand: "<< ligand << " with "<< stdev_angstroms << " stdev_angstroms"<< std::endl;
	chains_[ligand] = stdev_angstroms;
}

void Minimize_backbone::option(
		utility::vector1< std::string> const & tokens,
		std::set<core::Size> const & ligands_to_dock
) {
	if (tokens.size() != 2) {
		utility_exit_with_message("'minimize_ligand' takes one argument (Real)");
	}
	core::Real stdev_angstroms = strtod(tokens[2].c_str(), NULL);

	std::set<core::Size>::const_iterator i = ligands_to_dock.begin();
	for (; i != ligands_to_dock.end(); i++) {
		chains_[*i] = stdev_angstroms;
	}
}

void Minimize_backbone::apply() {
	if(chains_.empty()) return;
	setup_bbmin_foldtree();
}

void Minimize_backbone::setup_bbmin_foldtree() {
	assert(pose_.fold_tree().check_edges_for_atom_info());
	InterfaceBuilder interface_builder(pose_, chains_, 7.0, false, true, 3);
	Interface interface= interface_builder.build();
	restrict_to_protein_residues(interface);
	minimize_backbone_tracer.Debug << "interface for fold_tree: "<< interface << std::endl;
	assert(pose_.fold_tree().check_edges_for_atom_info());
	reorder_foldtree_around_mobile_regions(interface);
	restrain_protein_Calphas(interface);
	assert(pose_.fold_tree().check_edges_for_atom_info());
}

void Minimize_backbone::reorder_foldtree_around_mobile_regions(
		Interface const & interface
) {
	assert(pose_.fold_tree().check_edges_for_atom_info());
	core::kinematics::FoldTree f = pose_.fold_tree(); // a copy
	assert(f.check_edges_for_atom_info());
	minimize_backbone_tracer.Debug << "Initial foldtree " << f << std::endl;
	core::kinematics::FoldTreeOP better_ligand_jumps = create_fold_tree_with_ligand_jumps_from_attach_pts(f, interface);
	assert(better_ligand_jumps->check_edges_for_atom_info());
	minimize_backbone_tracer.Debug << "foldtree with ligand jumps from attach pts" << *better_ligand_jumps << std::endl;
	core::kinematics::FoldTreeOP with_cutpoints = create_fold_tree_with_cutpoints(*better_ligand_jumps, interface);
	minimize_backbone_tracer.Debug  << "foldtree with cutpoints" << *with_cutpoints << std::endl;
	assert(with_cutpoints->check_edges_for_atom_info());
	with_cutpoints->delete_extra_vertices();
	assert(with_cutpoints->check_edges_for_atom_info());
	minimize_backbone_tracer.Debug << "foldtree with less vertices" << *with_cutpoints << std::endl;
	assert(with_cutpoints->check_edges_for_atom_info());
	reorder_with_first_non_mobile_as_root(*with_cutpoints, interface);
	minimize_backbone_tracer.Debug << "Final loops foldtree " << *with_cutpoints << std::endl;
	if (!with_cutpoints->check_fold_tree()) {
		utility_exit_with_message("Invalid fold tree after trying to set up for minimization!");
	}
	assert(with_cutpoints->check_edges_for_atom_info());
	pose_.fold_tree(*with_cutpoints);
	assert(pose_.fold_tree().check_edges_for_atom_info());
}

void Minimize_backbone::reorder_with_first_non_mobile_as_root(
		core::kinematics::FoldTree & f,
		Interface const & interface
) {
	for (Size i = 1; i <= pose_.n_residue(); ++i) {
		if (pose_.residue(i).is_polymer() && interface[i].type == InterfaceInfo::non_interface) {
			f.reorder(i);
			return;
		}
	}
	utility_exit_with_message("Every residue is mobile!  We need a non-mobile residue");
}

core::kinematics::FoldTreeOP Minimize_backbone::create_fold_tree_with_cutpoints(
		core::kinematics::FoldTree const & f,
		Interface const & interface
) {
	core::kinematics::FoldTreeOP f_new= new core::kinematics::FoldTree();// Deleting edges is bad so add them to a new foldtree

	int new_jump = f.num_jump();

	for (
			core::kinematics::FoldTree::const_iterator edge_itr = f.begin();
			edge_itr != f.end();
			++edge_itr
		) {

		if (!edge_itr->is_polymer()) {
			f_new->add_edge(*edge_itr);
			continue; // only subdivide "peptide" edges of the fold tree
		}

		utility::vector1< protocols::loops::Loop> loops = add_cut_points( *edge_itr, interface);

		const int e_start = edge_itr->start();
		const int e_stop = edge_itr->stop();

		int last_rigid = e_start;
		for (Size i = 1; i <= loops.size(); ++i) {
			protocols::loops::Loop const & l = loops[i];
			int first = std::max(int(e_start), int(l.start() - 1));
			int last = std::min(int(e_stop), int(l.stop() + 1));
			if (first != last_rigid) {
				f_new->add_edge(last_rigid, first, core::kinematics::Edge::PEPTIDE);
			}
			f_new->add_edge(first, l.cut(), core::kinematics::Edge::PEPTIDE);
			f_new->add_edge(core::kinematics::Edge(first, last, ++new_jump, "CA", "CA", false ));
			f_new->add_edge(last, l.cut() + 1, core::kinematics::Edge::PEPTIDE);

			last_rigid = last;
		}

		if (last_rigid != int(e_stop)) {
			f_new->add_edge(last_rigid, e_stop, core::kinematics::Edge::PEPTIDE);
		}
	}
	return f_new;
}

utility::vector1< protocols::loops::Loop> Minimize_backbone::add_cut_points(
		core::kinematics::Edge const & edge, Interface const & interface
) {
	static numeric::random::RandomGenerator my_RG(4376910); // 4376910 is just a random seed
	if (edge.start() > edge.stop())
		utility_exit_with_message("Not prepared to deal with backwards fold tree edges!");
	utility::vector1< protocols::loops::Loop> loops;

	core::Size start= interface.find_first_interface_residue(edge.start(), edge.stop());
	core::Size stop= 0;

	while(start ){
		stop= interface.find_stop_of_this_interface_region(start, edge.stop());

		minimize_backbone_tracer.Debug << "start,stop:"<< start << ','<< stop << std::endl;
		minimize_backbone_tracer.Debug << "edge.start,edge.stop:"<< edge.start() << ','<< edge.stop() << std::endl;
		runtime_assert( stop >= start );
		if ((stop - start + 1) < 4) {
			minimize_backbone_tracer.Debug
					<< "WARNING: for backbone minimization to work properly, a stretch of at least 4 residues needs to be allowed to move. Stretch between "
					<< start << " and " << stop << " is too short.  This should never happen after extending the interface"
					<< std::endl;
			continue;
		}

		// Cutpoint should fall between start and stop, but not if the rsd on either side is a terminus.
		// Because this happens within one peptide edge, no "internal" residue should ever be a terminus.
		core::Size const cut_start = (pose_.residue(start).is_terminus() ? start + 1 : start);
		core::Size const cut_end = ( pose_.residue(stop).is_terminus() ? stop - 2 : stop - 1);

		runtime_assert( cut_start <= cut_end );

		core::Size cutpt = Size(my_RG.random_range(cut_start, cut_end)); // cut is made between cutpt and cutpt+1

		loops.push_back(protocols::loops::Loop(start, stop, cutpt));
		// also need to set up residue variants so chainbreak score works correctly!
		core::chemical::add_variant_type_to_pose_residue(pose_, core::chemical::CUTPOINT_LOWER, cutpt);
		core::chemical::add_variant_type_to_pose_residue(pose_, core::chemical::CUTPOINT_UPPER, cutpt + 1);

		start= interface.find_start_of_next_interface_region(stop+1,  edge.stop());
	}
	return loops;
}

core::kinematics::FoldTreeOP Minimize_backbone::create_fold_tree_with_ligand_jumps_from_attach_pts(
		core::kinematics::FoldTree const & f_const,
		Interface const & interface

) {
	std::map<core::Size, core::Size> jump_to_attach = find_attach_pts(interface);

	core::kinematics::FoldTreeOP new_fold_tree =
			new core::kinematics::FoldTree();

	for (
			core::kinematics::FoldTree::const_iterator e = f_const.begin(),
			edge_end = f_const.end();
			e != edge_end;
			++e
	) {
		std::map<core::Size, core::Size>::const_iterator const jump_attach = jump_to_attach.find(e->label());
		if (jump_attach != jump_to_attach.end()) {
			core::Size const jump_id = jump_attach->first;
			core::Size const attach_pt = jump_attach->second;
			core::Size const ligand_residue_id = pose_.fold_tree().downstream_jump_residue(jump_id);
			new_fold_tree->add_edge(attach_pt, ligand_residue_id, jump_id);
		} else {
			core::Size const attach_pt= find_peptide_attach_pt(e, jump_to_attach);

			if (e->label() == core::kinematics::Edge::PEPTIDE && attach_pt != 0) {
				new_fold_tree->add_edge(e->start(), attach_pt, core::kinematics::Edge::PEPTIDE);
				new_fold_tree->add_edge(attach_pt, e->stop(), core::kinematics::Edge::PEPTIDE);
			} else if (e->label() == core::kinematics::Edge::CHEMICAL){
				new_fold_tree->add_edge(e->start(), e->stop(), e->start_atom(), e->stop_atom());
			} else {// add a jump edge
				new_fold_tree->add_edge(e->start(), e->stop(), e->label());
			}
		}
	}

	if (!new_fold_tree->check_fold_tree())
		utility_exit_with_message("Fold tree did not pass check!");
	if (!new_fold_tree->check_edges_for_atom_info())
		utility_exit_with_message("Fold tree has chemical edge problems!");
	if (f_const.nres() != new_fold_tree->nres())
		utility_exit_with_message("Number of residues changed?!");
	if (f_const.num_jump() != new_fold_tree->num_jump())
		utility_exit_with_message("Number of jumps changed?!");
	if (!new_fold_tree->connected())
		utility_exit_with_message("Fold tree not connected?!");
	return new_fold_tree;
}

core::Size Minimize_backbone::find_peptide_attach_pt (
		core::kinematics::FoldTree::const_iterator const e,
		std::map<core::Size, core::Size> const jump_to_attach
) const {
	int const start = e->start();
	int const stop = e->stop();
	runtime_assert(start < stop);
	std::map<core::Size, core::Size>::const_iterator index =
			jump_to_attach.begin();
	std::map<core::Size, core::Size>::const_iterator const end =
			jump_to_attach.end();
	for (; index != end; ++index) {
		int const attach_pt = (int) index->second;

		if (start < attach_pt && attach_pt < stop){
			return index->second;
		}
		if (stop < attach_pt && attach_pt < start){
			return index->second;
		}
	}
	return 0;
}

std::map<core::Size, core::Size> Minimize_backbone::find_attach_pts(
		Interface const interface
) const {
	std::map<core::Size, core::Size> jumpToAttach;
	std::map<core::Size, core::Real>::const_iterator index = chains_.begin();
	for (; index != chains_.end(); ++index) { // these are just the ligand chains to dock
		core::Size jump_id = get_jump_id_from_chain_id(index->first, pose_);
		jumpToAttach[jump_id] = find_attach_pt(jump_id, interface);
	}
	return jumpToAttach;
}

core::Size Minimize_backbone::find_attach_pt(
		const core::Size jump_id,
		const Interface interface
)const {
	core::Size attach_pt = 1; // for now, attach ligand to residue 1
	core::Real shortest_dist2 = 1e99;
	//core::Vector center= protocols::geometry::downstream_centroid_by_jump(pose_, jump_id);
	const core::Size residue_id = pose_.fold_tree().downstream_jump_residue(jump_id);
	const core::conformation::Residue & residue = pose_.residue(residue_id);
	const core::Vector lig_nbr_vector = residue.nbr_atom_xyz();
	//for(core::Size i = 1; i <= pose_.n_residue(); ++i) { // only look at upstream residues, else multi-residue ligand will attach to itself
	for (core::Size i = 1; i < residue_id; ++i) {
		if (!pose_.residue(i).is_protein() && interface[i].type != InterfaceInfo::non_interface) {
			core::Real const new_dist2 = lig_nbr_vector.distance_squared(
					pose_.residue(i).xyz("CA"));
			if (new_dist2 < shortest_dist2) {
				shortest_dist2 = new_dist2;
				attach_pt = i;
			}
		}
	}

	return attach_pt;
}

///@detail: residues that are near the interface residues will be allowed to move with restraint
void Minimize_backbone::restrain_protein_Calphas(
		Interface const & interface
) {
	core::id::AtomID fixed_pt(pose_.atom_tree().root()->atom_id());
	minimize_backbone_tracer.Debug << interface << std::endl;
	for (core::Size residue_id = 1; residue_id <= interface.size(); ++residue_id) { // didn't use an iterator because pose needs a #
		core::conformation::Residue const & residue = pose_.residue(residue_id);
		if (interface[residue_id].type != InterfaceInfo::non_interface) {
			core::Size const ligand_chain_id= interface[residue_id].chain;
			core::id::AtomID const atom_ID(residue.atom_index("CA"), residue_id);
			core::scoring::constraints::FuncOP const harmonic_function= new core::scoring::constraints::HarmonicFunc(0, chains_[ligand_chain_id]);
			core::scoring::constraints::ConstraintOP const constraint =
				new core::scoring::constraints::CoordinateConstraint(
					atom_ID,
					fixed_pt,
					residue.xyz("CA"),
					harmonic_function
				)
			;
			minimize_backbone_tracer.Debug << "Restraining C-alpha of residue " << residue_id << " with "<< chains_[ligand_chain_id]<< std::endl;
			pose_.add_constraint(constraint);
		}
	}
}

const std::map<core::Size, core::Real> &
Minimize_backbone::get_const_ligands_reference() {
	return chains_;
}

void Minimize_backbone::restrict_to_protein_residues(Interface & interface)const{
	for(core::Size i=1; i <= interface.size(); ++i){
		if(interface[i].type == InterfaceInfo::is_interface
				&& pose_.residue(i).is_ligand()
		){
			interface[i].type = InterfaceInfo::non_interface;
		}
	}
}


} //namespace ligand_options
} //namespace ligand_docking
} //namespace protocols
