// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   core/pack/task/ResfileReader.cc
/// @brief  implementation of resfile reader and its command classes
/// @author Gordon Lemmon (glemmon@gmail.com), adapted from the ResfileReader code
/// by Steven Lewis (smlewi@unc.edu) and Andrew Leaver-Fay

// Unit Headers
#include <protocols/ligand_docking/ligand_options/Minimize_ligand.hh>
//#include <protocols/ligand_docking/ligand_options/InterfaceBuilder.hh>

// Utility Headers
#include <numeric/random/random.hh>
#include <utility/io/izstream.hh>
#include <utility/exit.hh>
#include <utility/assert.hh> //ASSERT_ONLY makes release build happy
#include <utility/string_util.hh>
#include <core/util/Tracer.hh>
#include <core/types.hh>
using core::util::T;
using core::util::Error;
using core::util::Warning;

//STL headers
//#include <string>
//#include <iostream> //need this for debugging
//#include <fstream>
//#include <sstream> //automatic checking string to int conversion
//#include <cctype> //for split_lines to handle '\t' tab characters
//#include <cmath> // for std::max, etc
//#include <stdlib.h>
//#include <set>

namespace protocols {
namespace ligand_docking {
namespace ligand_options {

Minimize_ligand::Minimize_ligand(core::pose::Pose & pose) :
	LigandOptionCommand(pose), LigandSpecificCommand(pose),
			DefaultCommand(pose), applied_(false) {
	ligand_torsion_restraints_.clear();
}

void Minimize_ligand::apply() {
	if (!applied_) {
		std::map<core::Size, core::Real>::iterator i = chains_.begin();
		for (; i != chains_.end(); ++i) {
			core::Size j = pose_.conformation().chain_begin(i->first);
			core::Size const end = pose_.conformation().chain_end(i->first);
			for (; j <= end; ++j) {
				ligand_torsion_restraints_.insert(
					new protocols::ligand_docking::ResidueTorsionRestraints(pose_, j, i->second));
			}
		}
		applied_ = true;
	}
}

void Minimize_ligand::option(
		utility::vector1< std::string> const & tokens,
		std::set<core::Size> const & ligands_to_dock
) {
	if (tokens.size() != 2) {
		utility_exit_with_message("'minimize_ligand' takes one argument (Real)");
	}
	core::Real stdev = strtod(tokens[2].c_str(), NULL);

	std::set<core::Size>::const_iterator i = ligands_to_dock.begin();
	for (; i != ligands_to_dock.end(); i++) {
		chains_[*i] = stdev;
	}
}

void Minimize_ligand::option(
		utility::vector1< std::string> const & tokens,
		core::Size const & ligand_chain
) {
	if (tokens.size() != 2) {
		utility_exit_with_message("'minimize_ligand' takes one argument (Real)");
	}
	core::Real stdev = strtod(tokens[2].c_str(), NULL);
	chains_[ligand_chain] = stdev;
}

const std::map<core::Size, core::Real> &
Minimize_ligand::get_const_ligands_reference() {
	return chains_;
}

const std::set< protocols::ligand_docking::ResidueTorsionRestraintsOP> &
Minimize_ligand::get_const_ligand_torsion_restraints_reference() const {
	return ligand_torsion_restraints_;
}

} //namespace ligand_options
} //namespace ligand_docking
} //namespace protocols
