// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   core/pack/task/ResfileReader.cc
/// @brief  implementation of resfile reader and its command classes
/// @author Gordon Lemmon (glemmon@gmail.com), adapted from the ResfileReader code
/// by Steven Lewis (smlewi@unc.edu) and Andrew Leaver-Fay

// Unit Headers
#include <protocols/ligand_docking/ligand_options/ProtocolOption.hh>
#include <protocols/ligand_docking/ligand_options/InterfaceBuilder.hh>
#include <protocols/ligand_docking/ligand_options/Interface.hh>

#include <protocols/docking/DockingInitialPerturbation.hh>
#include <protocols/ligand_docking/UnconstrainedTorsionsMover.hh>
#include <protocols/moves/MinMover.hh>
#include <protocols/moves/PackRotamersMover.hh>
#include <protocols/moves/RotamerTrialsMover.hh>
#include <protocols/moves/RigidBodyMover.hh>
#include <protocols/moves/MonteCarlo.hh>

#include <core/chemical/ResidueTypeSet.hh>
#include <core/optimization/MinimizerOptions.hh>
#include <core/kinematics/MoveMap.hh>
#include <core/pack/task/TaskFactory.hh>
#include <core/pack/task/operation/TaskOperations.hh>
#include <core/pack/rotamer_set/UnboundRotamersOperation.hh>

// Package Headers
#include <core/pose/Pose.hh>

// Project Headers
#include <core/chemical/ResidueType.hh>
#include <core/options/option.hh>
#include <core/options/keys/packing.OptionKeys.gen.hh>

// Utility Headers
#include <utility/assert.hh> //ASSERT_ONLY makes release build happy
#include <utility/string_util.hh>
#include <core/types.hh>

//STL headers
#include <string>

#include <set>

namespace protocols {
namespace ligand_docking {
namespace ligand_options {


Protocol::Protocol(
		core::pose::Pose & pose,
		core::scoring::ScoreFunctionOP const & score_fxn,
		std::map<core::Size, core::Real> const & ligand_chains_to_minimize,
		std::set< protocols::ligand_docking::ResidueTorsionRestraintsOP > const ligand_torsion_restraints,
		std::map<core::Size, core::Real> const & ligands_around_which_to_minimize_bb,
		std::set<core::Size> const & ligands_with_rotamers
):
		LigandOptionCommand(pose),
		DefaultCommand(pose),
		pose_(pose),
		score_fxn_(score_fxn),
		ligand_chains_to_minimize_(ligand_chains_to_minimize),
		ligand_torsion_restraints_(ligand_torsion_restraints),
		ligands_around_which_to_minimize_bb_(ligands_around_which_to_minimize_bb),
		ligands_with_rotamers_(ligands_with_rotamers)
{

	enum_map_["meiler2006"]=meiler2006;
	enum_map_["abbreviated"]= abbreviated;
	enum_map_["abbrev2"]= abbrev2;
	enum_map_["min_only"]= min_only;
	enum_map_["custom"]= custom;

	set_cycles(abbrev2);/// default option
}

void Protocol::option(
	utility::vector1< std::string > const & tokens,
	std::set<core::Size> const & ligands_to_dock
){
	if(tokens.size()< 2 ){
		utility_exit_with_message("'protocol' option must provide ONE protocol argument.\
			Choices are meiler2006, abbreviated, abbrev2, min_only, or 'custom int int'");
	}
	std::string protocol_name= tokens[2];
	std::map< std::string, protocol_enum >::const_iterator const enum_pair= enum_map_.find(protocol_name);
	if(enum_pair == enum_map_.end()){
		utility_exit_with_message("In your ligand option file, you specified a protocol that doesn't exist");
	}
	protocol_enum p= enum_pair->second;
	if(tokens.size()==2){
		set_cycles(p);
	}
	else if(tokens.size()==4){
		set_cycles(p, tokens[3], tokens[4]);
	}
	else{
		utility_exit_with_message("'protocol' options take 0 args except for custom which takes 2 ints");
	}

	ligands_to_dock_.insert(ligands_to_dock.begin(), ligands_to_dock.end());
}

void Protocol::set_cycles(protocol_enum const p, std::string const num_cycles, std::string const repack_every_Nth){
	switch (p){
		case meiler2006:
			num_cycles_= 50;
			repack_every_Nth_= 8;
			break;
		case abbreviated:
			num_cycles_= 5;
			repack_every_Nth_= 4;
			break;
		case abbrev2:
			num_cycles_= 6;
			repack_every_Nth_= 3;
			break;
		case min_only:
			num_cycles_=0;
			repack_every_Nth_=0;
		case custom:
			num_cycles_= atoi(num_cycles.c_str());
			repack_every_Nth_= atoi(repack_every_Nth.c_str());;
			break;
		default:
			utility_exit_with_message("'protocol' option must provide ONE protocol argument.\
				Choices are rescore, meiler2006, abbreviated, abbrev2, min_only, or 'custom int int'");
	}
}

//const std::set<core::Size> Protocol::get_docking_jump_ids() const{
//	return jump_ids_;
//}

core::kinematics::MoveMapOP
Protocol::make_movemap(
		bool const minimize_bb,
		bool const minimize_water
)const{
	// All DOF start false (frozen)
	core::kinematics::MoveMapOP movemap = new core::kinematics::MoveMap();

	std::set<core::Size>::const_iterator chain_id= ligands_to_dock_.begin();
	for(; chain_id!= ligands_to_dock_.end(); ++chain_id){
		core::Size jump_id= get_jump_id_from_chain_id(*chain_id, pose_);
		movemap->set_jump(jump_id, true);
	}

	set_all_chi(movemap);
	if(minimize_bb) set_all_bb(movemap);

	if( minimize_water ) {
		for(core::Size i = 1, i_end = pose_.total_residue(); i <= i_end; ++i) {
			if( ! pose_.residue(i).has_property("WATER") ) continue;
			core::kinematics::Edge const & e = pose_.fold_tree().get_residue_edge(i);
			if( ! e.is_jump() ) continue;
			movemap->set_jump( e.label(), true );
			protocol_tracer << "Minimize water jump " << e.label() << " to residue " << i << " " << pose_.residue_type(i).name3() << std::endl;
		}
	}
	//movemap->show(pose_.total_residue());
	return movemap;
}

void
Protocol::set_all_chi(core::kinematics::MoveMapOP movemap)const{
	InterfaceBuilder side_chain_interface_builder(pose_, ligands_to_dock_, 6.0, true, true, 0);
	Interface side_chain_interface= side_chain_interface_builder.build();
	remove_ligands_to_not_minimize(side_chain_interface);
	protocol_tracer.Debug<< "moveMap interface: "<< side_chain_interface << std::endl;
	for(core::Size i=1; i <= side_chain_interface.size(); ++i) {
		if ( side_chain_interface[i].type != InterfaceInfo::non_interface) { // Allow residue to minimize
			movemap->set_chi(i, true);
		}
	}
}

///@details You MUST call set_all_chi first
void
Protocol::set_all_bb(core::kinematics::MoveMapOP movemap)const{
	InterfaceBuilder bb_interface_builder(pose_, ligands_around_which_to_minimize_bb_, 7.0, false, true, 3);
	Interface bb_interface = bb_interface_builder.build();

	for(core::Size i=1; i <= bb_interface.size(); ++i) {
		if ( bb_interface[i].type != InterfaceInfo::non_interface
				&& pose_.residue(i).is_protein()
				&& movemap->get_chi(i)
		) { // Allow residue to minimize
			movemap->set_bb(i, true);
		}
	}
}

core::pack::task::PackerTaskOP
Protocol::make_packer_task_from_vector(Interface const allow_repack) const{
	core::pack::task::PackerTaskOP pack_task = core::pack::task::TaskFactory::create_packer_task(pose_);
	pack_task->initialize_from_command_line(); // -ex1 -ex2  etc.

	core::pack::rotamer_set::UnboundRotamersOperationOP unboundrot_ = new core::pack::rotamer_set::UnboundRotamersOperation();
	unboundrot_->initialize_from_command_line();
	pack_task->append_rotamerset_operation( unboundrot_ );

	bool const use_resfile= core::options::option[ core::options::OptionKeys::packing::resfile ].user() ;

	for(core::Size i = 1; i <= pose_.total_residue(); ++i) {
		if( pose_.residue(i).is_ligand() && ligands_with_rotamers_.find(i) != ligands_with_rotamers_.end() ) {
			enable_ligand_rotamer_packing(i, pack_task);
		}
	}

	if (use_resfile){
		protocol_tracer<< "using resfile"<< std::endl;
		pack_task->read_resfile();
	}
	else{
		protocol_tracer<< "restricting to repack"<< std::endl;
		for(core::Size i = 1; i <= pose_.total_residue(); ++i) {
			if( ! pose_.residue(i).is_ligand() )
			{
				pack_task->nonconst_residue_task( i ).restrict_to_repacking();
			}
			if ( allow_repack[i].type == InterfaceInfo::non_interface  )
				pack_task->nonconst_residue_task( i ).prevent_repacking();
		}
	}
	return pack_task;
}

core::pack::task::PackerTaskOP
Protocol::make_packer_task(bool all_residues) const{
	if ( all_residues ){
		Interface interface(pose_.n_residue(), InterfaceInfo(InterfaceInfo::is_interface)); // 0 is false, #
		return make_packer_task_from_vector(interface);
	}else{
		InterfaceBuilder side_chain_interface_builder(pose_, ligands_to_dock_, 6.0, true, true, 0);
		Interface side_chain_interface= side_chain_interface_builder.build();
		remove_ligands_to_not_minimize(side_chain_interface);
		return make_packer_task_from_vector(side_chain_interface);
	}
}

void
Protocol::remove_ligands_to_not_minimize(
		Interface & interface
)const{
	for ( Size i = 1; i <= pose_.conformation().num_chains(); ++i ) {
		core::Size begin = pose_.conformation().chain_begin( i );
		if(pose_.residue(begin).is_ligand()){
			std::map<core::Size, core::Real>::const_iterator found= ligand_chains_to_minimize_.find(i);
			if( found == ligand_chains_to_minimize_.end()){
				core::Size const end = pose_.conformation().chain_end( i );
				for(; begin <= end; ++begin) interface[begin].type= InterfaceInfo::non_interface;
			}
		}
	}
}

void
Protocol::enable_ligand_rotamer_packing(
		const core::Size ligand_residue_id,
		core::pack::task::PackerTaskOP & pack_task
) const{
	core::conformation::Residue const & this_residue= pose_.residue(ligand_residue_id);
	core::chemical::ResidueTypeSet const & rsd_type_set = this_residue.residue_type_set();
	core::chemical::ResidueTypeCAPs allowed_types = rsd_type_set.name3_map( this_residue.name3() ); // a vector1

	for( core::Size j = 1; j <= allowed_types.size(); ++j ) {
		if( allowed_types[j]->name() == this_residue.name() ) continue; // already in the task's list
		///TODO figure out why this is nonconst.  Perhaps it could be const
		pack_task->nonconst_residue_task( ligand_residue_id ).allow_noncanonical_aa( allowed_types[j]->name() );
	}
}

utility::vector1<protocols::moves::MoverOP>
Protocol::create_rigid_body_movers() const{
	utility::vector1<protocols::moves::MoverOP> rigid_body_movers;

	std::set<core::Size>::const_iterator chain_id= ligands_to_dock_.begin();
	for(; chain_id!= ligands_to_dock_.end(); ++chain_id){
		core::Size jump_id= get_jump_id_from_chain_id(*chain_id, pose_);
		protocols::moves::MoverOP rigid_body_mover= new protocols::moves::RigidBodyPerturbMover( jump_id, numeric::conversions::degrees(0.05), 0.1);
		rigid_body_movers.push_back(rigid_body_mover);
	}
	return rigid_body_movers;
}

void Protocol::apply_rigid_body_moves(
		utility::vector1<protocols::moves::MoverOP> & rigid_body_movers
){
	utility::vector1<protocols::moves::MoverOP>::iterator rigid_body_mover= rigid_body_movers.begin();
	for(; rigid_body_mover != rigid_body_movers.end(); ++rigid_body_mover){
		(*rigid_body_mover)->apply(pose_);
	}
}

void Protocol::apply(){
	core::kinematics::MoveMapOP movemap = make_movemap();
	protocols::moves::MonteCarloOP monteCarlo = new protocols::moves::MonteCarlo(pose_, *score_fxn_, 2.0);/* temperature, from RosettaLigand paper */
	score_fxn_->score( pose_ ); // without this neither of the movers below were working
	// I believe that this may have been related to adding constraints incorrectly at other places in my code.
	// Rigid body exploration
	utility::vector1<protocols::moves::MoverOP> rigid_body_movers= create_rigid_body_movers();
	for( core::Size cycle = 1; cycle <= num_cycles_; ++cycle ) {
		core::pack::task::PackerTaskOP packer_task = make_packer_task();
		protocols::moves::MoverOP pack_mover =
			(cycle % repack_every_Nth_ == 1) ?
			(protocols::moves::Mover *) new protocols::moves::PackRotamersMover(score_fxn_, packer_task)
			:(protocols::moves::Mover *) new protocols::moves::RotamerTrialsMover(score_fxn_, *packer_task)
		;
		// Wrap it in something to disable the torsion constraints before packing!
		pack_mover = new protocols::ligand_docking::UnconstrainedTorsionsMover( pack_mover, ligand_torsion_restraints_ );

		protocols::moves::MinMoverOP min_mover = new protocols::moves::MinMover( movemap, score_fxn_, "dfpmin_armijo_nonmonotone_atol", 1.0, true /*use_nblist*/ );
		min_mover->min_options()->nblist_auto_update(true); // does this cost us lots of time in practice?

		core::Real const score1 = (*score_fxn_)( pose_ );
		apply_rigid_body_moves(rigid_body_movers);
		pack_mover->apply(pose_);

		core::Real const score2 = (*score_fxn_)( pose_ );
		if(score2 - score1 < 15.0) {
			min_mover->apply(pose_);
		}

		monteCarlo->boltzmann( pose_ );

		// We always want the option (after the initial unbiased pack)
		// of sticking with our current nicely minimized conformation.
		packer_task->or_include_current(true);
	}

	// keep the best structure we found, not the current one
	monteCarlo->show_scores();
	monteCarlo->recover_low(pose_);
}

} //namespace ligand_options
} //namespace ligand_docking
} //namespace protocols
