// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   core/pack/task/ResfileReader.cc
/// @brief  implementation of resfile reader and its command classes
/// @author Gordon Lemmon (glemmon@gmail.com), adapted from the ResfileReader code
/// by Steven Lewis (smlewi@unc.edu) and Andrew Leaver-Fay

// Unit Headers
#include <protocols/ligand_docking/ligand_options/Random_conformer.hh>
#include <protocols/ligand_docking/ligand_options/InterfaceBuilder.hh>

#include <protocols/ligand_docking/RandomConformerMover.hh>
//#include <protocols/docking/DockingInitialPerturbation.hh>
#include <protocols/ligand_docking/UnconstrainedTorsionsMover.hh>

// Utility Headers
#include <numeric/random/random.hh>
#include <utility/io/izstream.hh>
#include <utility/exit.hh>
#include <utility/assert.hh> //ASSERT_ONLY makes release build happy
#include <utility/string_util.hh>
#include <core/util/Tracer.hh>
#include <core/types.hh>
using core::util::T;
using core::util::Error;
using core::util::Warning;

namespace protocols {
namespace ligand_docking {
namespace ligand_options {

///////////////////////////////////////////////////////////////////////
///@brief
Random_conformer::Random_conformer(
		core::pose::Pose & pose,
		std::set< protocols::ligand_docking::ResidueTorsionRestraintsOP> const & restraints) :
	LigandOptionCommand(pose), LigandSpecificCommand(pose),
			DefaultCommand(pose), ligand_torsion_restraints_(restraints) {
}

void Random_conformer::option(
		utility::vector1< std::string> const & tokens,
		std::set<core::Size> const & ligands_to_dock
) {
	if (tokens.size() != 1) {
		utility_exit_with_message("'Random_conformer' is a boolean option and does not take arguments");
	}
	chains_.insert(ligands_to_dock.begin(), ligands_to_dock.end());
}

void Random_conformer::option(
		utility::vector1< std::string> const & /* tokens*/,
		core::Size const & ligand_chain) {
	chains_.insert(ligand_chain);
}

void Random_conformer::apply() {
	std::set<core::Size>::iterator i = chains_.begin();
	std::set<core::Size>::iterator end = chains_.end();
	for (; i != end; ++i) {
		apply_chain(*i);
	}
}

void Random_conformer::apply_chain(core::Size const chain_id) {
	core::Size i = pose_.conformation().chain_begin(chain_id);
	core::Size end = pose_.conformation().chain_end(chain_id);

	for (; i != end; ++i) {
		apply_residue(i);
	}
}

void Random_conformer::apply_residue(core::Size const residue_id) {
	using namespace protocols::moves;
	using core::conformation::ResidueOP;
	RandomConformerMoverOP rcm = new RandomConformerMover(residue_id);
	UnconstrainedTorsionsMoverOP utm =
			new UnconstrainedTorsionsMover(rcm, ligand_torsion_restraints_);
	utm->apply(pose_);
}

} //namespace ligand_options
} //namespace ligand_docking
} //namespace protocols
