// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   core/pack/task/ResfileReader.cc
/// @brief  implementation of resfile reader and its command classes
/// @author Gordon Lemmon (glemmon@gmail.com)

// Unit Headers
#include <protocols/ligand_docking/ligand_options/Rotate.hh>
#include <protocols/ligand_docking/ligand_options/InterfaceBuilder.hh>

#include <protocols/ligand_docking/grid_functions.hh>
#include <protocols/geometry/RB_geometry.hh>
#include <protocols/moves/RigidBodyMover.hh>
#include <protocols/ligand_docking/ligand_options/chain_functions.hh>
#include <protocols/ligand_docking/grid_functions.hh>

#include <core/conformation/Conformation.hh>

// Utility Headers
#include <numeric/random/random.hh>
#include <utility/io/izstream.hh>
#include <utility/exit.hh>
#include <utility/assert.hh> //ASSERT_ONLY makes release build happy
#include <utility/string_util.hh>
#include <core/util/Tracer.hh>
#include <core/types.hh>
#include <algorithm>
using core::util::T;
using core::util::Error;
using core::util::Warning;

namespace protocols {
namespace ligand_docking {
namespace ligand_options {

///@brief
Rotate::Rotate(core::pose::Pose & pose) :
	LigandOptionCommand(pose), DefaultCommand(pose), LigandSpecificCommand(pose), InitialMoverOption(pose)
{}

Rotate_info
Rotate::parse_tokens(
		utility::vector1< std::string> const & tokens
) const{
	if (tokens.size() < 1 ) {
		utility_exit_with_message("This should be impossible because arg 1 got us here");
	}
	if (tokens.size() > 4 ) {
			utility_exit_with_message("'Rotate' has too many arguments");
	}

	Rotate_info rotate_info;
	if (tokens.size() > 1){
		std::string distribution= tokens[2].c_str();
		std::map< std::string, Distribution >::const_iterator iter= distribution_map_.find(distribution);
		rotate_info.distribution= iter->second;

		if(tokens.size() ==2){
			rotate_info.degrees= 360;
			rotate_info.cycles=1000;
		}
	}
	if (tokens.size() > 2){
		rotate_info.degrees= atoi(tokens[3].c_str());

		if (tokens.size() == 3){
			rotate_info.cycles= 1000;
		}
	}
	if (tokens.size() > 3){
		rotate_info.cycles= atoi(tokens[4].c_str());
	}
	return rotate_info;
}

void Rotate::option(
		utility::vector1< std::string> const & tokens,
		std::set<core::Size> const & ligands_to_dock
) {
	Rotate_info rotate_info= parse_tokens(tokens);
	option(rotate_info, ligands_to_dock);
}

void Rotate::option(
		Rotate_info const rotate_info,
		std::set<core::Size> const & ligands_to_dock
) {
	std::set<core::Size>::const_iterator i = ligands_to_dock.begin();
	for (; i != ligands_to_dock.end(); ++i) {
		chains_[*i] = rotate_info;
	}
}

void Rotate::option(
		utility::vector1< std::string> const & tokens,
		core::Size const & ligand_chain
) {
	chains_[ligand_chain] = parse_tokens(tokens);
}

void Rotate::apply() {
	std::map<core::Size, Rotate_info>::iterator i = chains_.begin();

	for (; i != chains_.end(); ++i) {
		apply(i->first, i->second);
	}
}

void Rotate::apply(
		core::Size chain_id,
		Rotate_info const rotate_info
) {
	core::Size const jump_id = get_jump_id_from_chain_id(chain_id, pose_);
	core::Vector const center = protocols::geometry::downstream_centroid_by_jump(pose_, jump_id);

	core::grid::CartGridOP const grid = make_atr_rep_grid(pose_, center);

	rotate_ligand(grid, jump_id, chain_id, rotate_info);// move ligand to a random point in binding pocket

}

utility::vector1< Ligand_info> Rotate::create_random_rotations(
		core::grid::CartGridOP const & grid,
		core::Size const cycles,
		protocols::moves::RigidBodyMoverOP const mover,
		core::Size const jump_id,
		core::Size const chain_id,
		core::Size const begin
)const{
	core::Size const end = pose_.conformation().chain_end(chain_id);
	core::Size const heavy_atom_number= num_heavy_atoms(begin, end, pose_);
	core::pose::Pose local_pose= pose_;
	local_pose.remove_constraints();
	core::Vector const center = protocols::geometry::downstream_centroid_by_jump(local_pose, jump_id);

	utility::vector1< Ligand_info> ligands;  ///TODO make this a set.  The set should check for another pose with a similar RMSD.
	core::Size max_diversity= 5*num_chi_angles(begin, end, local_pose)+1; // who knows why?  copied from Ian's code.
	Ligand_info best;
	for(core::Size i=0; i< cycles && ligands.size() <= max_diversity ; ++i){
		apply_rotate(mover, local_pose, center);
		rb_grid_rotamer_trials_atr_rep(*grid, local_pose, begin, end);
		core::kinematics::Jump jump= local_pose.jump(jump_id);
		std::pair<int, int> const scores= get_rb_atr_and_rep_scores(*grid, local_pose, begin, end);
		core::pose::Pose const ligand= local_pose.split_by_chain(chain_id);
		Ligand_info const ligand_info(ligand, scores, jump);
		if (i==0 || ligand_info < best){
			best= ligand_info;
		}
		add_ligand_conditionally(ligand_info, ligands, heavy_atom_number);
	}
	if(ligands.empty()){
		ligands.push_back(best);
	}
	return ligands;
}

///@brief for n random rotations, randomly pick one from among the best scoring set of diverse poses
void Rotate::rotate_ligand(
		core::grid::CartGridOP const & grid,
		core::Size const jump_id,
		core::Size const chain_id,
		Rotate_info const rotate_info
) {
	if(rotate_info.degrees == 0) return;

	protocols::moves::RigidBodyMoverOP mover;
	if(rotate_info.distribution == uniform){
		mover= new protocols::moves::RigidBodyRandomizeMover( pose_, jump_id, protocols::moves::partner_downstream, rotate_info.degrees, rotate_info.degrees);
	}
	else if(rotate_info.distribution == gaussian){
		mover= new protocols::moves::RigidBodyPerturbMover ( jump_id, rotate_info.degrees, 0 /*translate*/);
	}
	core::Size const begin = pose_.conformation().chain_begin(chain_id);

	utility::vector1< Ligand_info> ligands= create_random_rotations(grid, rotate_info.cycles, mover, jump_id, chain_id, begin);

	core::Size const jump_choice=  (core::Size) numeric::random::RG.random_range(1, ligands.size());
	{
		pose_.set_jump(jump_id, ligands[jump_choice].jump);
		core::pose::Pose & new_chain = ligands[jump_choice].pose;
		pose_.copy_segment(new_chain.n_residue(), new_chain, begin, 1);// is one always first?
	}
}

void add_ligand_conditionally(
		Ligand_info const & ligand_info,
		utility::vector1< Ligand_info> & ligands,
		core::Size const heavy_atom_number
){
	if(
			check_score(ligand_info, heavy_atom_number)
			&& check_RMSD(ligand_info, heavy_atom_number, ligands)
	){
		ligands.push_back(ligand_info);
	}
}

void apply_rotate(
		protocols::moves::RigidBodyMoverOP mover,
		core::pose::Pose & pose,
		core::Vector const & center
){
	mover->apply(pose);
	pose.update_actcoords();///TODO Verify necessity
	mover->rot_center(center); // restore the old center so ligand doesn't walk away (really necessary?)
}

bool check_score(
		Ligand_info const ligand,
		core::Size const heavy_atom_number
){
	int const atr_threshold=0;
	int const rep_threshold=-(int) (0.85 * heavy_atom_number);
	return ligand.atr <= atr_threshold && ligand.rep <= rep_threshold;
}

bool check_RMSD(
		Ligand_info const ligand,
		core::Size const heavy_atom_number,
		utility::vector1< Ligand_info> const & ligands
){
	assert(heavy_atom_number > 0);

	// This next parameter is a wild heuristic guesses that seem OK for the Meiler x-dock set.
	core::Real const diverse_rms = 0.65 * std::sqrt((double) heavy_atom_number);

	utility::vector1< Ligand_info >::const_iterator iter= ligands.begin();
	utility::vector1< Ligand_info >::const_iterator const end= ligands.end();

	for(; iter+1 != end; ++iter){
		core::pose::Pose const & this_pose= ligand.get_pose();
		core::pose::Pose const & compare_pose= iter->get_pose();

		core::Real const rms = core::scoring::rmsd_no_super(this_pose, compare_pose, core::scoring::is_ligand_heavyatom);

		if (rms < diverse_rms) return false;
	}
	return true;
}

} //namespace ligand_options
} //namespace ligand_docking
} //namespace protocols
