// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   core/pack/task/ResfileReader.hh
/// @brief  header of classes for resfile options
/// @author Gordon Lemmon

#ifndef INCLUDED_protocols_ligand_docking_ligand_options_Rotate_HH
#define INCLUDED_protocols_ligand_docking_ligand_options_Rotate_HH

// Unit Headers
#include <protocols/ligand_docking/ligand_options/BaseOptions.hh>
#include <protocols/ligand_docking/ligand_options/LigandOptionMap.fwd.hh>
#include <protocols/moves/RigidBodyMover.hh>

//// Project Headers
#include <core/scoring/rms_util.hh>
#include <core/grid/CartGrid.hh>

///////////////////////////////////////////////////////////////////////

namespace protocols {
namespace ligand_docking {
namespace ligand_options {

static core::util::Tracer rotate_tracer("protocols.ligand_docking.ligand_options.rotate", core::util::t_debug);

struct Rotate_info{ // including default values
	Distribution distribution;
	core::Size degrees;
	core::Size cycles;
	Rotate_info(): distribution(uniform), degrees(0), cycles(0){};
};

struct Ligand_info{
	core::pose::Pose pose;
	int atr;
	int rep;
	core::kinematics::Jump jump;
	Ligand_info():pose(core::pose::Pose()), atr(0), rep(0), jump(){}
	Ligand_info(core::pose::Pose const pose, int atr, int rep):pose(pose), atr(atr), rep(rep), jump(jump){}
	Ligand_info(core::pose::Pose const pose, std::pair<int,int> scores, core::kinematics::Jump jump):pose(pose), atr(scores.first), rep(scores.second), jump(jump){}
	bool operator<(Ligand_info const ligand_info) const{
		return ( rep < ligand_info.rep || (rep == ligand_info.rep && atr < ligand_info.atr ) );
	}
	bool operator<(std::pair<int,int> const scores) const{
		return rep < scores.second || (rep == scores.second && atr < scores.first);
	}
	core::pose::Pose const & get_pose() const{
		return pose;
	}
};

class Rotate : public InitialMoverOption
{
public:
	Rotate(core::pose::Pose & pose);

	void option(
		utility::vector1< std::string > const & tokens,
		std::set<core::Size> const & ligands_to_dock
	);

	void option(
		utility::vector1< std::string > const & tokens,
		core::Size const & ligand
	);
	void apply();

	ligand_options::option type()const {return rotate;}

private:
	// this is a map<chain_id, num_of_cycles>
	std::map<core::Size, Rotate_info> chains_;

	void option(
		core::Size const & num_cycles,
		std::set<core::Size> const & ligands_to_dock
	);

	Rotate_info
	parse_tokens(
			utility::vector1< std::string> const & tokens
	) const;

	void option(
			Rotate_info const rotate_info,
			std::set<core::Size> const & ligands_to_dock
	);

	void apply(
			core::Size chain_id,
			Rotate_info const rotate_info
	);

	void rotate_ligand(
			core::grid::CartGridOP const & grid,
			core::Size const jump_id,
			core::Size const chain_id,
			Rotate_info const rotate_info
	);

///@brief  These should have repulsive and attractive scores under the threshold
utility::vector1< Ligand_info> create_random_rotations(
		core::grid::CartGridOP const & grid,
		core::Size const cycles,
		protocols::moves::RigidBodyMoverOP const mover,
		core::Size const jump_id,
		core::Size const chain_id,
		core::Size const begin
)const;

}; // class Rotate

/// Convenience Functions for use with Rotate

bool check_score(
		Ligand_info const ligand,
		core::Size const heavy_atom_number
);

bool check_RMSD(
		Ligand_info const ligand,
		core::Size const heavy_atom_number,
		utility::vector1< Ligand_info> const & ligands
);

void apply_rotate(
		protocols::moves::RigidBodyMoverOP mover,
		core::pose::Pose & pose,
		core::Vector const & center
);

void add_ligand_conditionally(
		Ligand_info const & ligand_info,
		utility::vector1< Ligand_info> & ligands,
		core::Size const heavy_atom_number
);

} //namespace ligand_options
} //namespace ligand_docking
} //namespace protocols

#endif
