// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   core/pack/task/ResfileReader.cc
/// @brief  implementation of resfile reader and its command classes
/// @author Gordon Lemmon (glemmon@gmail.com), adapted from the ResfileReader code
/// by Steven Lewis (smlewi@unc.edu) and Andrew Leaver-Fay

// Unit Headers
#include <protocols/ligand_docking/ligand_options/Soft_rep.hh>
#include <protocols/ligand_docking/ligand_options/InterfaceBuilder.hh>

#include <core/scoring/ScoreFunction.hh>

#include <core/io/database/open.hh>

#include <core/scoring/methods/EnergyMethodOptions.hh>

// Utility Headers
#include <utility/exit.hh>
#include <utility/assert.hh>
#include <utility/string_util.hh>
#include <core/util/Tracer.hh>
#include <core/types.hh>

// option key includes

#include <core/options/keys/enzdes.OptionKeys.gen.hh>

using core::util::T;
using core::util::Error;
using core::util::Warning;

namespace protocols {
namespace ligand_docking {
namespace ligand_options {


Soft_rep::Soft_rep(core::pose::Pose & pose) :
		LigandOptionCommand(pose),
		DefaultCommand(pose),
		soft_(false),
		old_estat_(false),
		score_fxn_(new core::scoring::ScoreFunction())
{
	score_fxn_->reset();
}

void Soft_rep::option(
		utility::vector1< std::string> const & tokens,
		std::set<core::Size> const & /*ligands_to_dock*/
) {
	if (tokens.size() < 1) {
		utility_exit_with_message("this should be impossible since tokens[1] got us here");
	}
	soft_ = true;
	std::string const s = tokens[2];
	if (s.compare("new") == 0){
		old_estat_ = false;
		Soft_rep_tracer << "Using new electrostatics" << std::endl;
	}
	else if (s.compare("old") == 0){
		old_estat_ = true;
		Soft_rep_tracer << "Using old electrostatics" << std::endl;
	}
	else utility_exit_with_message("'soft_rep' expects 'old' or 'new', not "+s);
}

void Soft_rep::apply() {
	if(soft_) tweak_scorefxn("ligand_soft_rep", old_estat_);
	else tweak_scorefxn("ligand", old_estat_);
}

// regardless of soft or hard option, restore to hard
void Soft_rep::restore_to_hard_rep() {
	score_fxn_->reset();
	tweak_scorefxn("ligand", old_estat_);
}

void Soft_rep::tweak_scorefxn(
		std::string const & weights_tag,
		bool const old_estat,
		bool const hbonds_downweight
) {

	{// manipulate EnergyMethodOptions here
		bool const estat_exclude_protein= old_estat;
		core::scoring::methods::EnergyMethodOptions options( score_fxn_->energy_method_options());
		options.exclude_protein_protein_hack_elec(estat_exclude_protein);
		score_fxn_->set_energy_method_options(options);
		// Done manipulating
	}
	// over-wrights previously set score weights
	score_fxn_->add_weights_from_file(core::io::database::full_name("scoring/weights/" + weights_tag + ".wts"));

	add_score_terms_conditionally(old_estat, hbonds_downweight);
}

void Soft_rep::add_score_terms_conditionally(bool old_estat, bool hbonds_downweight){
	using namespace core::scoring;

	// Tiny weight here (like standard.wts) presumably eliminates the worst intra-ligand clashes...
	// Weight increased because I was still getting significant overlaps between ligand atoms,
	// enough to get new "bonds" in PyMol (though no more interpenetrating rings even at 0.004).
	// However, using 0.04 meant total fa_intra_rep ~ 20 in most structures, which I'm concerned
	// biased the selection of ligand conformers too much.  Needs more rigorous testing.
	if( score_fxn_->has_zero_weight( fa_intra_rep ) ) score_fxn_->set_weight( fa_intra_rep, 0.004 ); // from standard.wts

	// For some reason, electrostatics is not in the .wts files...
	// hack_elec has a different dielectric constant than Rosetta++ (10r vs. 6r in ++)
	// It also includes all atom pairs instead of only ligand-protein interactions.
	if( score_fxn_->has_zero_weight( hack_elec ) ) score_fxn_->set_weight( hack_elec, 0.25 ); // from Meiler & Baker 2006

	if( old_estat ) score_fxn_->set_weight( hack_elec, (10./6.) * score_fxn_->get_weight( hack_elec ) ); // make like Rosetta++

	if( hbonds_downweight ) {
		score_fxn_->set_weight( hbond_sc, 1.30 ); // from Lin Jiang
		score_fxn_->set_weight( hbond_bb_sc, 1.30 ); // from Lin Jiang
	}
	if( score_fxn_->has_zero_weight( coordinate_constraint ) ) score_fxn_->set_weight( coordinate_constraint, 1.0 );
	if( score_fxn_->has_zero_weight( atom_pair_constraint ) )  score_fxn_->set_weight( atom_pair_constraint, 1.0 );
	if( score_fxn_->has_zero_weight( angle_constraint ) )      score_fxn_->set_weight( angle_constraint, 1.0 );
	if( score_fxn_->has_zero_weight( dihedral_constraint ) )   score_fxn_->set_weight( dihedral_constraint, 1.0 );
	if( score_fxn_->has_zero_weight( chainbreak ) )            score_fxn_->set_weight( chainbreak, 1.0 );
	if( score_fxn_->has_zero_weight( omega ) )                 score_fxn_->set_weight( omega, 0.5 ); // from score12.wts_patch
	if( score_fxn_->has_zero_weight( rama ) )                  score_fxn_->set_weight( rama, 0.2 ); // from score12.wts_patch
}

const core::scoring::ScoreFunctionOP & Soft_rep::get_score_fxn() const {
	return score_fxn_;
}


} //namespace ligand_options
} //namespace ligand_docking
} //namespace protocols
