// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   core/pack/task/ResfileReader.cc
/// @brief  implementation of resfile reader and its command classes
/// @author Gordon Lemmon (glemmon@gmail.com)
///

// Unit Headers
#include <protocols/ligand_docking/ligand_options/Start_from.hh>
#include <protocols/ligand_docking/ligand_options/InterfaceBuilder.hh>

#include <protocols/geometry/RB_geometry.hh>
#include <protocols/moves/RigidBodyMover.hh>

// Utility Headers
#include <numeric/random/random.hh>
#include <utility/io/izstream.hh>
#include <utility/exit.hh>
#include <utility/assert.hh> //ASSERT_ONLY makes release build happy
#include <utility/string_util.hh>
#include <core/util/Tracer.hh>
#include <core/types.hh>
using core::util::T;
using core::util::Error;
using core::util::Warning;

namespace protocols {
namespace ligand_docking {
namespace ligand_options {

Start_from::Start_from(core::pose::Pose & pose) :
	LigandOptionCommand(pose), LigandSpecificCommand(pose) {
}

void Start_from::option(
		utility::vector1< std::string> const & tokens,
		core::Size const & ligand_chain
) {
	if ((tokens.size() - 1) < 1) {
		utility_exit_with_message("start_from requires one or more X,Y,Z triples!");
	}
	utility::vector1< core::Vector> start_from;
	for (Size ii = 2; ii <= tokens.size(); ++ii) {
		start_from.push_back(split(tokens[ii]));
	}
	chains_[ligand_chain] = start_from;
}

const core::Vector Start_from::split(std::string const s, char const token)const {
	utility::vector1< double > coords;

	core::Size start = 0;
	core::Size stop=0;
	while(stop != std::string::npos){
		stop = s.find(token, start);
		std::string double_string = s.substr(start, stop - start);
		coords.push_back( strtod(double_string.c_str(), NULL) );
		start = stop+1;
	}

	if ( coords.size() != 3) {
		utility_exit_with_message("start_from requires X,Y,Z triples");
	}
	start_from_tracer.Debug<< "coords: "<< coords[1] <<" , "<< coords[2] <<" , "<< coords[3] << std::endl;

	return core::Vector(coords[1], coords[2], coords[3]);// passing a pointer as shown utilizes a constructor that iterates over items 0-2
}

void Start_from::apply() {
	std::map<core::Size, utility::vector1< core::Vector> >::iterator i = chains_.begin();///TODO make const_iterator
	for (; i != chains_.end(); ++i) {
		core::Size jump_id = get_jump_id_from_chain_id(i->first, pose_);
		core::Vector desired_centroid = choose_desired_centroid(jump_id, i->second);
		move_ligand_to_desired_centroid(jump_id, desired_centroid);
		start_from_[jump_id] = desired_centroid;
	}
}

core::Vector Start_from::choose_desired_centroid(
		core::Size const jump_id,
		utility::vector1< core::Vector> const start_from_pts
) const {
	// Choose desired centroid:  either -start_from or the current position.
	core::Vector desired_centroid;
	if (!start_from_pts.empty()) {
		int const which_triple = numeric::random::RG.random_range(1, start_from_pts.size());
		desired_centroid = start_from_pts[which_triple];
	} else {
		desired_centroid = protocols::geometry::downstream_centroid_by_jump(pose_, jump_id);
	}
	return desired_centroid;
}

void Start_from::move_ligand_to_desired_centroid(
		core::Size const jump_id,
		core::Vector const desired_centroid
) const {
	core::Vector const ligand_centroid = protocols::geometry::downstream_centroid_by_jump(pose_, jump_id);
	core::Vector const trans_vec = desired_centroid - ligand_centroid;
	core::Real const trans_len = trans_vec.length();
	if (trans_len > 1e-3) { // otherwise we get NaNs
		protocols::moves::RigidBodyTransMover mover(pose_, jump_id);
		mover.step_size(trans_len);
		mover.trans_axis(trans_vec);
		mover.apply(pose_);
	}
}

} //namespace ligand_options
} //namespace ligand_docking
} //namespace protocols
