// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   core/pack/task/ResfileReader.cc
/// @brief  implementation of resfile reader and its command classes
/// @author Gordon Lemmon (glemmon@gmail.com), adapted from the ResfileReader code
/// by Steven Lewis (smlewi@unc.edu) and Andrew Leaver-Fay

// Unit Headers
#include <protocols/ligand_docking/ligand_options/Tether_ligand.hh>
#include <protocols/ligand_docking/ligand_options/InterfaceBuilder.hh>


#include <core/scoring/constraints/HarmonicFunc.hh>
#include <core/scoring/constraints/CoordinateConstraint.hh>

// Utility Headers
#include <numeric/random/random.hh>
#include <utility/io/izstream.hh>
#include <utility/exit.hh>
#include <utility/assert.hh> //ASSERT_ONLY makes release build happy
#include <utility/string_util.hh>
#include <core/util/Tracer.hh>
#include <core/types.hh>
using core::util::T;
using core::util::Error;
using core::util::Warning;

namespace protocols {
namespace ligand_docking {
namespace ligand_options {

static core::util::Tracer tether_ligand_tracer("protocols.ligand_docking.ligand_options.Tether_ligand", core::util::t_debug);

Tether_ligand::Tether_ligand(
		core::pose::Pose & pose
) :
		LigandOptionCommand(pose),
		LigandSpecificCommand(pose),
		DefaultCommand(pose)
{}

void Tether_ligand::option(
		utility::vector1< std::string> const & tokens,
		core::Size const & ligand_chain_id
) {
	if (tokens.size() != 2) {
		utility_exit_with_message("'tether_ligand' takes one argument (Real)");
	}
	core::Real r = std::strtod(tokens[2].c_str(), NULL);
	//core::Size jump_id= get_jump_id_from_chain_id(ligand_chain_id);

	chains_[ligand_chain_id] = r;

}

void Tether_ligand::option(
		utility::vector1< std::string> const & tokens,
		std::set<core::Size> const & ligands_to_dock
) {
	if (tokens.size() != 2) {
		utility_exit_with_message("'tether_ligand' takes one argument (Real)");
	}
	core::Real r = std::strtod(tokens[2].c_str(), NULL);

//#ifndef WIN32
// Now I am using a const_iterator so I think it is okay
// visual studio c++ is far less forgiving about the code wrapped in the #ifdef
// below. It fails because it can't get an iterator from a const std::set<Size>.
	std::set<core::Size>::const_iterator i = ligands_to_dock.begin();
	for (; i != ligands_to_dock.end(); ++i) {
		chains_[*i] = r;
	}
//#endif
}

void Tether_ligand::apply() {
	std::map<core::Size, core::Real>::iterator i = chains_.begin();
	for (; i != chains_.end(); ++i) {
		core::Size const chain = i->first;
		core::Size const residue_id = pose_.conformation().chain_begin(chain);
		///TODO find the centroid positioned residue rather than just taking the first
		ligand_tethers_.insert(restrain_ligand_nbr_atom(residue_id, i->second));
	}
}

core::scoring::constraints::ConstraintCOP Tether_ligand::restrain_ligand_nbr_atom(
		core::Size const lig_id,
		core::Real const stddev_Angstroms
) const {
	tether_ligand_tracer.Debug<< "stddev: " << stddev_Angstroms << std::endl;
	core::scoring::constraints::FuncOP const restraint_function = new core::scoring::constraints::HarmonicFunc(0, stddev_Angstroms);

	core::id::AtomID const fixed_pt(pose_.atom_tree().root()->atom_id());

	core::conformation::Residue const & residue = pose_.residue(lig_id);
	core::scoring::constraints::ConstraintCOP constraint = new core::scoring::constraints::CoordinateConstraint(
			core::id::AtomID( residue.nbr_atom(), lig_id),
			fixed_pt,
			residue.nbr_atom_xyz(),
			restraint_function
	);
	constraint = pose_.add_constraint(constraint);

	return constraint;
}

void Tether_ligand::release() {
	///TODO make this a const_iterator
	std::set<core::scoring::constraints::ConstraintCOP>::iterator i = ligand_tethers_.begin();
	for (; i != ligand_tethers_.end(); ++i) {
		pose_.remove_constraint(*i);
	}
}

const std::set<core::scoring::constraints::ConstraintCOP> &
Tether_ligand::get_const_ligand_tethers_reference() const {
	return ligand_tethers_;
}


} //namespace ligand_options
} //namespace ligand_docking
} //namespace protocols
