// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   core/pack/task/ResfileReader.cc
/// @brief  implementation of resfile reader and its command classes
/// @author Gordon Lemmon (glemmon@gmail.com)

// Unit Headers
#include <protocols/ligand_docking/ligand_options/Translate.hh>
#include <protocols/ligand_docking/ligand_options/InterfaceBuilder.hh>

#include <protocols/ligand_docking/grid_functions.hh>
#include <protocols/geometry/RB_geometry.hh>
//#include <protocols/moves/Mover.hh>
#include <protocols/moves/RigidBodyMover.hh>
#include <core/scoring/rms_util.hh>

// Utility Headers
#include <numeric/random/random.hh>
#include <utility/io/izstream.hh>
#include <utility/exit.hh>
#include <utility/assert.hh> //ASSERT_ONLY makes release build happy
#include <utility/string_util.hh>
#include <core/util/Tracer.hh>
#include <core/types.hh>
using core::util::T;
using core::util::Error;
using core::util::Warning;

namespace protocols {
namespace ligand_docking {
namespace ligand_options {

///@brief
Translate::Translate(core::pose::Pose & pose) :
	LigandOptionCommand(pose), DefaultCommand(pose), LigandSpecificCommand(pose), InitialMoverOption(pose)
{}

Translate_info
Translate::parse_tokens(
		utility::vector1< std::string> const & tokens
){
	if (tokens.size() < 1 ) {
		utility_exit_with_message("This should be impossible");
	}
	if (tokens.size() > 4 ) {
			utility_exit_with_message("'Translate' has too many arguments");
	}

	Translate_info translate_info;
	if (tokens.size() > 1){
		std::string distribution= tokens[2].c_str();
		std::map< std::string, Distribution >::const_iterator const distribution_pair= distribution_map_.find(distribution);
		translate_info.distribution= distribution_pair->second;

		if (tokens.size() == 2){
			translate_info.angstroms= 5;
			translate_info.cycles= 50;
		}
	}
	if (tokens.size() > 2){
		translate_info.angstroms= atof(tokens[3].c_str());
		if (tokens.size() == 3){
			translate_info.cycles= 50;
		}
	}
	if (tokens.size() > 3){
		translate_info.cycles= atoi(tokens[4].c_str());
		translate_tracer.Debug<< "set "<< translate_info.cycles<<" cycles"<< std::endl;
	}
	return translate_info;
}

void Translate::option(
		utility::vector1< std::string> const & tokens,
		std::set<core::Size> const & ligands_to_dock
) {
	Translate_info translate_info= parse_tokens(tokens);
	option(translate_info, ligands_to_dock);
}

void Translate::option(
		Translate_info translate_info,
		std::set<core::Size> const & ligands_to_dock
) {
	std::set<core::Size>::const_iterator i = ligands_to_dock.begin();
	for (; i != ligands_to_dock.end(); ++i) {
		chains_[*i] = translate_info;
	}
}

void Translate::option(
		utility::vector1< std::string> const & tokens,
		core::Size const & ligand_chain
) {
	chains_[ligand_chain] = parse_tokens(tokens);
}

void Translate::apply() {
	std::map<core::Size, Translate_info>::iterator i = chains_.begin();
	///TODO translate in a random order
	utility::vector1<core::Size> chains_to_exclude;
	for (; i != chains_.end(); ++i) {
		chains_to_exclude.push_back(i->first);
	}
	while(! chains_to_exclude.empty() ){
		core::Size const chain_to_translate= chains_to_exclude.back();
		chains_to_exclude.pop_back();
		apply(chain_to_translate, chains_[chain_to_translate], chains_to_exclude);
	}
}

void Translate::apply(
		core::Size chain_id,
		Translate_info const translate_info,
		utility::vector1<core::Size> chains_to_exclude
) {
	core::Size const jump_id = get_jump_id_from_chain_id(chain_id, pose_);
	core::Vector const center = protocols::geometry::downstream_centroid_by_jump(pose_, jump_id);

	core::grid::CartGridOP const grid = make_atr_rep_grid_without_ligands(pose_, center, chains_to_exclude);

	translate_ligand(grid, jump_id, translate_info);// move ligand to a random point in binding pocket

}

void Translate::translate_ligand(
		const core::grid::CartGridOP & grid,
		const core::Size jump_id,
		const Translate_info translate_info
) {
	if(translate_info.angstroms < 0) utility_exit_with_message("cannot have a negative translation value");
	if(translate_info.angstroms == 0) return;

	protocols::moves::RigidBodyMoverOP mover;
	if(translate_info.distribution == uniform){
		translate_tracer.Debug<< "making a uniform translator of up to "<< translate_info.angstroms<<" angstroms"<< std::endl;
		mover= new protocols::moves::UniformSphereTransMover( jump_id, translate_info.angstroms);
	}
	else if(translate_info.distribution == gaussian){
		translate_tracer.Debug<< "making a Gaussian translator of up to "<< translate_info.angstroms<<" angstroms";
		mover= new protocols::moves::RigidBodyPerturbMover ( jump_id, 0 /*rotation*/, translate_info.angstroms);
	}

	core::pose::Pose const orig_pose(pose_);

	translate_tracer.Debug << "time to cycle: " << translate_info.cycles << std::endl;
	for (Size cnt = 0; cnt < translate_info.cycles; ++cnt) {
		mover->apply(pose_);
		core::Vector c = protocols::geometry::downstream_centroid_by_jump(pose_, jump_id);
		// did our nbr_atom land in an empty space on the grid?
		// Don't want to insist the nbr_atom is in the attractive region, because it may not be!
		if (
				grid->is_in_grid(c.x(), c.y(), c.z())
				&& grid->getValue(c.x(), c.y(), c.z()) <= 0
		) {
			translate_tracer.Debug << "Accepting ligand position with nbr_atom at " << c << std::endl;
			return;
		}
		translate_tracer.Debug << "Rejecting ligand position with nbr_atom at " << c << std::endl;
		pose_ = orig_pose; // reset and try again
	}///TODO alert the user that we cannot find a placement for this ligand.  Do not proceed with docking this ligand.

	translate_tracer << "WARNING: cannot find placement for this ligand"<< std::endl;
}

} //namespace ligand_options
} //namespace ligand_docking
} //namespace protocols
