// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.
/*
 * chain_functions.cc
 *
 *  Created on: Nov 6, 2008
 *      Author: lemmongh
 */

// Unit headers
#include <protocols/ligand_docking/ligand_options/chain_functions.hh>

// Project headers
#include <core/conformation/Residue.hh>
#include <core/kinematics/FoldTree.hh>
#include <core/pose/Pose.hh>
#include <core/util/Tracer.hh>
#include <core/pose/PDBInfo.hh>
#include <core/conformation/Conformation.hh>

namespace protocols {
namespace ligand_docking {
namespace ligand_options {

std::set<core::Size>
get_jump_ids_from_chain_ids(const std::set<core::Size> chain_ids, const core::pose::Pose & pose){
	std::set<core::Size> jump_ids;
	std::set<core::Size>::const_iterator chain_id= chain_ids.begin();
	for(; chain_id != chain_ids.end(); ++chain_id){
		core::Size jump_id= get_jump_id_from_chain_id(*chain_id, pose);
		jump_ids.insert(jump_id);
	}
	return jump_ids;
}

core::Size
get_jump_id_from_chain_id(core::Size const & chain_id,const core::pose::Pose & pose){
	for(core::Size jump_id=1; jump_id <= pose.num_jump(); jump_id++){
		core::Size ligand_residue_id= (core::Size) pose.fold_tree().downstream_jump_residue(jump_id);
		core::Size ligand_chain_id= pose.chain(ligand_residue_id);
		if(chain_id==ligand_chain_id){
			return jump_id;
		}
	}
	utility_exit();
	return 0;// this will never happen
}

core::Size
get_chain_id_from_chain(std::string const & chain, core::pose::Pose const & pose){
	assert(chain.size()==1);// chain is one char
	char chain_char= chain[0];
	for(core::Size i=1; i <= pose.conformation().num_chains(); i++){
		char this_char= get_chain_from_chain_id(i, pose);
		if(this_char == chain_char){
			return i;
		}
	}
	core::util::Error() << "the user-specified '"<< chain_char<<"' chain is not found" << std::endl;
	utility_exit();
	return 0;// this will never happen
}

core::Size get_chain_id_from_jump_id(core::Size const & jump_id, core::pose::Pose const & pose){
	core::Size ligand_residue_id= (core::Size) pose.fold_tree().downstream_jump_residue(jump_id);
	return pose.chain(ligand_residue_id);
}

char
get_chain_from_jump_id(core::Size const & jump_id, core::pose::Pose const & pose){
	core::Size chain_id= get_chain_id_from_jump_id(jump_id, pose);
	return get_chain_from_chain_id(chain_id, pose);
}


char
get_chain_from_chain_id(core::Size const & chain_id, core::pose::Pose const & pose){
	core::Size first_chain_residue= pose.conformation().chain_begin( chain_id );
	return pose.pdb_info()->chain(first_chain_residue);
}

core::Size num_heavy_atoms(
		core::Size begin,
		const core::Size end,
		const core::pose::Pose & pose
) {
	core::Size total_heavy_atoms = 0;
	for (; begin <= end; ++begin) {
		total_heavy_atoms += pose.residue(begin).nheavyatoms();
	}
	chain_functions_tracer.Debug << "# of heavy atoms: "<< total_heavy_atoms << std::endl;
	return total_heavy_atoms;
}

core::Size num_chi_angles(
	core::Size begin,
	core::Size const end,
	core::pose::Pose const & pose
) {
	core::Size total_chi_angles = 0;
	for (; begin <= end; ++begin) {
		total_chi_angles += pose.residue(begin).nchi();
	}
	return total_chi_angles;
}


}
}
}
