// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file
/// @brief
/// @author Mike Tyka, Chu Wang

#ifndef INCLUDED_protocols_loops_LoopClass_HH
#define INCLUDED_protocols_loops_LoopClass_HH


#include <core/types.hh>
#include <core/util/Tracer.fwd.hh>
#include <core/kinematics/MoveMap.fwd.hh>
#include <core/id/types.hh>
#include <core/pose/Pose.fwd.hh>
#include <utility/pointer/ReferenceCount.hh>
// C++ Headers
//#include <iosfwd>
#include <functional>

// Utility Headers
#include <utility/vector1.hh>

///////////////////////////////////////////////////////////////////////////////
namespace protocols {
namespace loops {

/// single loop definition
class Loop : public utility::pointer::ReferenceCount
{
public:
	/// default constructor
	Loop():
		start_(0),
		stop_(0),
		cut_(0),
		skip_rate_( 0.0 ),
		extended_(false)
	{}
	/// input constructor
	Loop(
			core::Size const start_in, core::Size const stop_in,
			core::Size const cut_in = 0, core::Real skip_rate = 0.0,
			bool const extended_in = false
	):
		start_( start_in ),
		stop_( stop_in ),
		cut_( cut_in ),
		skip_rate_( skip_rate ),
		extended_( extended_in )
	{}

	inline bool is_extended() const { return extended_; }
	inline core::Size start() const { return start_; }
	inline core::Size stop() const { return stop_; }
	inline core::Size cut() const { return cut_; }
	inline core::Size size() const { return stop_ - start_ + 1; }
	inline core::Real skip_rate() const { return skip_rate_; }

	inline void set_extended( bool input    ) { extended_ = input; }
	inline void set_start( core::Size input ) { start_ = input; }
	inline void set_stop( core::Size input  ) { stop_ = input; }
	inline void set_cut( core::Size input   ) { cut_ = input; }
	bool operator< ( Loop const& larger ) const {
		return ( size() < larger.size() ? true :
			( start() < larger.start() ? true :	( cut() < larger.cut() ) ) );
	}
	bool operator==( Loop const& other ) const {
		return ( size() == other.size() &&
			start() == other.start() && cut() == other.cut() );
	}
	bool operator!=( Loop const& other ) const {
		return !(*this == other );
	}

	///@brief add all residues within this loop definition into selection
	void get_residues( utility::vector1< Size>& selection ) const;


	/// @brief switch DOF_Type for residues in loop. id::CHI, id::BB --- don't
	/// use with id::JUMP
	void switch_movemap( core::kinematics::MoveMap& movemap, core::id::TorsionType, bool allow_moves = true ) const;

	// @brief Autochoose a cutpoint using the secondary structure of the pose.
	void choose_cutpoint( core::pose::Pose const & pose );

	/// @brief Autochoose a cutpoint using the secondary structure of the pose
	/// unless cutpoint is already set
	void auto_choose_cutpoint( core::pose::Pose const & pose ){
		if ( cut_ <= 0) choose_cutpoint ( pose );
	}

	bool is_terminal( core::pose::Pose const & pose ) const;

	friend std::ostream & operator<<( std::ostream & os, const Loop & loop );

private:
	core::Size start_;
	core::Size stop_;
	core::Size cut_;
	core::Real skip_rate_;
	bool extended_;
};


//////////////////////////////////////////////////////////////////////
inline std::ostream & operator<<( std::ostream & os, const Loop & loop ) {
	// it's super-annoying that this returns a std::endl
	os << "LOOP " << loop.start_ << " " << loop.stop_ << " "
		<< loop.cut_ << " "
		//<< loop.skip_rate_ << " " << loop.extended_ << std::endl;
		<< loop.skip_rate_ << " " << loop.extended_;
	return os;
}



/// @brief used to sort Loops by start-res
class Loop_lt : public std::binary_function<double, double, bool> {
public:
	bool operator()(Loop x, Loop y) {
		return (x.start() < y.stop());
	}
};


///////////////////////////////////////////////////////////////////////////
// a list of loops
class Loops : public utility::pointer::ReferenceCount
{

public:
	typedef utility::vector1< Loop > LoopList;
	typedef LoopList::iterator iterator;
	typedef LoopList::const_iterator const_iterator;


public:
	inline core::Size num_loop() const { return loops_.size(); }
	inline const_iterator begin() const { return loops_.begin(); }
	inline const_iterator end() const { return loops_.end(); }
	inline iterator v_begin() { return loops_.begin(); }
	inline iterator v_end() { return loops_.end(); }

	//constructor
	Loops(){};
	//copy constructor
	Loops( const Loops & src ):
		utility::pointer::ReferenceCount(),
		loops_(src.loops_) {}
	//operator
	Loops & operator =( Loops const & src ) {
		loops_ = src.loops_;
		return *this;
	}

	friend std::ostream & operator<<( std::ostream & os, const Loops & loops );
	// friend core::util::VTracer & operator<<( core::util::VTracer & os, const Loops & loops );

	void read_loop_file(
		std::string filename,
		bool strict_looprelax_checks = true,
		std::string token = "LOOP"
	);
	void read_stream_to_END(
    std::istream &is,
		bool strict_looprelax_checks,
		std::string filename, /*for error msg */
		std::string token = "LOOP"
	);
	void
	write_loops_to_file(
		std::string const & filename,
		std::string token = "LOOP"
	) const;

	void
	add_loop( Loop loop, core::Size minimal_gap = 0 );

	void
	add_loop(
		core::Size const start,
		core::Size const stop,
		core::Size const cut = 0,
		core::Real skip_rate = 0.0,
		bool const extended = false
	);

	void
	add_loop(
		const Loops::const_iterator & it
	);

	void
	add_loop(
		const Loops::iterator & it
	);

	void push_back( Loop loop ) {
		add_loop( loop );
	}

	void
	add_overlap_loop(
		Loops loops
	);

	void
	add_overlap_loop(
		const Loop loop
	);

	void
	delete_loop(
		core::Size const start,
		core::Size const stop
	);

	const_iterator one_random_loop() const;

	core::Size
	loop_size(
		core::Size const loop_num
	) const;

	///@brief return number of residues in all loops of this definition -- sum_i( loop_size( i ) )
	core::Size
	loop_size() const;

	core::Size size() const {
		return loops_.size();
	}

	core::Size nr_residues() const {
		return loop_size();
	}

	void sequential_order();

	void clear();

	LoopList const& loops() const { return loops_; }

	/// @brief  Is seqpos contained in any of my loops?
	bool
	is_loop_residue( core::Size const seqpos, int const offset = 0 ) const;


	void set_extended( bool input );

  void make_sequence_shift( int shift );


	/// @brief yield the Loop which contains the residue seqpos, returns false if seqpos is not in any residue.
	bool loop_of_residue( core::Size const seqpos, Loop& loop ) const;

	///@brief yields the regions that are NOT loops in this loop-object.
	Loops invert( core::Size total_res ) const;

	//@brief switch DOF_Type for residues in loop. id::CHI, id::BB --- don't use with id::JUMP
	void switch_movemap( core::kinematics::MoveMap& movemap, core::id::TorsionType, bool allow_moves = true ) const;

	//@brief return index in list of the loop, 0 if not found
	core::Size loop_index_of_residue( core::Size const seqpos ) const;

	//@brief Autochoose a cutpoint using the secondary structure of the pose unless cutpoint is already set
	void auto_choose_cutpoints( core::pose::Pose const & pose );

	//@brief Autochoose a cutpoint using the secondary structure of the pose unless cutpoint is already set
	void choose_cutpoints( core::pose::Pose const & pose );

	void verify_against(	core::pose::Pose const & pose ) const;

	void remove_terminal_loops( core::pose::Pose const & pose );

	/// @brief Extend a loop
	void grow_all_loops(
		core::Size nres,
		core::Real magnitude
	);

	/// @brief Extend a loop
	void grow_loop(
		core::Size nres,
		Loop & loop,
		core::Real magnitude
	);

	/// @brief Extend a loop unequally in both dirs 
	void grow_loop( 
		core::Size nres, 
		Loop & loop, 
		core::Real mag_left, 
		core::Real mag_right 
	); 

	///@brief set each loop-residue in the vector to val.
	/// input vector of nres length ( if shorter last residues of loop are ignored )
	template< class T >
	void transfer_to_residue_vector( utility::vector1< T >&, T val );

	///@brief add all residues within this loop definition into selection
	void get_residues( utility::vector1< Size>& selection ) const;
	// i know this encourages old style for-loops (i.e. without iterators) but so much of the code
	// already used such loops, i opted for safety.
	inline
	const Loop&
	operator[] ( core::Size const i ) const
	{
		return loops_[i];
	}

	inline
	Loop&
	operator[] ( core::Size const i )
	{
		return loops_[i];
	}

	bool operator==( Loops const& other ) const;

	bool operator!=( Loops const& other ) const {
		return !( *this == other);
	}

private:
	LoopList loops_;
};

std::string get_loop_file_name();

Loops get_loops_from_file();


template< class T >
void Loops::transfer_to_residue_vector( utility::vector1< T > & vector, T val ) {
	core::Size nres = vector.size();
	for ( const_iterator it = begin(); it  != end(); ++it ) {
		if ( it->start() <= nres ) {
			for ( core::Size pos = it->start(); pos <= std::min( it->stop(), nres ); pos++ ) {
				vector[ pos ] = val;
			}
		}
	}
}

} //namespace loops
} //namespace protocols

#endif //INCLUDED_protocols_loops_LoopClass_HH
