// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file
/// @brief
/// @author Mike Tyka

#ifndef INCLUDED_protocols_loops_LoopMover_HH
#define INCLUDED_protocols_loops_LoopMover_HH


#include <core/types.hh>
#include <core/kinematics/MoveMap.hh>
#include <core/scoring/ScoreFunction.hh>  // Changing from .fwd.hh --> .hh to make python bindings compile
#include <core/pose/Pose.fwd.hh>

#include <protocols/moves/Mover.hh>
#include <protocols/loops/LoopClass.hh>

#include <core/fragment/FragSet.hh>
#include <core/fragment/FragSet.fwd.hh>

#include <protocols/checkpoint/CheckPointer.hh>

// C++ Headers


///////////////////////////////////////////////////////////////////////////////
namespace protocols {
namespace loops {


enum LoopResult { Success, CriticalFailure, Failure, ExtendFailure };


/// @brief The loop-rebuild protocol
class LoopMover: public protocols::moves::Mover {
public: // typedefs

	typedef core::kinematics::MoveMap MoveMap;

public:

	LoopMover(
		protocols::loops::Loops loops_in
	) : Mover(),
		loops_( loops_in ),
		checkpoints_("LoopMover")
	{
		Mover::type("LoopMover");
	}

	void set_scorefxn( const core::scoring::ScoreFunctionOP score_in ) {
		scorefxn_ = score_in;
	}

	/// @brief Apply the loop-build protocol to the input pose
	void apply( core::pose::Pose & ) {}

	/// @brief Extend a loop
	void grow_loop( core::pose::Pose & pose, Loop &loop, core::Real magnitude );

	/// @brief Extend a loop
	void grow_all_loops( core::pose::Pose & pose, core::Real magnitude ) {
		for ( core::Size i=1; i <= loops_.size(); i++ ) {
			grow_loop( pose, loops_[i], magnitude);
		}
	}

	const protocols::loops::Loops & loops() const {
		return loops_;
	}

	/// @brief Extend a loop
	virtual void set_extended_torsions(
		core::pose::Pose & pose,
		Loop const & loop
	);

public: // fragment libraries

	/// @brief add a fragment set
	void add_fragments( core::fragment::FragSetOP fragset );

	/// @brief clear the list of fragment sets
	void clear_fragments();

public: // movemap management

	/// @brief <b>explicit</b> False settings in this MoveMap will override any
	///  automatically generated MoveMap settings during the loop modeling
	///  protocol
	inline
	MoveMap const & false_movemap() const {
		return false_movemap_;
	}

	/// @brief <b>explicit</b> False settings in this MoveMap will override any
	///  automatically generated MoveMap settings during the loop modeling
	///  protocol
	inline
	void false_movemap( MoveMap const & mm ) {
		false_movemap_ = mm;
	}

public: // checkpointing

	checkpoint::CheckPointer & get_checkpoints() {
		return checkpoints_;
	}

protected: // movemap management

	/// @brief import the false_movemap's <b>explicit</b> False settings into the
	///  given MoveMap
	/// @return The number of False settings imported.
	inline
	Size enforce_false_movemap( MoveMap & mm ) const {
		return mm.import_false( false_movemap_ );
	}

protected: // data

	protocols::loops::Loops loops_;
	core::scoring::ScoreFunctionOP scorefxn_;

	checkpoint::CheckPointer checkpoints_;

	// it's really dumb that this is a std::vector rather a vector1
	std::vector< core::fragment::FragSetOP > frag_libs_;

private: // data

	/// @brief <b>explicit</b> False settings in this MoveMap will override any
	///  automatically generated MoveMap settings during the loop modeling
	///  protocol
	MoveMap false_movemap_;

}; // class LoopMover


// A subclass where all the loops are modelled independently from each
// other.  Deriving classes should not overload apply but instead
// model_loop. This is an unfortunate design decision, because there's no
// way to guarantee that deriving classes do the right thing and don't
// override the apply method.
class IndependentLoopMover: public LoopMover {
public:

	IndependentLoopMover(
		protocols::loops::Loops loops_in
	) : LoopMover( loops_in )
	{
		Mover::type("IndependentLoopMover");
		set_defaults();
	}

	void set_defaults();

	/// @brief Apply the loop-build protocol to the input pose
	virtual void apply( core::pose::Pose & pose );

	virtual LoopResult model_loop(
		core::pose::Pose & pose, protocols::loops::Loop const & loop
	) = 0;

	/// Accessors:

  void set_build_attempts_( int value ) { build_attempts_ =  value; }
  void set_grow_attempts_( int value )  { grow_attempts_ =  value; }
  void set_accept_aborted_loops_( bool value ) {   accept_aborted_loops_ =  value; }
  void set_strict_loops( bool value ) { strict_loops_ =  value; }
  void set_random_order_( bool value ) { random_order_ =  value; }
  void set_build_all_loops_( bool value ) {   build_all_loops_ =  value; }
  void set_loop_combine_rate_( core::Real value ) {   loop_combine_rate_ =  value; }

  int  get_build_attempts() const { return build_attempts_; }
  int  get_grow_attempts() const { return grow_attempts_; }
  bool get_accept_aborted_loops() const { return accept_aborted_loops_; }
  bool get_strict_loops() const { return strict_loops_; }
  bool get_random_order() const { return random_order_; }
  bool get_build_all_loops() const { return build_all_loops_; }
  bool get_loop_combine_rate() const { return loop_combine_rate_; }

private:

	/// select loops to be built
	void select_loops( Loops & selected_loops );

	/// Try loopbuilding n times before extending
	int build_attempts_;

	/// Try extending n times
	int grow_attempts_;

	/// danger - can lead to infinite loops !
	bool accept_aborted_loops_;

	/// Grow loops outwards if building fails.
	bool strict_loops_;

	/// Randomise loop build order
	bool random_order_;

	/// Force to build all loops (i.e. ignore skiprate)
	bool build_all_loops_;

	/// Loop combine rate
	core::Real loop_combine_rate_;

};

} //namespace loops
} //namespace protocols

#endif //INCLUDED_protocols_loops_LoopMover_HH
