// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file protocols/loops/LoopMover_Perturb_KIC.cc
/// @brief kinematic loop closure main protocols
/// @author Chu Wang
/// @author Daniel J. Mandell
/// @author Mike Tyka

//// Unit Headers
#include <protocols/antibody/AntibodyClass.hh>
#include <protocols/loops/loops_main.hh>
#include <protocols/loops/LoopMover.hh>
#include <protocols/loops/LoopMover_KIC.hh>
#include <protocols/loops/LoopClass.hh>
#include <protocols/moves/KinematicMover.hh>
#include <protocols/moves/MonteCarlo.hh>
#include <protocols/moves/MoverStatus.hh>

//
//// Rosetta Headers
#include <core/chemical/ChemicalManager.hh>
#include <core/chemical/VariantType.hh>
#include <core/chemical/util.hh>
#include <core/id/TorsionID.hh>
#include <core/kinematics/FoldTree.hh>
#include <core/kinematics/MoveMap.hh>
#include <core/optimization/AtomTreeMinimizer.hh>
#include <core/optimization/MinimizerOptions.hh>
#include <core/options/option.hh>
#include <core/pose/Pose.hh>
#include <core/pose/util.hh>
#include <core/scoring/Energies.hh>
#include <core/scoring/rms_util.hh>
#include <core/scoring/ScoreFunction.hh>
#include <core/scoring/ScoreFunctionFactory.hh>

#include <core/conformation/symmetry/util.hh>
#include <core/optimization/symmetry/SymAtomTreeMinimizer.hh>

#include <core/pack/task/TaskFactory.hh>
#include <core/pack/task/PackerTask.hh>
#include <core/pack/task/operation/TaskOperations.hh>
#include <core/pack/rotamer_trials.hh>
#include <core/pack/pack_rotamers.hh>

#include <core/util/prof.hh> // profiling
#include <core/util/Tracer.hh>

//Utility Headers
#include <numeric/random/random.hh>
#include <utility/io/izstream.hh>

// C++ Headers
#include <iostream>
#include <map>
#include <string>

// option key includes

#include <core/options/keys/out.OptionKeys.gen.hh>
#include <core/options/keys/loops.OptionKeys.gen.hh>
#include <core/options/keys/packing.OptionKeys.gen.hh>
#include <core/options/keys/antibody.OptionKeys.gen.hh>
#include <core/options/keys/run.OptionKeys.gen.hh>

namespace protocols {
namespace loops {

///////////////////////////////////////////////////////////////////////////////
using namespace core;

static numeric::random::RandomGenerator RG(42444);

extern core::util::Tracer tr;

LoopMover_Perturb_KIC::LoopMover_Perturb_KIC(
	protocols::loops::Loops  loops_in
) : IndependentLoopMover( loops_in )
{
	std::string function_tag("cen_std"), patch_tag("score4L");
	scorefxn_ = core::scoring::ScoreFunctionFactory::create_score_function(function_tag, patch_tag);
	scorefxn_->set_weight( core::scoring::chainbreak, 1.0*10.0/3.0);

	protocols::moves::Mover::type("LoopMover_Perturb_KIC");
	set_default_settings();

}

LoopMover_Perturb_KIC::LoopMover_Perturb_KIC(
	protocols::loops::Loops  loops_in,
	core::scoring::ScoreFunctionOP  scorefxn
) : IndependentLoopMover( loops_in )
{
	if( scorefxn ){
		scorefxn_ = scorefxn;
	}else{
		std::string function_tag("cen_std"), patch_tag("score4L");
		scorefxn_ = core::scoring::ScoreFunctionFactory::create_score_function(function_tag, patch_tag);
		scorefxn_->set_weight( core::scoring::chainbreak, 1.0*10.0/3.0);
	}
	protocols::moves::Mover::type("LoopMover_Perturb_KIC");
	set_default_settings();
}

void LoopMover_Perturb_KIC::set_default_settings()
{
	set_strict_loops(true); // obey loop definitions in kinematic mode
	max_seglen_ = core::options::option[core::options::OptionKeys::loops::kic_max_seglen];
	recover_low_ = ( ! core::options::option[core::options::OptionKeys::loops::kic_recover_last] );
	max_kic_build_attempts_ = core::options::option[core::options::OptionKeys::loops::max_kic_build_attempts];
	remodel_kic_attempts_ = core::options::option[core::options::OptionKeys::loops::remodel_kic_attempts];
}

void  LoopMover_Perturb_KIC::set_extended_torsions(
												   core::pose::Pose & ,
												   Loop const &
												   )
{
	// do nothing for now, overriding LoopMover::set_extended_torsions()
}

LoopResult LoopMover_Perturb_KIC::model_loop(
	core::pose::Pose & pose,
  protocols::loops::Loop const & loop
){
	static int cur_struct=0; // for movie output
	/*
	if ( loop.is_terminal( pose ) ) {
		tr.Error << "[WARNING] perturb_one_loop_with_KIC cannot handle terminal loops (it segfaults in KinematicMover). Doing nothing. " << std::endl;
		return CriticalFailure;
	}
	 */
	// Dont allow loops < 3 residues.
	if( (loop.stop() - loop.start() < 2 )){
		tr.Error << "[WARNING] KinematicMover cannot handle loops smaller than 3 residues. Doing nothing. " << std::endl;
		return CriticalFailure;
	}

	// store starting fold tree and cut pose
 	//kinematics::FoldTree f_orig=pose.fold_tree();
	//set_single_loop_fold_tree( pose, loop );
	//set_loop_cutpoint_in_pose_fold_tree( loop.cut(), pose, loop.start(), loop.stop() );
	//TR << "fold_tree_after_setting in remodel " << pose.fold_tree() << std::endl;

	//if ( remodel() == "perturb_KIC" ) {
	//	TR << "fold_tree_before_setting in remodel " << pose.fold_tree() << std::endl;
	//core::kinematics::FoldTree f_new, f_orig=pose.fold_tree();
	//protocols::loops::fold_tree_from_loops( pose, loops,  f_new );
	//pose.fold_tree( f_new );
	//	TR << "fold_tree_after_setting in remodel " << pose.fold_tree() << std::endl;
	//}

	// Objects representing one loop
	Loops one_loop_loops;
	one_loop_loops.add_loop( loop );

	using namespace scoring;
	using namespace optimization;
	using namespace options;

	//bool const verbose( true );
	bool const local_debug( false );
	bool const local_movie( false );

	core::pose::Pose native_pose;
	if( get_native_pose() ){
		native_pose = *get_native_pose();
	}else{
		native_pose = pose;
	}


	Size const loop_begin( loop.start() ), loop_end( loop.stop() ), loop_cut( loop.cut() );
	Size const loop_size( loop_end - loop_begin + 1 );
	runtime_assert( loop.is_terminal( pose ) || pose.fold_tree().is_cutpoint( loop_cut ) );
	std::ofstream loop_outfile; // for movie
	// see if we should skip large frags. Don't if that would mean no fragment insertions:

	tr << "perturb_one_loop_with_KIC: " << loop_begin << ' ' << loop_size << std::endl;

	// set cutpoint variant for chainbreak scoring.
	chemical::add_variant_type_to_pose_residue( pose, chemical::CUTPOINT_LOWER, loop_cut );
	chemical::add_variant_type_to_pose_residue( pose, chemical::CUTPOINT_UPPER, loop_cut+1 );

	if (local_debug) {
		std::ofstream out("score.tmp_input_cen");
		out << "scoring before cen_perturb: " << (*scorefxn_)(pose) << std::endl;
		/// Now handled automatically.  scorefxn_->accumulate_residue_total_energies(pose);
		scorefxn_->show( out );
		out << pose.energies().total_energies().weighted_string_of( scorefxn_->weights() ) << std::endl;
		tr << "before cen_perturb: "
			<< pose.energies().total_energies().weighted_string_of( scorefxn_->weights() ) << std::endl;
		out << pose.energies();
	}

	kinematics::MoveMap mm_one_loop;
	utility::vector1<bool> allow_sc_move( pose.total_residue(), false );
	loops_set_move_map( one_loop_loops, allow_sc_move, mm_one_loop);
	if ( core::conformation::symmetry::is_symmetric( pose ) )  {
		core::conformation::symmetry::make_symmetric_movemap( pose, mm_one_loop );
	}


	// scheduler
	bool const fast = option[OptionKeys::loops::fast];
	int outer_cycles( 3 );
	if ( option[ OptionKeys::loops::outer_cycles ].user() ) {
		outer_cycles = option[ OptionKeys::loops::outer_cycles ]();
	}
	if ( option[ OptionKeys::run::test_cycles ]() ) {
		outer_cycles = 3;
	}
	int inner_cycles( fast ? std::min( Size(250), loop_size*5 ) : std::min( Size(1000), loop_size*20 ) );
	if ( option[ OptionKeys::loops::max_inner_cycles ].user() ) {
		inner_cycles = option[ OptionKeys::loops::max_inner_cycles ]();
	}
	if ( option[ OptionKeys::run::test_cycles ]() ) {
		inner_cycles = 3;
	}

	// Monte Carlo vars
	float const init_temp( option[ OptionKeys::loops::remodel_init_temp ]() );
	float const	final_temp( option[ OptionKeys::loops::remodel_final_temp ]() );
	float const gamma = std::pow( (final_temp/init_temp), (1.0f/(outer_cycles*inner_cycles)) );
	float temperature = init_temp;
	if ( local_debug ) { // hacking
		(*scorefxn_)(pose);
		tr << "before mc ctor: "
			<< pose.energies().total_energies().weighted_string_of( scorefxn_->weights() ) << std::endl;
	}

	// minimizer
	AtomTreeMinimizerOP minimizer;
	float const dummy_tol( 0.001 ); // linmin sets tol internally
	bool const use_nblist( false ), deriv_check( false ); // true ); // false );
	MinimizerOptions options( "linmin", dummy_tol, use_nblist, deriv_check);
	if ( core::conformation::symmetry::is_symmetric( pose ) ) {
		minimizer = new core::optimization::symmetry::SymAtomTreeMinimizer;
	} else {
		minimizer = new core::optimization::AtomTreeMinimizer;
	}

	// show temps
	tr << "remodel init temp: " << init_temp << std::endl;
	tr << "remodel final temp: " << final_temp << std::endl;


	// perform the initial perturbation
	// setup the kinematic mover
	//protocols::moves::KinematicMover myKinematicMover( temperature );
	protocols::moves::KinematicMover myKinematicMover;
	protocols::moves::kinematic_closure::TorsionSamplingKinematicPerturberOP perturber =
		new protocols::moves::kinematic_closure::TorsionSamplingKinematicPerturber( &myKinematicMover );
	perturber->set_vary_ca_bond_angles( true );
	myKinematicMover.set_perturber( perturber );

	myKinematicMover.set_vary_bondangles( true );
	//myKinematicMover.set_vary_bondangles( false );  // trying without varying angles
	myKinematicMover.set_sample_nonpivot_torsions(
												  option[ OptionKeys::loops::nonpivot_torsion_sampling ]());
	myKinematicMover.set_rama_check( true );
	myKinematicMover.allow_soft_bump_if_allowed( true );
	// DJM: debug seeing if this is causing problems with the loopmodel extend loops
	Size kic_start, kic_middle, kic_end; // three pivot residues for kinematic loop closure
	kic_start = loop_begin;
	kic_end = loop_end;
	Size middle_offset = (kic_end - kic_start) / 2; // need to ensure this isn't a proline
	kic_middle = kic_start + middle_offset;
	tr << "kinematic initial perturb with start_res: "  << kic_start << "  middle res: " << kic_middle << "  end_res: "
	   << kic_end << std::endl;
	myKinematicMover.set_pivots(kic_start, kic_middle, kic_end);
	myKinematicMover.set_temperature(temperature);

	tr << "loop rmsd before initial kinematic perturbation:" << loop_rmsd( pose, native_pose, one_loop_loops ) << std::endl;
	if (loop.is_extended() ) {
		myKinematicMover.set_idealize_loop_first( true ); // start without any native angles or lengths
		core::Size nits=0;
		while (nits < max_kic_build_attempts_) {
			tr << "Attempting loop building: " << nits << " ... " << std::endl;
			myKinematicMover.apply( pose );
			myKinematicMover.allow_soft_bump_if_allowed( false );
			if (myKinematicMover.last_move_succeeded()) {
				set_last_move_status(protocols::moves::MS_SUCCESS);
				tr << "initial kinematic perturbation complete" << std::endl;
				myKinematicMover.set_idealize_loop_first( false ); // now the loop is idealized
				break;
			}
			nits++;
		}
		if (!myKinematicMover.last_move_succeeded()) {
			tr.Error << "[WARNING] Failed to build loop with kinematic Mover during initial kinematic perturbation after " << nits << " trials: " << loop << std::endl;
			set_last_move_status(protocols::moves::FAIL_RETRY);
			//pose.fold_tree( f_orig ); // DJM: doing above in LoopRelaxMover now
			return CriticalFailure;
		}
		(*scorefxn_)(pose);
		minimizer->run( pose, mm_one_loop, *scorefxn_, options );
		tr << "loop rmsd after initial kinematic perturbation:" << loop_rmsd( pose, native_pose, one_loop_loops ) << std::endl;
	}
	else {
		tr << "not performing initial kinematic perturbation" << std::endl;
		if (option[ OptionKeys::loops::vicinity_sampling ]()) {
			perturber->set_sample_vicinity( true );
			perturber->set_degree_vicinity( option[ OptionKeys::loops::vicinity_degree ]() );
		}
	}

	if (local_movie) {
		std::string outname_base = option[ OptionKeys::loops::output_pdb ]().name();
		std::string outname_prefix = option[ OptionKeys::out::prefix ];
		std::string outname = outname_prefix + outname_base + "_centroid_movie_" +
			right_string_of(cur_struct,4,'0') + ".pdb";
		loop_outfile.open(outname.c_str(), std::ios::out | std::ios::binary);
		loop_outfile << "MODEL" << std::endl;
		utility::vector1<Size> indices(loop_end - loop_begin + 3);
		for (Size i=loop_begin-1, j=1; i<=loop_end+1; i++, j++) {
			indices[j]=i;
		}
		//pose.dump_pdb(loop_outfile, indices, "init_perturb");
		loop_outfile << "ENDMDL" << std::endl;
	}

	// Monte Carlo object
	protocols::moves::MonteCarlo mc( pose, *scorefxn_, temperature);
	mc.show_scores();

	for( int i=1; i<=outer_cycles; ++i ) {
		if ( local_debug) { // debug
			(*scorefxn_)( pose );
			tr << "befor rLOW: " << pose.energies().total_energies().weighted_string_of( scorefxn_->weights() ) <<
				" rmsd: " << F(9,3,loop_rmsd( pose, native_pose, one_loop_loops )) << std::endl;
		}

		// recover low
		if ( recover_low_ ) {
			mc.recover_low(pose);
		}

		if ( local_debug) { // debug
			(*scorefxn_)( pose );
			tr << "after rLOW: " << pose.energies().total_energies().weighted_string_of( scorefxn_->weights() ) <<
				" rmsd: " << F(9,3,loop_rmsd( pose, native_pose, one_loop_loops )) << std::endl;
		}

		for( int j=1; j<=inner_cycles; ++j ) {
			// change temperature
			temperature *= gamma;
			mc.set_temperature( temperature );
			core::Size nits=0;
			while (nits < remodel_kic_attempts_) {
				nits++;
				kic_start = RG.random_range(loop_begin, loop_end-2);
				// choose a random end residue so the length is >= 3, <= min(loop_end, start+maxlen)
				kic_end = RG.random_range(kic_start+2, std::min((kic_start+max_seglen_ - 1), loop_end));
				middle_offset = (kic_end - kic_start) / 2;
				kic_middle = kic_start + middle_offset;
				myKinematicMover.set_pivots(kic_start, kic_middle, kic_end);
				myKinematicMover.set_temperature(temperature);
				myKinematicMover.apply( pose );
				if (myKinematicMover.last_move_succeeded()) {
					break;
				}
			}
			if (myKinematicMover.last_move_succeeded()) {
				//fpd symmetrize 'mm_one_loop'
				if ( core::conformation::symmetry::is_symmetric( pose ) )  {
					core::conformation::symmetry::make_symmetric_movemap( pose, mm_one_loop );
				}
				(*scorefxn_)(pose);
				minimizer->run( pose, mm_one_loop, *scorefxn_, options );
				std::string move_type = "kinematic_perturb";
				bool accepted = mc.boltzmann( pose );
				if (accepted) {
					tr << "new centroid perturb rmsd: " << loop_rmsd( pose, native_pose, one_loop_loops ) << std::endl;
					if (local_movie) {
						loop_outfile << "MODEL" << std::endl;
						utility::vector1<Size> indices(loop_end - loop_begin + 3);
						for (Size i=loop_begin-1, j=1; i<=loop_end+1; i++, j++) {
							indices[j]=i;
						}
						pose.dump_pdb(loop_outfile, indices, "init_perturb");
						loop_outfile << "ENDMDL" << std::endl;
					}
					//tr << "chainbreak score: " << pose.energies().total_energies()[ core::scoring::chainbreak ] << std::endl;
				}
				//mc.show_scores();
			}else{
				tr.Error << "[WARNING] Failed to build loop with kinematic Mover after " << nits << " trials: " << loop << std::endl;
				// return to original fold tree
				//pose.fold_tree( f_orig ); // DJM: doing above in LoopRelaxMover now
				return Failure;
			}
		} // inner_cycles
	} // outer_cycles
	if ( recover_low_ ) {
		pose = mc.lowest_score_pose();
	}
	else {
		pose = mc.last_accepted_pose();
	}
	if (local_movie) {
		// this assumes there is only one loop.
		Size begin_loop=loop.start();
		Size end_loop=loop.stop();
		loop_outfile << "MODEL" << std::endl;
		utility::vector1<Size> indices(end_loop - begin_loop + 3);
		for (Size i=begin_loop-1, j=1; i<=end_loop+1; i++, j++) {
			indices[j]=i;
		}
		pose.dump_pdb(loop_outfile, indices, "final_perturb");
		loop_outfile << "ENDMDL" << std::endl;
	}
	if (local_debug) {
		std::ofstream out("score.tmp_perturb_cen");
		out << "scoring after cen_perturb: " << (*scorefxn_)(pose) << std::endl;
		/// Now handled automatically.  scorefxn_->accumulate_residue_total_energies(pose);
		scorefxn_->show( out );
		out << pose.energies();
	}

	// return to original fold tree
	//pose.fold_tree( f_orig ); // DJM: doing above in LoopRelaxMover now

	return Success;
}


LoopMover_Refine_KIC::LoopMover_Refine_KIC(
	protocols::loops::Loops  loops_in
) : LoopMover( loops_in )
{
	scorefxn_ = core::scoring::getScoreFunction();
	protocols::moves::Mover::type("LoopMover_Refine_KIC");
	set_default_settings();
}

LoopMover_Refine_KIC::LoopMover_Refine_KIC(
	protocols::loops::Loops  loops_in,
	core::scoring::ScoreFunctionOP  scorefxn
) : LoopMover( loops_in )
{
	scorefxn_ = scorefxn;
	protocols::moves::Mover::type("LoopMover_Refine_KIC");
	set_default_settings();
}

void
LoopMover_Refine_KIC::set_default_settings(){
	using namespace core;
	using namespace options;

	redesign_loop_ = false;
	movemap_tweak_ = false;
	foldtree_tweak_ = false;
	neighbor_dist_ = option[ OptionKeys::loops::neighbor_dist ]();
	if( option[ OptionKeys::antibody::snug_loops ]() ) {
		movemap_tweak_ = true;
		foldtree_tweak_ = true;
	}
	max_seglen_ = option[OptionKeys::loops::kic_max_seglen];
	recover_low_ = ( ! core::options::option[core::options::OptionKeys::loops::kic_recover_last] );
}


void LoopMover_Refine_KIC::apply(
	core::pose::Pose & pose
){
	static int cur_struct=0; // for movie output

	using namespace core;
	using namespace optimization;
	using namespace scoring;
	using namespace options;

	// Set the native pose if provided
	core::pose::Pose native_pose;
	if( get_native_pose() ){
		native_pose = *get_native_pose();
	}else{
		native_pose = pose;
	}

	if( movemap_tweak_ || foldtree_tweak_ ) {
		protocols::antibody::Antibody antibody_in( pose );
		antibody_in.all_cdr_fold_tree();
		tweaked_foldtree_ = antibody_in.Fv.fold_tree();
		loops::loops_set_move_map( antibody_in.Fv,
															 antibody_in.all_cdr_loops,
															 true, // include loop neighbors
															 tweaked_movemap_ );
	}

	// 'verbose' should be an option, or better yet Tracer's output filtering capabilities should be used instead
	bool const verbose( true );
	bool const local_debug( option[ OptionKeys::loops::debug ].user() );
	bool const local_movie( false );
	std::ofstream loop_outfile; // for movie

	// prepare for movie output if requested
	if (local_movie) {
		std::string outname_base = option[ OptionKeys::loops::output_pdb ]().name();
		std::string outname_prefix = option[ OptionKeys::out::prefix ];
		std::string outname = outname_prefix + outname_base + "_fullatom_movie_" +
			right_string_of(cur_struct,4,'0') + ".pdb";
		loop_outfile.open(outname.c_str(), std::ios::out | std::ios::binary);
	}

	// set cutpoint variants for correct chainbreak scoring
	Size const nres( pose.total_residue() );
	utility::vector1< bool > is_loop( nres, false );

	for( Loops::const_iterator it=loops_.begin(), it_end=loops_.end();
		 it != it_end; ++it ) {
		for ( Size i= it->start(); i<= it->stop(); ++i ) {
			is_loop[i] = true;
		}
		Size const loop_cut(it->cut());

		if ( loop_cut != nres ) { //c-terminal loop
			if ( ! pose.residue(loop_cut).has_variant_type(chemical::CUTPOINT_LOWER) )
				chemical::add_variant_type_to_pose_residue( pose, chemical::CUTPOINT_LOWER, loop_cut );
			if ( ! pose.residue(loop_cut+1).has_variant_type(chemical::CUTPOINT_UPPER) )
				chemical::add_variant_type_to_pose_residue( pose, chemical::CUTPOINT_UPPER, loop_cut+1 );
		}
	}

	// scheduler
	int const fast = option[OptionKeys::loops::fast];
	int outer_cycles(3);
	if ( option[ OptionKeys::loops::outer_cycles ].user() ) {
		outer_cycles = option[ OptionKeys::loops::outer_cycles ]();
	}
	if ( option[ OptionKeys::run::test_cycles ]() ) {
		outer_cycles = 3;
	}
	int max_inner_cycles( 200 );
	if ( option[ OptionKeys::loops::max_inner_cycles ].user() ) {
		max_inner_cycles = option[ OptionKeys::loops::max_inner_cycles ]();
	}
	if ( option[ OptionKeys::run::test_cycles ]() ) {
		max_inner_cycles = 3;
	}

	int const inner_cycles = std::min( Size(max_inner_cycles), fast ? (int)loops_.loop_size() : 10*loops_.loop_size() );
	int repack_period = 20; // should be an option
	if ( option[ OptionKeys::loops::repack_period ].user() ) {
		repack_period = option[ OptionKeys::loops::repack_period ]();
	}

	// scorefxn
	scoring::ScoreFunctionOP scorefxn;
	scoring::ScoreFunctionOP min_scorefxn;
	if ( scorefxn_ != 0 ) scorefxn = scorefxn_->clone();
	else {
		scorefxn = ScoreFunctionFactory::create_score_function( STANDARD_WTS, SCORE12_PATCH );
	}

	// For testing JK's new solvation term. The exact form isn't yet differentiable
	min_scorefxn = scorefxn->clone();
	if ( min_scorefxn->get_weight( core::scoring::occ_sol_exact ) > 0.0 ) {
		min_scorefxn->set_weight( core::scoring::fa_sol, 0.65 );
		min_scorefxn->set_weight( core::scoring::occ_sol_exact, 0.0 );
	}

	//scoring::ScoreFunctionOP scorefxn( ScoreFunctionFactory::create_score_function(STANDARD_WTS, SCORE12_PATCH) );
	//scorefxn->set_weight( chainbreak, 1.0 ); // confirm that chainbreak weight is set
	scorefxn->set_weight( chainbreak, 1.0*10.0/3.0 ); // confirm that chainbreak weight is set
	min_scorefxn->set_weight( chainbreak, 1.0*10.0/3.0 ); // confirm that chainbreak weight is set
	//scorefxn->set_weight(core::scoring::mm_bend, 1.0);


	// monte carlo
	float const init_temp( option[ OptionKeys::loops::refine_init_temp ]() );
	float const	final_temp( option[ OptionKeys::loops::refine_final_temp ]() );
	float const gamma = std::pow( (final_temp/init_temp), 1.0f/(outer_cycles*inner_cycles) );
	float temperature = init_temp;
	protocols::moves::MonteCarlo mc( pose, *scorefxn, temperature );

	// minimizer
 	AtomTreeMinimizerOP minimizer;
 	MinimizerOptions options( "dfpmin", 0.001, true /*use_nblist*/, false /*deriv_check*/ );
 	if ( core::conformation::symmetry::is_symmetric( pose ) ) {
 		minimizer = new core::optimization::symmetry::SymAtomTreeMinimizer;
 	} else {
 		minimizer = new core::optimization::AtomTreeMinimizer;
 	}

	// show temps
	tr << "refine init temp: " << init_temp << std::endl;
	tr << "refine final temp: " << final_temp << std::endl;

	// default move map
	bool const fix_natsc = option[OptionKeys::loops::fix_natsc];

	// Set up the packer tasks: one for rotamer trials, one for repacking (with design if resfile supplied)
	using namespace pack::task;
	if ( task_factory == 0 ) {
		task_factory = new TaskFactory;
		// TaskOperations replace the following kind of code:
		// base_packer_task->initialize_from_command_line().or_include_current( true );
		task_factory->push_back( new operation::InitializeFromCommandline );
		task_factory->push_back( new operation::IncludeCurrent );
		if ( option[ OptionKeys::packing::resfile ].user() ) {
			// Note - resfile is obeyed, so use NATAA as default to maintain protocol behavior
			task_factory->push_back( new core::pack::task::operation::ReadResfile );
		}
	}

	PackerTaskOP base_packer_task = task_factory->create_task_and_apply_taskoperations( pose );
	base_packer_task->set_bump_check( true );
	pack::task::PackerTaskOP rottrials_packer_task( base_packer_task->clone() );
	pack::task::PackerTaskOP repack_packer_task( base_packer_task->clone() );
	rottrials_packer_task->restrict_to_repacking();
	if ( !option[ OptionKeys::packing::resfile ].user() && !redesign_loop_ ) {
		// Not designing -- just repack
		repack_packer_task->restrict_to_repacking();
		tr << "Not designing" << std::endl;
	}
	else tr << "Activating design" << std::endl;

	// setting redes loop, but no resfile specified. all non-loop positions only repack. loop positions can design.
	if( redesign_loop_ && !option[ OptionKeys::packing::resfile ].user() ) {
		for( core::Size i = 1; i <= pose.total_residue(); ++i ) {
			if( !is_loop[i] ) repack_packer_task->nonconst_residue_task( i ).restrict_to_repacking();
		}
	}

	// setup kinematic mover
	//protocols::moves::KinematicMover myKinematicMover( init_temp );
	protocols::moves::KinematicMover myKinematicMover;

	protocols::moves::kinematic_closure::TorsionSamplingKinematicPerturberOP perturber =
		new protocols::moves::kinematic_closure::TorsionSamplingKinematicPerturber( &myKinematicMover );
	if (option[ OptionKeys::loops::vicinity_sampling ]()) {
		perturber->set_sample_vicinity( true );
		perturber->set_degree_vicinity( option[ OptionKeys::loops::vicinity_degree ]() );
	}
	perturber->set_vary_ca_bond_angles( true );
	myKinematicMover.set_perturber( perturber );

	myKinematicMover.set_vary_bondangles( true );
	myKinematicMover.set_sample_nonpivot_torsions( option[ OptionKeys::loops::nonpivot_torsion_sampling ]());
	myKinematicMover.set_rama_check( true );
	Size kic_start, kic_middle, kic_end; // three pivot residues for kinematic loop closure

	// perform initial repack trial
	utility::vector1<bool> allow_sc_move( nres, false );
	(*scorefxn)(pose); // update 10A nbr graph, silly way to do this
	// here we'll optimize all side-chains within neighbor_dist_ of any loop (unless fix_natsc is true)
	select_loop_residues( pose, loops_, !fix_natsc, allow_sc_move, neighbor_dist_);
	core::conformation::symmetry::make_residue_mask_symmetric( pose, allow_sc_move );
	repack_packer_task->restrict_to_residues( allow_sc_move );
	rottrials_packer_task->restrict_to_residues( allow_sc_move );
	core::pack::pack_rotamers( pose, *scorefxn, repack_packer_task );
	std::string move_type = "repack";
	// minimize after initial repack
	pose.update_residue_neighbors(); // to update 10A nbr graph
	kinematics::MoveMap mm_all_loops; // DJM tmp
	loops_set_move_map( pose, loops_, fix_natsc, mm_all_loops, neighbor_dist_);
	pre_min_map_tree_changes( mm_all_loops, pose );
	minimizer->run( pose, mm_all_loops, *min_scorefxn, options ); // DJM tmp
	post_min_map_tree_changes( mm_all_loops, pose );
	mc.boltzmann( pose, move_type );
	mc.show_scores();

	if (local_movie) {
		// this assumes there is only one loops_.
		Loops::const_iterator one_loop( loops_.begin() );
		Size begin_loop=one_loop->start();
		Size end_loop=one_loop->stop();
		loop_outfile << "MODEL" << std::endl;
		utility::vector1<Size> indices(end_loop - begin_loop + 3);
		for (Size i=begin_loop-1, j=1; i<=end_loop+1; i++, j++) {
			indices[j]=i;
		}
		pose.dump_pdb(loop_outfile, indices, "initial_repack");
		loop_outfile << "ENDMDL" << std::endl;
	}

	// move map
	//kinematics::MoveMap mm_all_loops; // DJM: tmp commented out
	loops_set_move_map( pose, loops_, fix_natsc, mm_all_loops, neighbor_dist_);

	for (int i=1; i<=outer_cycles; ++i) {
		// increase CHAINBREAK weight and update monte carlo
		scorefxn->set_weight( chainbreak, float(i)*10.0/3.0 );
		min_scorefxn->set_weight( chainbreak, float(i)*10.0/3.0 );
		//scorefxn->set_weight( chainbreak, float(i) );
		mc.score_function( *scorefxn );
		// recover low
		if ( recover_low_ ) {
			mc.recover_low( pose );
		}
		// score info
		if ( verbose ) tr << "cycle: " << i << "  " << (*scorefxn)(pose) << std::endl;
		//pose.energies().show( tr );
		for ( int j=1; j<=inner_cycles; ++j ) {
			temperature *= gamma;
			mc.set_temperature( temperature );
			if ( verbose ) tr << "refinement cycle (outer/inner): "
				<< i << "/" << outer_cycles << " "
				<< j << "/" << inner_cycles << " "
				<< std::endl;

			// choose a random loop for both rounds
			Loops::const_iterator it( loops_.one_random_loop() );
			Loops one_loop;
			one_loop.add_loop( it );
			// set up movemap properly
			kinematics::MoveMapOP mm_one_loop( new kinematics::MoveMap() );
			loops_set_move_map( pose, one_loop, fix_natsc, *mm_one_loop, neighbor_dist_);
			// get loop endpoints
			Size begin_loop=one_loop.begin()->start();
			Size end_loop=one_loop.begin()->stop();

			{// kinematic trial first round
				kic_start = RG.random_range(begin_loop,end_loop-2);
				// choose a random end residue so the length is >= 3, <= min(loop_end, start+maxlen)
				kic_end = RG.random_range(kic_start+2, std::min((kic_start+max_seglen_ - 1), end_loop));
				Size middle_offset = (kic_end - kic_start) / 2;
				kic_middle = kic_start + middle_offset;
				myKinematicMover.set_pivots(kic_start, kic_middle, kic_end);
				myKinematicMover.set_temperature(temperature);
				myKinematicMover.apply( pose );
				if ( myKinematicMover.last_move_succeeded() ) {
					//fpd symmetrize mm_all_loops
					if ( core::conformation::symmetry::is_symmetric( pose ) )  {
						core::conformation::symmetry::make_residue_mask_symmetric( pose, allow_sc_move );
					}
					//rottrials_packer_task->restrict_to_residues( allow_sc_move );

					// do optimizations
					pack::rotamer_trials( pose, *scorefxn, rottrials_packer_task );
					pose.update_residue_neighbors(); // to update 10A nbr graph
					//loops_set_move_map( pose, loops_, fix_natsc, mm_all_loops, neighbor_dist_ ); // DJM: trying only in repack
					pre_min_map_tree_changes( mm_all_loops, pose );
					if ( core::conformation::symmetry::is_symmetric( pose ) )  {
						//fpd  minimizing with the reduced movemap seems to cause occasional problems
						//     in the symmetric case ... am looking into this
						loops_set_move_map( pose, loops_, fix_natsc, mm_all_loops, neighbor_dist_ );
						minimizer->run( pose, mm_all_loops, *min_scorefxn, options );
					} else {
						//minimizer->run( pose, *mm_one_loop, *min_scorefxn, options );
						minimizer->run( pose, mm_all_loops, *min_scorefxn, options );
					}
					post_min_map_tree_changes( mm_all_loops, pose );

					// test for acceptance
					std::string move_type = "kic_refine_r1";
					bool accepted = mc.boltzmann( pose, move_type );
					if (accepted) {
						tr << "RMS to native after accepted kinematic round 1 move: "
						   << loop_rmsd(pose, native_pose, loops_) << std::endl;
						//tr << "after accepted move res " << kic_start + 2 << " omega is " << pose.omega(kic_start+2) << std::endl;
						if (local_movie) {
							loop_outfile << "MODEL" << std::endl;
							utility::vector1<Size> indices(end_loop - begin_loop + 3);
							for (Size i=begin_loop-1, j=1; i<=end_loop+1; i++, j++) {
								indices[j]=i;
							}
							pose.dump_pdb(loop_outfile, indices, "refine_r1");
							loop_outfile << "ENDMDL" << std::endl;
						}
						//tr << "temperature: " << temperature << std::endl;
						//tr << "chainbreak: " << pose.energies().total_energies()[ scoring::chainbreak ] << std::endl;
					}
					//mc.show_scores();
				}
			}

			{// kinematic trial second round
				kic_start = RG.random_range(begin_loop,end_loop-2);
				// choose a random end residue so the length is >= 3, <= min(loop_end, start+maxlen)
				kic_end = RG.random_range(kic_start+2, std::min((kic_start+max_seglen_ - 1), end_loop));
				Size middle_offset = (kic_end - kic_start) / 2;
				kic_middle = kic_start + middle_offset;
				myKinematicMover.set_pivots(kic_start, kic_middle, kic_end);
				myKinematicMover.set_temperature(temperature);
				myKinematicMover.apply( pose );

				if ( myKinematicMover.last_move_succeeded() ) {
					if ( core::conformation::symmetry::is_symmetric( pose ) )  {
						core::conformation::symmetry::make_residue_mask_symmetric( pose, allow_sc_move );
					}
					//rottrials_packer_task->restrict_to_residues( allow_sc_move );

					// do optimizations
					pack::rotamer_trials( pose, *scorefxn, rottrials_packer_task );
					pose.update_residue_neighbors(); // to update 10A nbr graph
					//loops_set_move_map( pose, loops_, fix_natsc, mm_all_loops, neighbor_dist_); // DJM: trying only in repack
					pre_min_map_tree_changes( mm_all_loops, pose );
					//minimizer->run( pose, mm_all_loops, *min_scorefxn, options );
					if ( core::conformation::symmetry::is_symmetric( pose ) )  {
						//fpd  minimizing with the reduced movemap seems to cause occasional problems
						//     in the symmetric case ... am looking into this
						loops_set_move_map( pose, loops_, fix_natsc, mm_all_loops, neighbor_dist_ );
						minimizer->run( pose, mm_all_loops, *min_scorefxn, options );
					} else {
						minimizer->run( pose, mm_all_loops, *min_scorefxn, options );
					}
					post_min_map_tree_changes( mm_all_loops, pose );

					// test for acceptance
					std::string move_type = "kic_refine_r2";
					bool accepted = mc.boltzmann( pose, move_type );
					if (accepted) {
						tr << "RMS to native after accepted kinematic round 2 move: "
						<< loop_rmsd(pose, native_pose, loops_) << std::endl;
						if (local_movie) {
							loop_outfile << "MODEL" << std::endl;
							utility::vector1<Size> indices(end_loop - begin_loop + 3);
							for (Size i=begin_loop-1, j=1; i<=end_loop+1; i++, j++) {
								indices[j]=i;
							}
							pose.dump_pdb(loop_outfile, indices, "refine_r2");
							loop_outfile << "ENDMDL" << std::endl;
						}
						//tr << "chainbreak score: " << pose.energies().total_energies()[ core::scoring::chainbreak ] << std::endl;
					}
					//mc.show_scores();
				}
			}

		 	{ //main_repack_trial
 				if ( (j%repack_period)==0 || j==inner_cycles ) {
 					// repack trial
					//pack::task::PackerTaskOP this_packer_task( base_packer_task->clone() );
					// DJM: try updating existing packer task
					loops_set_move_map( pose, loops_, fix_natsc, mm_all_loops, neighbor_dist_); // DJM: trying only in repack
					utility::vector1<bool> allow_sc_move_all_loops( nres, false );
					select_loop_residues( pose, loops_, !fix_natsc, allow_sc_move_all_loops, neighbor_dist_);
					core::conformation::symmetry::make_residue_mask_symmetric( pose, allow_sc_move_all_loops );  //fpd symmetrize res mask -- does nothing if pose is not symm
					repack_packer_task->restrict_to_residues( allow_sc_move_all_loops );
					rottrials_packer_task->restrict_to_residues( allow_sc_move_all_loops ); // for subsequent cycles until repack
					pack::pack_rotamers( pose, *scorefxn, repack_packer_task ); // design here if resfile supplied
					std::string move_type = "repack";
					mc.boltzmann( pose, move_type );
					mc.show_scores();
				}
			}
			if ( verbose || local_debug ) tr << std::flush;
		} //inner_cycle
	} //outer_cycle

	mc.show_counters();
	if ( recover_low_ ) {
		pose = mc.lowest_score_pose();
	}
	else {
		pose = mc.last_accepted_pose();
	}
	if (local_movie) {
		// this assumes there is only one loops_.
		Loops::const_iterator one_loop( loops_.begin() );
		Size begin_loop=one_loop->start();
		Size end_loop=one_loop->stop();
		loop_outfile << "MODEL" << std::endl;
		utility::vector1<Size> indices(end_loop - begin_loop + 3);
		for (Size i=begin_loop-1, j=1; i<=end_loop+1; i++, j++) {
			indices[j]=i;
		}
		pose.dump_pdb(loop_outfile, indices, "final_refine");
		loop_outfile << "ENDMDL" << std::endl;
	}


}


void
LoopMover_Refine_KIC::pre_min_map_tree_changes(
  core::kinematics::MoveMap & current_mm,
	core::pose::Pose & current_pose) {

	default_movemap_ = current_mm;
	default_foldtree_ = current_pose.fold_tree();

	if( movemap_tweak_ ) {
		std::cout << "Changing map, "; // aroop_temp remove
		current_mm = tweaked_movemap_;
	}

	if( foldtree_tweak_ ) {
		std::cout << "Changing tree "; // aroop_temp remove
		current_pose.fold_tree( tweaked_foldtree_ );
	}

	std::cout << std::endl; // aroop_temp remove

	return;
}


void
LoopMover_Refine_KIC::post_min_map_tree_changes(
	core::kinematics::MoveMap & current_mm,
	core::pose::Pose & current_pose) {

	if( movemap_tweak_ )
		current_mm = default_movemap_;

	if( foldtree_tweak_ )
		current_pose.fold_tree( default_foldtree_ );

	return;
}

} // namespace loops
} // namespace protocols
