// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
// :noTabs=false:tabSize=4:indentSize=4:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   protocols/match/OccupiedSpaceHash.hh
/// @brief  Declaration for classes in 6D hasher
/// @author Alex Zanghellini (zanghell@u.washington.edu)
/// @author Andrew Leaver-Fay (aleaverfay@gmail.com), porting to mini

#ifndef INCLUDED_protocols_match_HitHasher_HH
#define INCLUDED_protocols_match_HitHasher_HH

// Unit headers
#include <protocols/match/MatchSet.fwd.hh>

// Package headers
#include <protocols/match/SixDHasher.hh>

// Utility headers
#include <utility/pointer/ReferenceCount.hh>
#include <utility/fixedsizearray1.hh>
#include <utility/vector1.hh>

/// Boost headers
#include <boost/cstdint.hpp>
#include <boost/unordered_map.hpp>

/// C++ headers
#include <list>

namespace protocols {
namespace match {


/// @brief This object hashes hits into 6D voxels.  This hash can then be traversed
/// to retrieve the hits that hash to the same voxel (matches!).  There are 64 hashes
/// representing the 2^6 ways to perturb the bins in 6D by 1/2 of their bin width.
///
/// @details The hit hasher expects someone else to own the hits.  It takes as input
/// constant pointers to the hits that exist and uses their addresses to hash upon.
/// The hit hasher should only be used if you can guarantee that the hits it points
/// to will outlive the hasher.
class HitHasher : public utility::pointer::ReferenceCount {
public:
	typedef core::Real                               Real;
	typedef core::Size                               Size;
	typedef core::Vector                             Vector;
	typedef numeric::geometry::BoundingBox< Vector > BoundingBox;
	typedef utility::vector1< std::list< Hit const * > >     MatchSet;
	typedef boost::unordered_map< boost::uint64_t, MatchSet, bin_index_hasher > HitHash;

public:
	HitHasher();
	~HitHasher();

	void
	set_bounding_box(
		BoundingBox const & bb
	);

	void
	set_uniform_xyz_bin_width( Real bin_width );

	void
	set_uniform_euler_angle_bin_width( Real bin_width_degrees );

	void
	set_xyz_bin_widths( Vector const & bin_widths );

	void
	set_euler_bin_widths( Vector const & euler_bin_widths );

	void
	set_nhits_per_match( Size num_geometric_constraints );

	void
	initialize();

	/// @brief Insert a hits into all 64 hash maps
	void
	insert_hit( Size geometric_constraint_id, Hit const * hit );

	/// @brief Insert a hits into a particular hash maps
	void
	insert_hit( Size which_hash_map, Size geometric_constraint_id, Hit const * hit );

	void
	clear_hash_map( Size which_hash_map );

	HitHash::const_iterator
	hit_hash_begin( Size which_hash_map ) const;

	HitHash::const_iterator
	hit_hash_end( Size which_hash_map ) const;

private:
	static Size const N_HASH_MAPS = 64; // 2^6 -- all combinations of offsets for 6 dimensions

	BoundingBox bb_;

	bool initialized_;

	utility::fixedsizearray1< Real, 3 > xyz_bin_widths_;
	utility::fixedsizearray1< Real, 3 > euler_bin_widths_;
	Size n_geometric_constraints_per_match_;

	utility::vector1< std::pair< SixDCoordinateBinnerOP, HitHash > > hit_hashes_;

};

typedef utility::vector1< Hit const * > match_lite;

struct match_lite_equals {

	bool
	operator() ( match_lite const & lhs, match_lite const & rhs ) const {
		if ( lhs.size() != rhs.size() ) return false;
		for ( core::Size ii = 1; ii <= lhs.size(); ++ii ) {
			if ( lhs[ ii ] != rhs[ ii ] ) {
				return false;
			}
		}
		return true;
	}

};

struct match_lite_hasher {
public:
	typedef core::Size Size;

public:
	Size
	operator() ( match_lite const & m ) const {
		/// Crazy hash function!
		Size hash = 1;
		for ( core::Size ii = 1; ii <= m.size(); ++ii ) {
			hash += ( hash * reinterpret_cast< Size > ( m[ ii ] ) * ii ) % 5527;
		}
		return hash % 7351;
	}
};

class MatchOutputTracker
{
public:

	typedef boost::unordered_map< match_lite, bool, match_lite_hasher, match_lite_equals > MatchHash;

public:
	MatchOutputTracker();

	void
	note_output_match( match_lite const & );

	bool
	match_has_been_output( match_lite const & m ) const;

private:
	MatchHash hash_;

};


}
}

#endif
