// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
// :noTabs=false:tabSize=4:indentSize=4:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   protocols/match/Mather.hh
/// @brief
/// @author Alex Zanghellini (zanghell@u.washington.edu)
/// @author Andrew Leaver-Fay (aleaverfay@gmail.com), porting to mini

// Unit headers
#include <protocols/match/Matcher.hh>

// Package headers
#include <protocols/match/Hit.hh>
#include <protocols/match/BumpGrid.hh>
#include <protocols/match/MatcherTask.hh>
#include <protocols/match/MatchSet.hh>
#include <protocols/match/OccupiedSpaceHash.hh>

#include <protocols/match/downstream/ActiveSiteGrid.hh>
#include <protocols/match/downstream/ClassicMatchAlgorithm.hh>
#include <protocols/match/downstream/DownstreamAlgorithm.hh>
#include <protocols/match/downstream/DownstreamBuilder.hh>
#include <protocols/match/downstream/ExternalGeomSampler.hh>
#include <protocols/match/downstream/RigidLigandBuilder.hh>
#include <protocols/match/downstream/SecondaryMatcherToDownstreamResidue.hh>
#include <protocols/match/downstream/SecondaryMatcherToUpstreamResidue.hh>
#include <protocols/match/downstream/GeometrySecMatchRPE.hh>

#include <protocols/match/output/MatchProcessor.hh>

#include <protocols/match/upstream/ProteinSCSampler.hh>
#include <protocols/match/upstream/ProteinUpstreamBuilder.hh>
#include <protocols/match/upstream/OriginalScaffoldBuildPoint.hh>
#include <protocols/match/upstream/UpstreamBuilder.hh>

// Project headers
#include <protocols/enzdes/EnzConstraintIO.hh>
#include <protocols/enzdes/EnzConstraintParameters.hh>
#include <protocols/enzdes/MatchConstraintFileInfo.hh>

#include <core/pose/Pose.hh>

#include <core/util/Tracer.hh>

#include <core/scoring/ScoringManager.hh>
#include <core/scoring/dunbrack/SingleResidueDunbrackLibrary.hh>
#include <core/scoring/dunbrack/SemiRotamericSingleResidueDunbrackLibrary.hh>
#include <core/scoring/dunbrack/SemiRotamericSingleResidueDunbrackLibrary.tmpl.hh>
#include <core/scoring/etable/count_pair/CountPairFunction.hh>
#include <core/scoring/etable/count_pair/CountPairGeneric.hh>

// Utility headers
#include <utility/LexicographicalIterator.hh>
#include <utility/OrderedTuple.hh>
#include <utility/string_util.hh>

// C++ headers
#include <map>
#include <fstream>
#include <string>
#include <ctime>

namespace protocols {
namespace match {

static core::util::Tracer TR( "protocols.match.Matcher" );

/// Construction and Destruction
Matcher::Matcher() :
	same_build_resids_for_all_csts_( true ),
	n_geometric_constraints_( 0 ),
	read_gridlig_file_( false ),
	use_input_sc_( false ),
	output_matches_as_singular_downstream_positioning_( false )
{}

Matcher::~Matcher() {}

void Matcher::set_upstream_pose( core::pose::Pose const & pose )
{
	upstream_pose_ = new core::pose::Pose( pose );
}

void Matcher::set_downstream_pose(
	core::pose::Pose const & pose,
	utility::vector1< core::id::AtomID > orientation_atoms
)
{
	runtime_assert( orientation_atoms.size() == 3 );

	downstream_pose_ = new core::pose::Pose( pose );
	downstream_orientation_atoms_ = orientation_atoms;
}

void
Matcher::set_original_scaffold_build_points( utility::vector1< Size > const & resids )
{
	same_build_resids_for_all_csts_ = true;
	pose_build_resids_ = resids;
	std::sort( pose_build_resids_.begin(), pose_build_resids_.end() ); // keep sorted
	per_cst_build_resids_.resize( 0 );
}

void Matcher::set_original_scaffold_build_points_for_constraint(
	Size cst_id,
	utility::vector1< Size > const & resids
)
{
	same_build_resids_for_all_csts_ = false;
	runtime_assert( n_geometric_constraints_ > 0 ); // n_geometric_constraints_ must be set first
	if ( per_cst_build_resids_.size() == 0 ) {
		per_cst_build_resids_.resize( n_geometric_constraints_ );
		pose_build_resids_.clear();
	}
	per_cst_build_resids_[ cst_id ] = resids;
	std::sort( per_cst_build_resids_[ cst_id ].begin(), per_cst_build_resids_[ cst_id ].end() );

	/// keep track of the individual resids that are built from by any geometric constraint
	std::list< Size > uniq_resids;
	for ( Size ii = 1; ii <= pose_build_resids_.size(); ++ii ) {
		uniq_resids.push_back( pose_build_resids_[ ii ] );
	}
	for ( Size ii = 1; ii <= per_cst_build_resids_[ cst_id ].size(); ++ii ) {
		uniq_resids.push_back( per_cst_build_resids_[ cst_id ][ ii ] );
	}

	uniq_resids.sort();
	uniq_resids.unique();

	pose_build_resids_.resize( uniq_resids.size() );
	std::copy( uniq_resids.begin(), uniq_resids.end(), pose_build_resids_.begin() );

}



void Matcher::set_n_geometric_constraints( Size n_constraints )
{
	n_geometric_constraints_ = n_constraints;

	hits_.resize( n_geometric_constraints_ );
	upstream_builders_.resize( n_geometric_constraints_ );
	build_set_id_for_restype_.resize( n_geometric_constraints_ );
	representative_downstream_algorithm_.resize( n_geometric_constraints_, 0 );
	//std::fill( representative_downstream_algorithm_.begin(), representative_downstream_algorithm_.end(), 0 );
	downstream_algorithms_.resize( n_geometric_constraints_ );
	geomcst_is_upstream_only_.resize( n_geometric_constraints_, false );
	downstream_builders_.resize( n_geometric_constraints_ );
	//std::fill( downstream_builders_.begin(), downstream_builders_.end(), 0 );
	per_constraint_build_points_.resize( n_geometric_constraints_ );
	geom_cst_has_primary_modification_.resize( n_geometric_constraints_ );
	std::fill( geom_cst_has_primary_modification_.begin(), geom_cst_has_primary_modification_.end(), false );
}

void Matcher::add_upstream_restype_for_constraint(
	Size cst_id,
	core::chemical::ResidueTypeCAP restype
)
{
	/// ASSUMPTION: matching from protein sidechain
	assert( restype->aa() <= core::chemical::num_canonical_aas );
	assert( build_set_id_for_restype_[ cst_id ].find( restype->name() ) == build_set_id_for_restype_[ cst_id ].end() );


	if ( ! upstream_builders_[ cst_id ] ) {
		upstream::ProteinUpstreamBuilderOP prot_sc_builder = new upstream::ProteinUpstreamBuilder;
		/// default to dunbrack sampler
		prot_sc_builder->set_sampler( new upstream::DunbrackSCSampler );
		prot_sc_builder->set_use_input_sidechain( use_input_sc_ );
		upstream_builders_[ cst_id ] = prot_sc_builder;
	}

	upstream::BuildSet build_set;
	build_set.set_residue_type( restype, restype->aa() == core::chemical::aa_gly ); // HACK gly means backbone only

	runtime_assert( dynamic_cast< upstream::ProteinUpstreamBuilder * > ( upstream_builders_[ cst_id ].get() ) );

	upstream::ProteinUpstreamBuilderOP prot_sc_builder( dynamic_cast< upstream::ProteinUpstreamBuilder * > ( upstream_builders_[ cst_id ].get() ));
	prot_sc_builder->add_build_set( build_set );
	build_set_id_for_restype_[ cst_id ][ restype->name() ] = prot_sc_builder->n_build_sets();

}

void Matcher::set_sample_startegy_for_constraint(
	Size cst_id,
	core::chemical::ResidueTypeCAP restype,
	Size chi,
	upstream::SampleStrategyData const & strat
)
{
	runtime_assert( build_set_id_for_restype_[ cst_id ].find( restype->name() )
		!= build_set_id_for_restype_[ cst_id ].end() );

	//Size build_set_id = build_set_id_for_restype_[ cst_id ][ restype->name() ];
	runtime_assert( dynamic_cast< upstream::ProteinUpstreamBuilder * > ( upstream_builders_[ cst_id ].get() ) );
	upstream::ProteinUpstreamBuilderOP prot_sc_builder( dynamic_cast< upstream::ProteinUpstreamBuilder * > ( upstream_builders_[ cst_id ].get() ));

	upstream::BuildSet & build_set = prot_sc_builder->build_set( restype );

	build_set.set_sample_strategy_for_chi( chi, strat );
}

void Matcher::add_external_geometry_samples_for_constraint(
	Size cst_id,
	core::chemical::ResidueTypeCAP restype,
	utility::vector1< std::string >  const & upstream_launch_atoms,
	utility::vector1< core::id::AtomID > const & downstream_3atoms,
	downstream::ExternalGeomSampler const & exgeom,
	Size const exgeom_id,
	bool catalytic_bond /*= false */
)
{
	TR << "     Adding Classical Match Algorithm with geometry samples: " << std::endl;
	TR << "     tor_U3D1:";
	for ( Size ii = 1; ii <= exgeom.n_tor_U3D1_samples(); ++ii ) {
		TR << " " << exgeom.tor_U3D1_samples()[ ii ];
	}
	TR << std::endl;

	TR << "     ang_U2D1:";
	for ( Size ii = 1; ii <= exgeom.n_ang_U2D1_samples(); ++ii ) {
		TR << " " << exgeom.ang_U2D1_samples()[ ii ];
	}
	TR << std::endl;

	TR << "     dis_U1D1:";
	for ( Size ii = 1; ii <= exgeom.n_dis_U1D1_samples(); ++ii ) {
		TR << " " << exgeom.dis_U1D1_samples()[ ii ];
	}
	TR << std::endl;

	TR << "     tor_U2D2:";
	for ( Size ii = 1; ii <= exgeom.n_tor_U2D2_samples(); ++ii ) {
		TR << " " << exgeom.tor_U2D2_samples()[ ii ];
	}
	TR << std::endl;

	TR << "     ang_U1D2:";
	for ( Size ii = 1; ii <= exgeom.n_ang_U1D2_samples(); ++ii ) {
		TR << " " << exgeom.ang_U1D2_samples()[ ii ];
	}
	TR << std::endl;

	TR << "     tor_U1D3:";
	for ( Size ii = 1; ii <= exgeom.n_tor_U1D3_samples(); ++ii ) {
		TR << " " << exgeom.tor_U1D3_samples()[ ii ];
	}
	TR << std::endl;


	runtime_assert( upstream_launch_atoms.size() == 3 );
	runtime_assert( downstream_3atoms.size() == 3 );

	runtime_assert( build_set_id_for_restype_[ cst_id ].find( restype->name() )
		!= build_set_id_for_restype_[ cst_id ].end() );

	downstream::DownstreamBuilderOP ds_builder = create_ds_builder(
		cst_id, restype, upstream_launch_atoms,
		downstream_3atoms, catalytic_bond );

	//Size build_set_id = build_set_id_for_restype_[ cst_id ][ restype->name() ];
	runtime_assert( dynamic_cast< upstream::ProteinUpstreamBuilder * > (
		upstream_builders_[ cst_id ].get() ) );
	upstream::ProteinUpstreamBuilderOP prot_sc_builder(
		static_cast< upstream::ProteinUpstreamBuilder * > ( upstream_builders_[ cst_id ].get() ));

	upstream::BuildSet & build_set = prot_sc_builder->build_set( restype );

	if ( ! build_set.has_algorithm() ) {
		downstream::ClassicMatchAlgorithmOP match_algorithm = new downstream::ClassicMatchAlgorithm( cst_id );
		match_algorithm->set_residue_type( restype );
		build_set.set_downstream_algorithm( match_algorithm );

		downstream_algorithms_[ cst_id ].push_back( match_algorithm );
		representative_downstream_algorithm_[ cst_id ] = match_algorithm;
		all_downstream_algorithms_.push_back( match_algorithm );
	}

	runtime_assert( dynamic_cast< downstream::ClassicMatchAlgorithm * > ( & build_set.algorithm() ) );
	downstream::ClassicMatchAlgorithm & algorithm( static_cast< downstream::ClassicMatchAlgorithm & > (build_set.algorithm() ) );

	algorithm.add_external_geom_sampler(
		exgeom,
		exgeom_id,
		upstream_launch_atoms[ 1 ],
		upstream_launch_atoms[ 2 ],
		upstream_launch_atoms[ 3 ],
		ds_builder
	);

}

/// Initialize a secondary matcher object based on the
/// geometry from the upstream to the downstream residue types.
void Matcher::add_secondary_upstream_match_geometry_for_constraint(
	Size geom_cst_id,
	Size target_geom_cst_id,
	core::chemical::ResidueTypeCAP candidate_restype,
	core::chemical::ResidueTypeCAP target_restype,
	utility::vector1< Size > const & candidate_atids,
	utility::vector1< Size > const & target_atids,
	enzdes::MatchConstraintFileInfoCOP mcfi
)
{
	using namespace downstream;

	runtime_assert( candidate_atids.size() == 3 );
	runtime_assert( target_atids.size() == 3 );

	runtime_assert( build_set_id_for_restype_[ geom_cst_id ].find( candidate_restype->name() )
		!= build_set_id_for_restype_[ geom_cst_id ].end() );

	runtime_assert( dynamic_cast< upstream::ProteinUpstreamBuilder * > (
		upstream_builders_[ geom_cst_id ].get() ) );
	upstream::ProteinUpstreamBuilderOP prot_sc_builder(
		static_cast< upstream::ProteinUpstreamBuilder * > ( upstream_builders_[ geom_cst_id ].get() ));

	upstream::BuildSet & build_set = prot_sc_builder->build_set( candidate_restype );

	if ( ! build_set.has_algorithm() ) {
		SecondaryMatcherToUpstreamResidueOP secondary_match_algorithm
			= new SecondaryMatcherToUpstreamResidue( geom_cst_id );
		build_set.set_downstream_algorithm( secondary_match_algorithm );
		secondary_match_algorithm->set_target_geomcst_id( target_geom_cst_id );
		downstream_algorithms_[ geom_cst_id ].push_back( secondary_match_algorithm );
		representative_downstream_algorithm_[ geom_cst_id ] = secondary_match_algorithm;
		all_downstream_algorithms_.push_back( secondary_match_algorithm );
	}

	runtime_assert( dynamic_cast< downstream::SecondaryMatcherToUpstreamResidue * > ( & build_set.algorithm() ) );
	downstream::SecondaryMatcherToUpstreamResidue & algorithm( static_cast< downstream::SecondaryMatcherToUpstreamResidue & > (build_set.algorithm() ) );

	algorithm.add_target_restype( target_restype );
	GeometrySecMatchRPEOP geom_evaluator = new GeometrySecMatchRPE( *mcfi, target_atids, candidate_atids );
	for ( Size ii = 1; ii <= geom_evaluator->atom_geom_rpes().size(); ++ii ) {
		TR << "    Upstream 2ndary Match: " << geom_evaluator->atom_geom_rpes()[ ii ]->print( candidate_restype, target_restype ) << std::endl;
	}

	algorithm.add_evaluator_for_target_restype( target_restype, geom_evaluator, mcfi->index() );
}

void
Matcher::add_secondary_downstream_match_geometry_for_constraint(
	Size geom_cst_id,
	core::chemical::ResidueTypeCAP candidate_restype,
	core::chemical::ResidueTypeCAP downstream_restype,
	utility::vector1< Size > const & candidate_atids,
	utility::vector1< Size > const & target_atids,
	enzdes::MatchConstraintFileInfoCOP mcfi
)
{
	using namespace downstream;

	runtime_assert( candidate_atids.size() == 3 );
	runtime_assert( target_atids.size() == 3 );

	runtime_assert( build_set_id_for_restype_[ geom_cst_id ].find( candidate_restype->name() )
		!= build_set_id_for_restype_[ geom_cst_id ].end() );

	runtime_assert( dynamic_cast< upstream::ProteinUpstreamBuilder * > (
		upstream_builders_[ geom_cst_id ].get() ) );
	upstream::ProteinUpstreamBuilderOP prot_sc_builder(
		static_cast< upstream::ProteinUpstreamBuilder * > ( upstream_builders_[ geom_cst_id ].get() ));

	upstream::BuildSet & build_set = prot_sc_builder->build_set( candidate_restype );

	if ( ! build_set.has_algorithm() ) {
		SecondaryMatcherToDownstreamResidueOP secondary_match_algorithm
			= new SecondaryMatcherToDownstreamResidue( geom_cst_id );
		build_set.set_downstream_algorithm( secondary_match_algorithm );
		secondary_match_algorithm->set_downstream_restype( downstream_restype );
		downstream_algorithms_[ geom_cst_id ].push_back( secondary_match_algorithm );
		representative_downstream_algorithm_[ geom_cst_id ] = secondary_match_algorithm;
		all_downstream_algorithms_.push_back( secondary_match_algorithm );
	}

	runtime_assert( dynamic_cast< downstream::SecondaryMatcherToDownstreamResidue * > ( & build_set.algorithm() ) );
	downstream::SecondaryMatcherToDownstreamResidue & algorithm( static_cast< downstream::SecondaryMatcherToDownstreamResidue & > (build_set.algorithm() ) );

	GeometrySecMatchRPEOP geom_evaluator = new GeometrySecMatchRPE( *mcfi, target_atids, candidate_atids );
	for ( Size ii = 1; ii <= geom_evaluator->atom_geom_rpes().size(); ++ii ) {
		TR << "    Downstream 2ndary Match: " << geom_evaluator->atom_geom_rpes()[ ii ]->print( candidate_restype, downstream_restype ) << std::endl;
	}

	algorithm.add_evaluator( geom_evaluator, mcfi->index() );

}


void
Matcher::set_occupied_space_bounding_box( BoundingBox const & bb )
{
	occ_space_bounding_box_ = bb;
	TR << "Set occupied space bounding box: Lower (";
	TR << bb.lower().x() << ", ";
	TR << bb.lower().y() << ", ";
	TR << bb.lower().z() << ") Upper (";
	TR << bb.upper().x() << ", ";
	TR << bb.upper().y() << ", ";
	TR << bb.upper().z() << ")" << std::endl;
}

void Matcher::set_hash_euclidean_bin_width( Real width )
{
	//std::fill( euclidean_bin_widths_.begin(), euclidean_bin_widths_.end(), width );
	euclidean_bin_widths_ = width;
}

void Matcher::set_hash_euler_bin_width( Real width )
{
	//std::fill( euler_bin_widths_.begin(), euler_bin_widths_.end(), width );
	euler_bin_widths_ = width;
}

void Matcher::set_hash_euclidean_bin_widths( Vector widths )
{
	euclidean_bin_widths_ = widths;
}

void Matcher::set_hash_euler_bin_widths( Vector widths)
{
	euler_bin_widths_ = widths;
}

void Matcher::set_bump_tolerance( Real permitted_overlap )
{
	runtime_assert( upstream_pose_.get() );

	if ( ! bb_grid_ ) {
		bb_grid_ = bump_grid_to_enclose_pose( *upstream_pose_ );
	}
	bb_grid_->set_general_overlap_tolerance( permitted_overlap );
}

void
Matcher::initialize_from_task(
	MatcherTask const & task
)
{
	set_upstream_pose( *task.upstream_pose() );
	set_downstream_pose( *task.downstream_pose(), task.downstream_orientation_atoms() );
	set_bump_tolerance( task.permitted_overlap() );
	set_occupied_space_bounding_box( task.occ_space_bounding_box() );
	set_hash_euclidean_bin_widths( task.euclidean_bin_widths() );
	set_hash_euler_bin_widths( task.euler_bin_widths() );

	/// active site definition
	read_gridlig_file_ = task.gridlig_active_site_definition();
	if ( read_gridlig_file_ ) {
		gridlig_fname_ = task.gridlig_file_name();
		upstream_resids_and_radii_defining_active_site_.clear();
	} else {
		upstream_resids_and_radii_defining_active_site_ = task.upstream_resids_and_radii_defining_active_site();
	}

	downstream_atoms_required_inside_active_site_ = task.downstream_atoms_required_inside_active_site();

	use_input_sc_ = task.use_input_sc();

	initialize_from_file( * task.enz_input_data() );

	for ( Size ii = 1; ii <= n_geometric_constraints_; ++ii ) {
		set_original_scaffold_build_points_for_constraint(
			ii, task.upstream_pose_build_resids_for_geometric_constraint( ii ) );
	}

	/// Should we use the match_dspos1 output pathway?
	/// TEMP -- No valid MatchEvaluator yet exists for the match_dspos1 struct; do not use
	/// the match-by-single-downstream-positioning code until one comes online.
	if ( ! task.consolidate_matches() && task.define_match_by_single_downstream_positioning() ) {
		output_matches_as_singular_downstream_positioning_ = true;
		output_match_dspos1_for_geomcst_.resize( n_geometric_constraints_, false );
		for ( Size ii = 1; ii <= task.geom_csts_downstream_output().size(); ++ii ) {
			output_match_dspos1_for_geomcst_[ task.geom_csts_downstream_output()[ ii ] ] = true;
		}
	}
}

/// @details Inside the CST::BEGIN blocks, The following ALGORITHM_INFO:: match input data
/// may be provided to give additional data to the matcher.  Rotamer building
/// instructions for particular geometric constraints may be given.
/// Here is an example of the kinds of geometric data.  Each CHI_STRATEGY line is appropriate
/// on it's own, but they are not appropriate together.  Lines beginning with "#" are comments
/// describing the meaning of each of the lines.
///
///    ALGORITHM_INFO:: match
///       # If your upstream residue includes a proton chi (e.g. SER/THR/TYR) but the geometry
///       # of the downstream partner does not depend on the location of the proton, then
///       # use the following line to avoid enumerating rotamers that differ only in their proton coordinate.
///       IGNORE_UPSTREAM_PROTON_CHI
///
///       # Secondary Matching:
///       # You can activate secondary matching, a continuous version of the classic discrete algorithm,
///       # by adding the line
///       SECONDARY_MATCH: DOWNSTREAM
///
///       #or
///       SECONDARY_MATCH: UPSTREAM_CST 2
///
///       # which instead of building its own hits and hashing them examines the hits generated in
///       # previous rounds for compatibility with the geometry that you're seeking.
///       # When performing secondary matching, it is not required that you specify all 6 degrees
///       # of freedom.  If you specify only a subset the geometry of only the subset you've specified
///       # will be examined (e.g. if you don't care about torsion_AB, don't specify it
///       # in the CST::BEGIN/CST::END block.
///       # You can perform secondary matching to the ligand (the downstream target) or to an upstream
///       # residue whose geometry was constructed by an earlier geometric constraint.  In the example
///       # above, the geometric constraint pointed to is #2.
///       # The first geometric constraint cannot use secondary matching, it must always use the discrete
///       # classic match algorithm.  (A later geometric constraint may of course perform secondary
///       # matching to the hits produced by the first geometric constraint.  At that point the first
///       # constraint is playing the same role in upstream matching as the ligand plays in downstream matching)
///       # END Secondary Matching comments.
///
///       # Below: chi sample strategies -- "1" is for chi-1
///       # These are the traditional ex?::level options ( ? == 1, 2, 3 or 4 )
///       CHI_STRATEGY:: CHI 1 EX_ONE_STDDEV
///       CHI_STRATEGY:: CHI 1 EX_ONE_HALF_STEP_STDDEV
///       CHI_STRATEGY:: CHI 1 EX_TWO_FULL_STEP_STDDEVS
///       CHI_STRATEGY:: CHI 1 EX_TWO_HALF_STEP_STDDEVS
///       CHI_STRATEGY:: CHI 1 EX_FOUR_HALF_STEP_STDDEVS
///       CHI_STRATEGY:: CHI 1 EX_THREE_THIRD_STEP_STDDEVS
///       CHI_STRATEGY:: CHI 1 EX_SIX_QUARTER_STEP_STDDEVS
///
///       # Below: The "AA" field, followed by the 3-letter AA code gives the sub-specification for
///       # a block that contains multiple amino acids.  If this block allowed both an ASN and a GLN, then
///       # the first line would apply to GLN rotamers only and the second line to ASN rotamers only.
///       # "AA" fields may be included with
///       # any of the CHI_STRATEGY options listed here.
///       CHI_STRATEGY:: AA GLN CHI 2 EX_SIX_QUARTER_STEP_STDDEVS
///       CHI_STRATEGY:: AA ASN CHI 2 EX_ONE_STDDEV
///
///       # Everything below: additional chi sample strategies unique to the matcher.
///
///       CHI_STRATEGY:: CHI 1 STEP_WITHIN_SD_RANGE STEP 3.0
///       # the above line says "step by 3 degrees within a single standard deviation
///       # to the left and right of the mean for chi 1"
///
///       CHI_STRATEGY:: CHI 1 STEP_WITHIN_SD_RANGE STEP 3.0 SD_RANGE 2.5
///       # the above line says "step by 3 degrees within 2.5 standard deviations to
///       # the left and the right of the mean for chi 1"
///
///       CHI_STRATEGY:: CHI 2 AA HIS NON_ROTAMERIC_CHI_EXPANSION N_SAMPLES 3 REQUISIT_PROBABILITY 0.10
///       # the line above says that "for histidine chi 2 (which is non-rotameric in the 2008 Dunbrak library)
///       # take 3 extra samples to the left and right of the median chi value inside the non-rotameric chi well
///       # for rotamer wells that have a probability at least 0.1".  The 2008 Dunbrack library must be active.
///       # (the -dun08 flag should be on the command line).
///
///       CHI_STRATEGY:: CHI 2 AA HIS NON_ROTAMERIC_CHI_EXPANSION N_SAMPLES 3
///       # the line above is similar for the previous command, but leaves off the REQUISIT_PROBABILITY
///       # which sets a default REQUISIT_PROBABILITY of 2 / #chi-bins.  For HIS, this is
///       # 2 / 36th, since there are 12 bins for chi 2 and 3 bins for chi1 (36 chi bins total).
///
///
///   ALGORITHM_INFO::END
///
void Matcher::initialize_from_file(
	enzdes::EnzConstraintIO const & enz_data
)
{
	using namespace enzdes;

	runtime_assert( upstream_pose_ );
	runtime_assert( downstream_pose_ );
	TR << "Matcher::initialize_from_file: n geometric constraints: " << enz_data.mcfi_lists_size() << std::endl;

	set_n_geometric_constraints( enz_data.mcfi_lists_size() );

	for ( Size ii = 1; ii <= n_geometric_constraints_; ++ii ) {
		TR << "Begin constraint" << ii << std::endl;

		MatchConstraintFileInfoListCOP constraint_list = enz_data.mcfi_list( ii );

		utility::vector1< core::chemical::ResidueTypeCAP > const & upres( constraint_list->upstream_restypes() );

		for ( Size jj = 1; jj <= upres.size(); ++jj ) {
			add_upstream_restype_for_constraint( ii, upres[ jj ] );

			utility::vector1< MatchConstraintFileInfoCOP > const & jj_mcfis(
				constraint_list->mcfis_for_upstream_restype( upres[ jj ] ));
			TR << " Upstream residue type " << upres[ jj ]->name() << " for geometric constraint #" << ii << std::endl;

			for ( Size kk = 1; kk <= jj_mcfis.size(); ++kk ) {
				bool secondary_matching( false );
				bool secondary_match_upstream_residue( false );
				Size secondary_match_upstream_geomcst_id( 0 );

				utility::vector1< bool > chi_sample_data_in_file( upres[ jj ]->nchi(), false );
				//utility::vector1< SampleStrategyData > file_chi_sample_data( upres[ jj ]->nchi() );

				/// Process ALGORITHM_INFO data
				std::map< std::string, utility::vector1< std::string > > const &
					alg_info( jj_mcfis[ kk ]->algorithm_inputs() );
				if ( alg_info.find( "match" ) != alg_info.end() ) {
					utility::vector1< std::string > const & info( alg_info.find( "match" )->second );

					/// Line by line.  Currently, there are no checks for consistency across multiple lines.
					for ( Size ll = 1; ll <= info.size(); ++ll ) {
						std::string llstr = info[ ll ];
						std::istringstream llstream( llstr );
						std::string first;
						llstream >> first;
						if ( first == "IGNORE_UPSTREAM_PROTON_CHI" ) {
							/// iterate across the proton chi, set their sample strategy to no_samples.
							upstream::SampleStrategyData nosamps; nosamps.set_strategy( upstream::no_samples );
							for ( Size mm = 1; mm <= upres[ jj ]->n_proton_chi(); ++mm ) {
								Size mmchi = upres[ jj ]->proton_chi_2_chi( mm );
								if ( chi_sample_data_in_file[ mmchi ] ) {
									TR << "  WARNING:: Already encountered chi sampling strategy for proton chi " << mmchi << " and will NOT ignore this proton chi" << std::endl;
								} else {
									set_sample_startegy_for_constraint(	ii, upres[ jj ], mmchi, nosamps );
									TR << "  ALGORITHM_INFO:: match -- Ignoring proton chi for " << upres[ jj ]->name() << " chi # " << mmchi << std::endl;
									chi_sample_data_in_file[ mmchi ] = true;
								}
							}
						} else if ( first == "CHI_STRATEGY::" ) {
							std::string chi_string;
							llstream >> chi_string;
							if ( llstream.bad() ) {
								utility_exit_with_message( "Expected 'CHI' or 'AA' following CHI_STRATEGY:: on line '" + llstr + "'" );
							}
							if ( chi_string == "AA" ) {
								std::string aa3;
								llstream >> aa3;
								//std::map< std::string, core::chemical::AA >::const_iterator iter = core::chemical::name2aa().find( aa3 );
								//if ( iter == core::chemical::name2aa().end() ) {
								//	utility_exit_with_message( "Expected amino acid 3-letter code following 'CHI_STRATEGY:: AA ' but read " + aa3 );
								//}
								//core::chemical::AA aa = *iter;
								core::chemical::AA aa = core::chemical::aa_from_name( aa3 );
								bool aa_matches_any = false;
								for ( Size mm = 1; mm <= upres.size(); ++mm ) {
									if ( upres[ mm ]->aa() == aa ) {
										aa_matches_any = true;
										break;
									}
								}
								if ( ! aa_matches_any ) {
									std::cerr << "ERROR: amino acid " << aa3 << " on line\n" << llstr << "\nis not accessible for this geometric constraint." << std::endl;
									std::cerr << "Available amino acids:";
									for ( Size mm = 1; mm <= upres.size(); ++mm ) {
										std::cerr << " " << upres[ mm ]->name();
									}
									std::cerr << std::endl;
									utility_exit_with_message( "Amino acid restriction in CHI_STRATEGY:: block is invalid" );
								}
								if ( aa != upres[ jj ]->aa() ) {
									//TR << "  Ignoring line '" << llstr << "' in processing amino acid " << upres[ jj ]->name() << std::endl;
									continue;
								}

								llstream >> chi_string;
							}

							if ( chi_string != "CHI" ) {
								utility_exit_with_message( "Expected 'CHI' following CHI_STRATEGY:: on line '" + llstr + "'" );
							}
							Size which_chi;
							llstream >> which_chi;
							if ( chi_sample_data_in_file[ which_chi ] ) {
								TR << "  WARNING:: Repeat chi info for chi " << which_chi << " is being ignored!" << std::endl;
								TR << "  WARNING:: Ignoring: '" << llstr << "'" << std::endl;
								continue;
							}
							chi_sample_data_in_file[ which_chi ] = true;
							if ( llstream.bad() ) {
								utility_exit_with_message( "Error parsing CHI_STRATEGY.  Expected an "
									"integer following 'CHI_STRATEGY::' on line '" + llstr );
							}
							if ( which_chi > upres[ jj ]->nchi() ) {
								TR <<  "WARNING: Ignoring rotamer sampling strategy data for chi # "
									<< which_chi
									<< " for residue type " << upres[ jj ]->name() << std::endl;
								continue;
							}
							std::string strategy;
							llstream >> strategy;
							if ( core::pack::task::is_rot_sample_name( strategy ) ) {
								core::pack::task::ExtraRotSample sample_level =  core::pack::task::rot_sample_from_name( strategy );
								upstream::SampleStrategyData stratdat; stratdat.set_strategy( upstream::rotameric_chi_mimic_EX_flags );
								stratdat.set_sample_level( sample_level );
								set_sample_startegy_for_constraint(	ii, upres[ jj ], which_chi, stratdat );

								TR << "  ALGORITHM_INFO:: match -- chi sampling strategy " << strategy << " for chi # " << which_chi << std::endl;

							} else if ( strategy == "STEP_WITHIN_SD_RANGE" ) {
								/// PARSE SD_RANGE
								Real step_size( 1.0 ), sd_range( 1.0 );
								std::string step, range;
								llstream >> step;
								if ( step != "STEP" ) {
									std::cerr << "ERROR:: Bad line in CHI_STRATEGY: \"" << llstr << "\"" << std::endl;
									utility_exit_with_message( "While parsing CHI_STRATEGY:: STEP_WITHIN_SD_RANGE, expected \"STEP\" after \"STEP_WITHIN_SD_RANGE\"" );
								}
								if ( llstream.bad() ) {
									std::cerr << "ERROR:: Bad line in CHI_STRATEGY: \"" << llstr << "\"" << std::endl;
									utility_exit_with_message( "While parsing CHI_STRATEGY:: STEP_WITHIN_SD_RANGE, unexpected EOF" );
								}
								llstream >> step_size;
								if ( llstream.bad() ) {
									std::cerr << "ERROR:: Bad line in CHI_STRATEGY: \"" << llstr << "\"" << std::endl;
									utility_exit_with_message( "While parsing CHI_STRATEGY:: STEP_WITHIN_SD_RANGE.  Could not read step size" );
								}
								if ( step_size == 0.0 ) {
									std::cerr << "ERROR:: Bad line in CHI_STRATEGY: \"" << llstr << "\"" << std::endl;
									utility_exit_with_message( "While parsing CHI_STRATEGY:: STEP_WITHIN_SD_RANGE.  Invalid step size of 0.0" );
								}
								llstream >> range;
								if ( range != "" ) {
									if ( range != "SD_RANGE" ) {
										std::cerr << "ERROR:: Bad line in CHI_STRATEGY: \"" << llstr << "\"" << std::endl;
										utility_exit_with_message( "While parsing CHI_STRATEGY:: STEP_WITHIN_SD_RANGE.  Expected to read SD_RANGE to specify the standard deviation range" );
									}
									llstream >> sd_range;
									if ( llstream.bad() ) {
										std::cerr << "ERROR:: Bad line in CHI_STRATEGY: \"" << llstr << "\"" << std::endl;
										utility_exit_with_message( "While parsing CHI_STRATEGY:: STEP_WITHIN_SD_RANGE.  Could not read standard-deviation range" );
									}
									if ( sd_range <= 0.0 ) {
										std::cerr << "ERROR:: Bad line in CHI_STRATEGY: \"" << llstr << "\"" << std::endl;
										utility_exit_with_message( "While parsing CHI_STRATEGY:: STEP_WITHIN_SD_RANGE.  Invalid standard-deviation range (must be positive). Read: " + utility::to_string( sd_range ) );
									}
								} else {
									/// Implicit sd_range of 1.0
									sd_range = 1.0;
								}
								upstream::SampleStrategyData stratdat;
								stratdat.set_strategy( upstream::rotameric_chi_step_wi_sd_range );
								stratdat.set_step_size( step_size );
								stratdat.set_sd_range( sd_range );
								set_sample_startegy_for_constraint(	ii, upres[ jj ], which_chi, stratdat );

								TR << "  ALGORITHM_INFO:: match -- chi sampling strategy STEP_WITHIN_SD_RANGE with step_size= " << step_size << " degrees across " << sd_range << " standard deviations for chi # " << which_chi << std::endl;
							} else if ( strategy == "NON_ROTAMERIC_CHI_EXPANSION" ) {
								Size npossiblerots( 0 );
								/// Expand the number of rotamers for a non-rotameric chi.
								/// 1st check to make sure that this chi is nonrotameric!
								{/// SCOPE
								using namespace core::scoring;
								using namespace core::scoring::dunbrack;
								RotamerLibrary const & rotlib( ScoringManager::get_instance()->get_RotamerLibrary() );
								SingleResidueRotamerLibraryCAP res_rotlib( rotlib.get_rsd_library( *upres[ jj ] ) );

								if ( res_rotlib != 0 ) {

									SingleResidueDunbrackLibraryCAP dun_rotlib(
										dynamic_cast< SingleResidueDunbrackLibrary const * >
										( res_rotlib.get() ));

									if ( dun_rotlib == 0 ) {
										utility_exit_with_message( "Failed to retrieve a Dunbrack rotamer library for AA: " +
											utility::to_string( upres[ jj ]->aa() ) +  " named " +  upres[ jj ]->name() );
									}
									npossiblerots = dun_rotlib->n_rotamer_bins();
									if ( npossiblerots == 0 ) {
										std::cerr << "Error while reading line: " << llstr << std::endl;
										utility_exit_with_message( "Rotamer library for " + upres[ jj ]->name() + " says it contains no rotamers" );
									}
									if ( which_chi != dun_rotlib->nchi() ) {
										utility_exit_with_message( "Cannot treat chi " + utility::to_string( which_chi ) +
											" on residue " + upres[ jj ]->name() +
											" as non-rotameric since there are " + utility::to_string( dun_rotlib->nchi() ) +
											" chi in the library, and the last chi is the only one that could be non-rotameric" );
									}
									bool failed_cast = false;
									switch ( dun_rotlib->nchi() ) {
										case 2: {
											SemiRotamericSingleResidueDunbrackLibrary< ONE > const * sr2 =
												dynamic_cast< SemiRotamericSingleResidueDunbrackLibrary< ONE > const * >
												( dun_rotlib.get() );
											failed_cast = sr2 == 0;
										} break;
										case 3: {
											SemiRotamericSingleResidueDunbrackLibrary< TWO > const * sr3 =
												dynamic_cast< SemiRotamericSingleResidueDunbrackLibrary< TWO > const * >
												( dun_rotlib.get() );
											failed_cast = sr3 == 0;
										} break;
										default: {
											utility_exit_with_message( "While parsing CHI_STRATEGY::NON_ROTAMERIC_CHI_EXPANSION\n"
												"All semi-rotameric libraries have 2 or 3 chi, but the library for "+
												upres[ jj ]->name() + " has " + utility::to_string( dun_rotlib->nchi() ) + " chi." );
										}
									}
									if ( failed_cast ) {
										utility_exit_with_message( "While parsing CHI_STRATEGY::NON_ROTAMERIC_CHI_EXPANSION\n"
											"Failed to find a semi-rotameric rotamer library for " + upres[ jj ]->name() +
											" (Did you forget the -dun08 flag?)\n"
											"The following amino acids define semi-rotameric rotamer libraries: DEFHNQ" );
									}
								} else {
									utility_exit_with_message( "While parsing CHI_STRATEGY::NON_ROTAMERIC_CHI_EXPANSION\n"
										"Failed to find a rotamer library for " + upres[ jj ]->name() );
								}
								} // scope to check we're looking at a semi-rotameric rotamer library.
								std::string nsamps;
								Size nsamples;
								if ( ! llstream.good() ) {
									std::cerr << "Error reading line: " << llstr << std::endl;
									utility_exit_with_message( "While parsing CHI_STRATEGY::NON_ROTAMERIC_CHI_EXPANSION\n"
										"Expected to read N_SAMPLES after NON_ROTAMERIC_CHI_EXPANSIONbut reached an end of line");
								}
								llstream >> nsamps;
								if ( nsamps != "N_SAMPLES" ) {
									std::cerr << "Error reading line: " << llstr << std::endl;
									utility_exit_with_message( "While parsing CHI_STRATEGY::NON_ROTAMERIC_CHI_EXPANSION\n"
										"Expected to read N_SAMPLES after NON_ROTAMERIC_CHI_EXPANSION but found '" + nsamps + "'");
								}
								llstream >> nsamples;
								if ( !llstream ) {
									std::cerr << "Error reading line: " << llstr << std::endl;
									utility_exit_with_message( "While parsing CHI_STRATEGY::NON_ROTAMERIC_CHI_EXPANSION\n"
										"Expected to read an integer following N_SAMPLES but could not" );
								}
								std::string minprob;
								Real minprobability( 2.0 / npossiblerots );
								llstream >> minprob;
								if ( minprob != "" ) {
									if ( minprob != "REQUISIT_PROBABILITY" ) {
										std::cerr << "Error reading line: " << llstr << std::endl;
										utility_exit_with_message( "While parsing CHI_STRATEGY::NON_ROTAMERIC_CHI_EXPANSION\n"
											"Expected end-of-line or 'REQUISIT_PROBABILITY' after reading the number of samples, but found '" +  minprob + "'" );
									}
									if ( ! llstream.good() ) {
										std::cerr << "Error reading line: " << llstr << std::endl;
										utility_exit_with_message( "While parsing CHI_STRATEGY::NON_ROTAMERIC_CHI_EXPANSION\n"
											"Expected to read a probability following the 'REQUISIT_PROBABILITY' string but found an end-of-line" );
									}
									llstream >> minprobability;
									if ( llstream.bad() ) {
										std::cerr << "Error reading line: " << llstr << std::endl;
										utility_exit_with_message( "While parsing CHI_STRATEGY::NON_ROTAMERIC_CHI_EXPANSION\n"
											"Expected to read a probability following the 'REQUISIT_PROBABILITY' string but could not" );
									}
								}
								// OK -- Now set the chi-sample strategy.
								upstream::SampleStrategyData stratdat;
								stratdat.set_strategy( upstream::nonrotameric_chi_sample_wi_nrchi_bin );
								stratdat.set_n_samples_per_side_of_nrchi_bin( nsamples );
								stratdat.set_nrchi_prob_minimum_for_extra_samples( minprobability );
								set_sample_startegy_for_constraint(	ii, upres[ jj ], which_chi, stratdat );
								TR << "  ALGORITHM_INFO:: match -- chi sampling strategy NON_ROTAMERIC_CHI_EXPANSION with nsteps= " << nsamples << " for rotamers with a probability better than " << minprobability << std::endl;
							} else {
								utility_exit_with_message( "While parsing CHI_STRATEGY:: unsupported sample strategy: " + strategy  + " for chi " + utility::to_string( which_chi ) );
							}
						} else if ( first == "SECONDARY_MATCH:" ) {
							if ( ii == 1 ) {
								std::cerr << "ERROR Reading line " << llstr << " " << " for geometric constraint " << ii << std::endl;
								utility_exit_with_message( "Seconary matching cannot be chosen for the first geometric constraint!" );
							}
							if ( ! llstream ) {
								utility_exit_with_message( "While parsing SECONDARY_MATCH: line, exptected to read 'UPSTREAM_CST <int>' but reached an end-of-line" );
							}
							std::string second;
							llstream >> second;
							if ( second != "UPSTREAM_CST" && second != "DOWNSTREAM" ) {
								std::cerr << "Error reading line: " << llstr << std::endl;
								utility_exit_with_message( "While parsing SECONDARY_MATCH: line, expected 'UPSTREAM_CST' or 'DOWNSTREAM' but encountered '" + second +"'." );
							}
							if ( second == "UPSTREAM_CST" ) {
								Size cst_id;
								llstream >> cst_id;
								if ( ! llstream ) {
									std::cerr << "Error reading line: " << llstr << std::endl;
									utility_exit_with_message( "While parsing SECONDARY_MATCH: line, read 'UPSTREAM_CST' and expected to read an integer following, but did not find one." );
								}
								if ( cst_id < 1 || cst_id >= ii ) {
									std::cerr << "Error reading line: '" << llstr << "' for geometric constraint " << ii << std::endl;
									utility_exit_with_message( "Secondary match algorithm requested to an upstream residue "
										"produced by an invalid geometric constraint " + utility::to_string( cst_id ) +
										"\nThe geometric constraint ID must be less than the current geometric constraint id"
										"and greater than 0" );
								}
								secondary_match_upstream_residue = true;
								secondary_match_upstream_geomcst_id = cst_id;
							}
							secondary_matching = true;
						} else {
							utility_exit_with_message( "While parsing ALGORITHM:: match data. Command '" + first +"' not supported on line '" + llstr + "'");
						}
					}
				} else {
					TR << "  Did not locate ALGORITHM_INFO:: match for constraint " << ii << std::endl;
				}


				Size const upstream_id = jj_mcfis[ kk ]->upstream_res();
				Size const downstream_id = jj_mcfis[ kk ]->downstream_res();

				downstream::ExternalGeomSamplerCOP exgs;
				if ( !secondary_matching ) {
					exgs = jj_mcfis[ kk ]->create_exgs();
					if ( exgs.get() == 0 ) {
						utility_exit_with_message( "ERROR: could not define external geometry between upstream and downstream residues.  All 6 parameters must be defined." );
					}
				}

				if ( jj_mcfis[ kk ]->allowed_restypes( downstream_id ).size() == 0 ) {
					utility_exit_with_message( "Input file lists no residue types for template residue " +
						utility::to_string( downstream_id ) + " for geometric constraint " +
						utility::to_string( ii ) + ".  There must be at least one." );
				}

				for ( Size ll = 1; ll <= jj_mcfis[ kk ]->allowed_restypes( downstream_id ).size(); ++ll ) {

					core::chemical::ResidueTypeCAP ll_downres( jj_mcfis[ kk ]->allowed_restypes( downstream_id )[ ll ] );

					utility::vector1< std::string > upstream_launch_atoms( 3 );
					//utility::vector1< std::string > downstream_launch_atoms( 3 ); /// TEMP HACK.  Assume single ligand downstream
					utility::vector1< core::id::AtomID > downstream_3atoms( 3 );
					for ( Size mm = 1; mm <= 3; ++mm ) downstream_3atoms[ mm ].rsd() = 1;


					utility::vector1< utility::vector1< Size > > up_ats( 3 ), down_ats( 3 );
					for ( Size mm = 1; mm <= 3; ++mm ) up_ats[ mm ] = jj_mcfis[ kk ]->template_atom_inds( upstream_id, mm, *upres[ jj ] );
					for ( Size mm = 1; mm <= 3; ++mm ) down_ats[ mm ] = jj_mcfis[ kk ]->template_atom_inds( downstream_id, mm, *ll_downres );

					runtime_assert( up_ats[ 1 ].size() == up_ats[ 2 ].size() );
					runtime_assert( up_ats[ 1 ].size() == up_ats[ 3 ].size() );

					utility::vector1< Size > up_down_ncombs( 2 );
					up_down_ncombs[ 1 ] = up_ats[ 1 ].size();
					up_down_ncombs[ 2 ] = down_ats[ 2 ].size();

					utility::LexicographicalIterator lex( up_down_ncombs );

					while ( ! lex.at_end() ) {
						for ( Size mm = 1; mm <= 3; ++mm ) upstream_launch_atoms[ mm ] = upres[ jj ]->atom_name( up_ats[ mm ][ lex[ 1 ] ] );
						for ( Size mm = 1; mm <= 3; ++mm ) downstream_3atoms[ mm ].atomno() = down_ats[ mm ][ lex[ 2 ] ];

						TR << "   " << upres[ jj ]->name() << " " << ll_downres->name() << std::endl;
						TR << "   ";
						TR << " U3: " << upstream_launch_atoms[ 3 ];
						TR << " U2: " << upstream_launch_atoms[ 2 ];
						TR << " U1: " << upstream_launch_atoms[ 1 ];
						TR << " D1: " << ll_downres->atom_name( down_ats[ 1 ][ lex[ 2 ] ] );
						TR << " D2: " << ll_downres->atom_name( down_ats[ 2 ][ lex[ 2 ] ] );
						TR << " D3: " << ll_downres->atom_name( down_ats[ 3 ][ lex[ 2 ] ] );
						TR << std::endl;

						if ( secondary_matching ) {
							utility::vector1< Size > candidate_atids( 3 );
							utility::vector1< Size > target_atids( 3 );
							for ( Size nn = 1; nn <= 3; ++nn ) {
								candidate_atids[ nn ] = up_ats[ nn ][ lex[ 1 ]];
								target_atids[ nn ]    = down_ats[ nn ][ lex[ 2 ]];
							}
							if ( secondary_match_upstream_residue ) {
								add_secondary_upstream_match_geometry_for_constraint(
									ii, secondary_match_upstream_geomcst_id, upres[ jj ],
									ll_downres, candidate_atids, target_atids,
									jj_mcfis[ kk ] );
							} else {
								add_secondary_downstream_match_geometry_for_constraint(
									ii, upres[ jj ], ll_downres,
									candidate_atids, target_atids, jj_mcfis[ kk ] );
							}
						} else {
							add_external_geometry_samples_for_constraint(
								ii, upres[ jj ], upstream_launch_atoms, downstream_3atoms, *exgs,
								jj_mcfis[ kk ]->index(), enz_data.enz_cst_params( ii )->is_covalent() );
						}
						++lex;
					} //lex loop
				} // ll loop over downstream residue types
			} //kk loop over mcfis
		} //jj loop over restypes for constraint
	} //ii loop over geometric constraints
}


/// @brief Main worker function
void Matcher::find_hits()
{
	initialize_scaffold_build_points();
	initialize_bump_grids();
	initialize_active_site_grid();
	initialize_occupied_space_hash();
	initialize_downstream_algorithms();

	generate_hits();
}


void
Matcher::process_matches( output::MatchProcessor & processor ) const
{
	processor.begin_processing();
	for ( Size ii = 1; ii <= n_geometric_constraints_; ++ii ) {
		representative_downstream_algorithm_[ ii ]->prepare_for_match_enumeration( *this );
	}
	if ( ! output_matches_as_singular_downstream_positioning_ ) {
		process_matches_main_loop_enumerating_all_hit_combos( processor );
	} else {
		process_matches_where_one_geomcst_defines_downstream_location( processor );
	}
	processor.end_processing();
}

/*utility::vector1< std::list< Hit > > const &
Matcher::hits() const
{
	return hits_;
}
*/

upstream::ScaffoldBuildPointCOP
Matcher::build_point( Size index ) const
{
	return all_build_points_[ index ];
}

upstream::UpstreamBuilderCOP
Matcher::upstream_builder( Size cst_id ) const
{
	return upstream_builders_[ cst_id ];
}

downstream::DownstreamBuilderCOP
Matcher::downstream_builder( Size geom_cst ) const
{
	if ( downstream_builders_[ geom_cst ].empty() ) {
		return 0;
	} else {
		return *(downstream_builders_[ geom_cst ].begin());
	}
}

std::list< downstream::DownstreamAlgorithmCOP >
Matcher::downstream_algorithms( Size cst_id ) const
{
	std::list< downstream::DownstreamAlgorithmCOP > dsalgs;
	for ( std::list< downstream::DownstreamAlgorithmOP >::const_iterator
			iter = downstream_algorithms_[ cst_id ].begin(),
			iter_end = downstream_algorithms_[ cst_id ].end();
			iter != iter_end; ++iter ) {
		dsalgs.push_back( *iter );
	}
	return dsalgs;
}

downstream::DownstreamAlgorithmCOP
Matcher::representative_downstream_algorithm( Size cst_id ) const
{
	return representative_downstream_algorithm_[ cst_id ];
}


Matcher::HitList const &
Matcher::hits( Size cst_id ) const
{
	return hits_[ cst_id ];
}


OccupiedSpaceHashCOP
Matcher::occ_space_hash() const {
	return occ_space_hash_;
}

utility::vector1< upstream::ScaffoldBuildPointCOP > const &
Matcher::per_constraint_build_points( Size cst_id ) const
{
	return per_constraint_build_points_[ cst_id ];
}


upstream::ScaffoldBuildPointOP
Matcher::build_point( Size index )
{
	return all_build_points_[ index ];
}

upstream::UpstreamBuilderOP
Matcher::upstream_builder( Size cst_id )
{
	return upstream_builders_[ cst_id ];
}

downstream::DownstreamBuilderOP
Matcher::downstream_builder( Size cst_id )
{
	runtime_assert( ! downstream_builders_[ cst_id ].empty() );
	return *(downstream_builders_[ cst_id ].begin());
}

std::list< downstream::DownstreamBuilderOP > const &
Matcher::downstream_builders( Size cst_id ) const
{
	return downstream_builders_[ cst_id ];
}

std::list< downstream::DownstreamAlgorithmOP > const &
Matcher::nonconst_downstream_algorithms( Size cst_id )
{
	return downstream_algorithms_[ cst_id ];
}


OccupiedSpaceHashOP
Matcher::occ_space_hash() {
	return occ_space_hash_;
}

Matcher::HitListIterator
Matcher::hit_list_begin( Size geom_cst_id )
{
	return hits_[ geom_cst_id ].begin();
}

Matcher::HitListIterator
Matcher::hit_list_end( Size geom_cst_id )
{
	return hits_[ geom_cst_id ].end();
}

void
Matcher::erase_hit(
	downstream::DownstreamAlgorithm const & dsalg,
	Size geom_cst_id_for_hit,
	HitListIterator const & iter
)
{
	hits_[ geom_cst_id_for_hit ].erase( iter );
	if ( geom_cst_id_for_hit != dsalg.geom_cst_id() ) {
		/// A downstream algorithm is making a primary modification to
		/// the hit list for some other geometric constraint.

		// ASSUMPTION: hits for geom_cst i may not depend on hits for geom_cst j, for j > i.
		// Under this assumption, the downstream algorithm for geom_cst i is unable to
		// direct the deletion of hits from geom_cst j.  Maybe the downstream algorithm
		// for geom_cst j deletes its own hits during a call to respond_to_peripheral_hitlist_change,
		// as a result of round i hits disappearing; however, round i may not delete round j's hits
		// directly.
 		runtime_assert( dsalg.geom_cst_id() > geom_cst_id_for_hit );

		note_primary_change_to_geom_csts_hitlist( geom_cst_id_for_hit );
	}
}


void Matcher::generate_hits() {
	for ( Size ii = 1; ii <= n_geometric_constraints_; ++ii ) {
		prepare_for_hit_generation_for_constraint( ii );
		generate_hits_for_constraint( ii );
		finish_hit_generation_for_constraint( ii );
	}
}

void Matcher::prepare_for_hit_generation_for_constraint( Size cst_id )
{

	if ( same_build_resids_for_all_csts_ ) {
		per_constraint_build_points_[ cst_id ].reserve( all_build_points_.size() );
		for ( Size ii = 1; ii <= all_build_points_.size(); ++ii ) {
			/// if ( logic ) goes here to decide if all build points
			/// are apropriate for a pariticular geometric constraint.
			per_constraint_build_points_[ cst_id ].push_back( all_build_points_[ ii ] );
		}
	} else {
		per_constraint_build_points_[ cst_id ].reserve( per_cst_build_resids_[ cst_id ].size() );

		/// "merge sort" inspired algorithm; advance through two sorted lists.
		Size counter = 1;
		for ( Size ii = 1; ii <= pose_build_resids_.size(); ++ii ) {
			if ( per_cst_build_resids_[ cst_id ][ counter ] == pose_build_resids_[ ii ] ) {
				per_constraint_build_points_[ cst_id ].push_back( all_build_points_[ ii ] );
				++counter;
				if ( counter > per_cst_build_resids_[ cst_id ].size() ) break;
			}
		}
	}

}

void Matcher::generate_hits_for_constraint( Size cst_id )
{

	/* Putting this here temporarily
	if ( cst_id != 1 ) {
		/// Greedy matching: only accept hits that could lead to a match.
		/// Greedy matching begins after round 1.
		for ( std::list< downstream::DownstreamBuilderOP >::const_iterator
				iter = all_downstream_builders_.begin(),
				iter_end = all_downstream_builders_.end();
				iter != iter_end; ++iter ) {
			(*iter)->set_occupied_space_hash( occ_space_hash_ );
		}
		for ( std::list< DownstreamAlgorithmOP >::const_iterator
				iter = all_downstream_algorithms_.begin(),
				iter_end = all_downstream_algorithms_.end();
				iter != iter_end; ++iter ) {
			(*iter)->set_occupied_space_hash( occ_space_hash_ );
		}

	}*/

	/// At the conclusion of hit generation, there will be new hits for this geometric constraint;
	/// put this constraint ID at the front of the list of geom-csts to trigger primary-change responses.
	note_primary_change_to_geom_csts_hitlist( cst_id );

	std::list< Hit > hits = representative_downstream_algorithm_[ cst_id ]->build_hits_at_all_positions( *this );
	hits_[ cst_id ].splice( hits_[ cst_id ].end(), hits );

}

void
Matcher::finish_hit_generation_for_constraint( Size cst_id )
{

	/// During the primary and peripheral hit-list prunings, new elements may be pushed-back
	/// into the hit-lists-modified-by-primary-deletion list.
	for ( std::list< Size >::iterator iter = geom_csts_with_primary_hitlist_modificiations_.begin();
			iter != geom_csts_with_primary_hitlist_modificiations_.end(); /* no increment */ ) {

		representative_downstream_algorithm_[ *iter ]->respond_to_primary_hitlist_change( *this, cst_id );

		/// There is an implicit dependency DAG between geometric constraints:
		/// round i cannot be dependent on round j, if j > i.
		/// Therefore a very simple "topological sort" may be performed offline to come up with
		/// the appropriate order in which to update peripheral hits: go from 1 to n.
		for ( Size jj = 1; jj <= cst_id; ++jj ) {
			if ( jj == *iter ) continue;
			representative_downstream_algorithm_[ jj ]->respond_to_peripheral_hitlist_change( *this );
		}

		std::list< Size >::iterator iter_next( iter );
		++iter_next;
		geom_cst_has_primary_modification_[ *iter ] = false;
		geom_csts_with_primary_hitlist_modificiations_.erase( iter );
		iter = iter_next;
	}

	if ( cst_id == n_geometric_constraints_ ) {
		/// We're completely done with hit generation; clean up.
		/// Delete the occ_space_hash_
		for ( std::list< downstream::DownstreamBuilderOP >::const_iterator
				iter = all_downstream_builders_.begin(),
				iter_end = all_downstream_builders_.end();
				iter != iter_end; ++iter ) {
			(*iter)->set_occupied_space_hash( 0 );
		}
		occ_space_hash_ = 0;
	}
}

void
Matcher::initialize_scaffold_build_points()
{
	runtime_assert( upstream_pose_.get() );

	all_build_points_.resize( pose_build_resids_.size() );
	for ( Size ii = 1; ii <= pose_build_resids_.size(); ++ii ) {
		all_build_points_[ ii ] = new upstream::OriginalBackboneBuildPoint(
			upstream_pose_->residue( pose_build_resids_[ ii ] ), ii );
	}

}


void
Matcher::initialize_bump_grids()
{
	runtime_assert( upstream_pose_.get() );

	clock_t starttime = clock();
	TR << "Initializing BumpGrids... " << std::endl;

	if ( ! bb_grid_ ) {
		bb_grid_ = bump_grid_to_enclose_pose( *upstream_pose_ );
	}

	/// This code is currently fixed-backbone only... it needs to be expanded.
	original_scaffold_residue_bump_grids_.resize( upstream_pose_->total_residue() );
	for ( Size ii = 1; ii <= upstream_pose_->total_residue(); ++ii ) {
		BumpGridOP resbgop = bump_grid_to_enclose_residue_backbone( upstream_pose_->residue( ii ), *bb_grid_ );
		fill_grid_with_backbone_heavyatom_spheres( upstream_pose_->residue( ii ), *resbgop );
		bb_grid_->or_with( *resbgop );
		original_scaffold_residue_bump_grids_[ ii ] = resbgop;
	}


	// Inform everyone of the grid
	for ( std::list< downstream::DownstreamBuilderOP >::const_iterator
			iter = all_downstream_builders_.begin(),
			iter_end = all_downstream_builders_.end();
			iter != iter_end; ++iter ) {
		(*iter)->set_bb_grid( bb_grid_ );
	}
	for ( std::list< downstream::DownstreamAlgorithmOP >::const_iterator
			iter = all_downstream_algorithms_.begin(),
			iter_end = all_downstream_algorithms_.end();
			iter != iter_end; ++iter ) {
		(*iter)->set_bb_grid( bb_grid_ );
	}
	for ( utility::vector1< upstream::UpstreamBuilderOP >::const_iterator
			iter = upstream_builders_.begin(),
			iter_end = upstream_builders_.end();
			iter != iter_end; ++iter ) {
		(*iter)->set_bb_grid( bb_grid_ );
	}

	TR << "...done" << std::endl;
	clock_t stoptime = clock();
	TR << " TIMING: Bump grids took " << ((double) stoptime - starttime )/CLOCKS_PER_SEC << " seconds to compute" << std::endl;
}

void
Matcher::initialize_active_site_grid()
{
	/* TEMP
	if ( downstream_atoms_required_inside_active_site_.empty() ) return;

	if ( upstream_resids_and_radii_defining_active_site_.empty() ) {
		utility_exit_with_message( "ERROR: Active site undefined, yet downstream atoms are required to be in active site" );
	}*/

	if ( read_gridlig_file_ ) {
		active_site_grid_ = new downstream::ActiveSiteGrid;
		active_site_grid_->initialize_from_gridlig_file( gridlig_fname_ );

		/*Bool3DGridKinemageWriter writer;
		writer.set_write_facets( true );
		std::ofstream ostr( "active_site_gridlig.kin" );
		writer.set_line_color( "green" );
		writer.write_grid_to_kinemage( ostr, "act_site", active_site_grid_->grid() );*/
	} else {
		active_site_grid_ = new downstream::ActiveSiteGrid;
		active_site_grid_->set_bin_width( 0.25 ); /// Same resolution as the bump grid!

		for ( std::list< std::pair< Size, Real > >::const_iterator
				iter = upstream_resids_and_radii_defining_active_site_.begin(),
				iter_end = upstream_resids_and_radii_defining_active_site_.end();
				iter != iter_end; ++iter ) {
			runtime_assert( iter->first <= upstream_pose_->total_residue() );
			active_site_grid_->enlargen_to_capture_volume_within_radius_of_residue(
				upstream_pose_->residue( iter->first ), iter->second );
		}

		for ( std::list< std::pair< Size, Real > >::const_iterator
				iter = upstream_resids_and_radii_defining_active_site_.begin(),
				iter_end = upstream_resids_and_radii_defining_active_site_.end();
				iter != iter_end; ++iter ) {
			active_site_grid_->or_within_radius_of_residue(
				upstream_pose_->residue( iter->first ), iter->second );
		}

		// Inform everyone of the grid
		for ( std::list< downstream::DownstreamBuilderOP >::const_iterator
				iter = all_downstream_builders_.begin(),
				iter_end = all_downstream_builders_.end();
				iter != iter_end; ++iter ) {
			(*iter)->set_active_site_grid( active_site_grid_ );
		}
		for ( std::list< downstream::DownstreamAlgorithmOP >::const_iterator
				iter = all_downstream_algorithms_.begin(),
				iter_end = all_downstream_algorithms_.end();
				iter != iter_end; ++iter ) {
			(*iter)->set_active_site_grid( active_site_grid_ );
		}

		/// Inform dowsntream builders of their atoms that must be contained inside the
		/// active site grid.
		for ( std::list< downstream::DownstreamBuilderOP >::const_iterator
				iter = all_downstream_builders_.begin(),
				iter_end = all_downstream_builders_.end();
				iter != iter_end; ++iter ) {
			for ( std::list< core::id::AtomID >::const_iterator
					atid_iter = downstream_atoms_required_inside_active_site_.begin(),
					atid_iter_end = downstream_atoms_required_inside_active_site_.end();
					atid_iter != atid_iter_end; ++atid_iter ) {
				(*iter)->require_atom_to_reside_in_active_site( *atid_iter );
			}
		}

		/*std::cout << "Writing active-site kinemage" << std::endl;

		Bool3DGrid copy_active = active_site_grid_->grid();
		copy_active.subtract( bb_grid_->grid( C_ALA ) );

		Bool3DGridKinemageWriter writer;
		writer.set_write_facets( true );
		std::ofstream ostr( "active_site_grid.kin" );
		writer.set_line_color( "green" );
		writer.write_grid_to_kinemage( ostr, "act_site", copy_active );
		*/
	}
}

void
Matcher::initialize_occupied_space_hash()
{
	occ_space_hash_ = new OccupiedSpaceHash;
	occ_space_hash_->set_bounding_box( occ_space_bounding_box_ );
	occ_space_hash_->set_xyz_bin_widths( euclidean_bin_widths_ );
	occ_space_hash_->set_euler_bin_widths( euler_bin_widths_ );

	occ_space_hash_->initialize();

}

void
Matcher::initialize_downstream_algorithms()
{
	for ( Size ii = 1; ii <= n_geometric_constraints_; ++ii ) {
		geomcst_is_upstream_only_[ ii ] = representative_downstream_algorithm_[ ii ]->upstream_only();
	}
}

downstream::DownstreamBuilderOP
Matcher::create_ds_builder(
	Size const cst_id,
	core::chemical::ResidueTypeCAP restype,
	utility::vector1< std::string >  const & upstream_launch_atoms,
	utility::vector1< core::id::AtomID > const & downstream_3atoms,
	bool catalytic_bond
)
{
	runtime_assert( upstream_launch_atoms.size() == 3 );
	runtime_assert( downstream_3atoms.size() == 3 );

	runtime_assert( build_set_id_for_restype_[ cst_id ].find( restype->name() )
		!= build_set_id_for_restype_[ cst_id ].end() );

	//Size build_set_id = build_set_id_for_restype_[ cst_id ][ restype->name() ];
	runtime_assert( dynamic_cast< upstream::ProteinUpstreamBuilder * > ( upstream_builders_[ cst_id ].get() ) );
	upstream::ProteinUpstreamBuilderOP prot_sc_builder( static_cast< upstream::ProteinUpstreamBuilder * > ( upstream_builders_[ cst_id ].get() ));

	upstream::BuildSet & build_set = prot_sc_builder->build_set( restype );

	runtime_assert( build_set.has_restype() );

	/// Only supports rigid-ligand builders for now... This code will expand in the future.
	runtime_assert( downstream_pose_ );
	runtime_assert( downstream_pose_->total_residue() == 1 );

	for ( Size ii = 1; ii <= 3; ++ii ) {
		runtime_assert( downstream_3atoms[ ii ].rsd() == 1 );
		runtime_assert( downstream_3atoms[ ii ].atomno() <= downstream_pose_->residue( 1 ).natoms() );
	}

	downstream::RigidLigandBuilderOP rigid_builder = new downstream::RigidLigandBuilder;
	rigid_builder->ignore_h_collisions( true );
	rigid_builder->initialize_from_residue(
		downstream_3atoms[ 1 ].atomno(),
		downstream_3atoms[ 2 ].atomno(),
		downstream_3atoms[ 3 ].atomno(),
		downstream_orientation_atoms_[ 1 ].atomno(),
		downstream_orientation_atoms_[ 2 ].atomno(),
		downstream_orientation_atoms_[ 3 ].atomno(),
		downstream_pose_->residue(1) );

	if ( catalytic_bond ) {
		using namespace core::scoring::etable::count_pair;
		utility::vector1< std::pair< Size, Size > > bond_list;

		Size upstream_atom_id = build_set.restype().atom_index( upstream_launch_atoms[ 1 ] );
		Size downstream_atom_id = downstream_3atoms[ 1 ].atomno();

		bond_list.push_back( std::make_pair( upstream_atom_id, downstream_atom_id ) );

		CountPairGenericOP cpgen = new CountPairGeneric(
			build_set.restype(),
			downstream_pose_->residue_type(1),
			bond_list );
		/// Unclear what the xover value should be... 3 ignores collisions for
		/// atoms that are 3 bonds apart
		cpgen->set_crossover( 3 );

		rigid_builder->initialize_upstream_residue( & build_set.restype(), cpgen );

	} else {
		rigid_builder->initialize_upstream_residue( & build_set.restype() );
	}

	downstream_builders_[ cst_id ].push_back( rigid_builder );
	all_downstream_builders_.push_back( rigid_builder );
	return rigid_builder;
}


//// @details Select one of several different ways to subsample the matches in a particular
/// 6D hasher voxel.
void
Matcher::select_hit_representatives(
	utility::vector1< utility::vector1< Hit const * > > const & /*hit_vectors*/,
	utility::vector1< Size > & n_hits_per_geomcst,
	utility::vector1< utility::vector1< Size > > & reps
) const
{
	/// TERRIBLY CRUDE!  TAKE THE FIRST 10 ONLY.
	for ( Size ii = 1; ii <= n_geometric_constraints_; ++ii  ) {
		//if ( representative_downstream_algorithm_[ ii ]->generates_primary_hits() ) {
			if ( n_hits_per_geomcst[ ii ] > 10 ) {
				n_hits_per_geomcst[ ii ] = 10;
			}
			reps[ ii ].resize( n_hits_per_geomcst[ ii ] );
			for ( Size jj = 1; jj <= n_hits_per_geomcst[ ii ]; ++jj ) {
				reps[ ii ][ jj ] = jj;
			}
		//} else {
			/// just take the first hit? -- maybe they should be sorted into the unique hits
			//n_hits_per_geomcst[ ii ] = 1;
			//reps[ ii ].resize( 1 );
			//reps[ ii ][ 1 ] = 1;
		//}
	}
}

/// @brief Returns false if all non-upstream-only hits are compatible.
/// Returns true if any non-upstream-only hits are incompatible, and increments
/// the lexicographical iterator at the most-significant dimension possible
bool
Matcher::check_non_upstream_only_hit_incompatibility(
	match_dspos1 const & m1,
	utility::LexicographicalIterator & lex
) const
{
	/// Before descending into the secondary matches, check that none of the non-upstream-only
	/// hits are incompatible with each other.  Once we're iterating over the upstream-only
	/// hits, we'll assume that all the non-upstream-only hits are compatible with each other.
	/// Structure this loop so that the "earliest" incompatibility is found.
	/// ii will iterate from 2 to n_geometrict_constraints and look at whether
	/// it is incomaptible with any of the hits from 1 to ii-1; if it is,
	/// then we can increment the lexicographical iterator at position ii.
	/// This guarantees the most-significant dimension producing an incompatibility
	/// will be advanced, skipping the largest possible number of (incompatible)
	/// hit combinations.
	for ( Size ii = 2; ii <= n_geometric_constraints_; ++ii ) {
		if ( geomcst_is_upstream_only_[ ii ] ) continue; // ignore upstream-only hits

		Size ii_bp( m1.upstream_hits[ ii ].scaffold_build_id() );
		for ( Size jj = 1; jj < ii; ++jj ) {
			if ( geomcst_is_upstream_only_[ jj ] ) continue; // ignore upstream-only hits

			Size jj_bp( m1.upstream_hits[ jj ].scaffold_build_id() );
			if ( ! all_build_points_[ ii_bp ]->compatible( *all_build_points_[ jj_bp ]) ) {
				lex.continue_at_dimension( ii );
				//std::cout << "Incompatible at 1153 " << ii << " " << jj << " " << ii_bp << " " << jj_bp << std::endl;
			}
			if ( ! upstream_builders_[ ii ]->compatible( fake_hit( m1.upstream_hits[ ii ] ),
					*all_build_points_[ ii_bp ], *upstream_builders_[ jj ],
					fake_hit( m1.upstream_hits[ jj ] ), *all_build_points_[ jj_bp ] ))  {
				lex.continue_at_dimension( ii );
				//std::cout << "Incompatible at 1159 " << ii << " " << jj << " " << ii_bp << " " << jj_bp << std::endl;
				return true;
			}
		}
	}
	//std::cout << "Compatible at 1377" << std::endl;
	return false; // no incompatibility
}

/// @details returns false if all upstream-only hits in a particular combination
/// are compatible with each other and with the non-upstream-only hits.  Returns
/// true if any are incompatible.  Ignores compatibility between non-upstream-only hits.
/// If there is an incompatibility, the upstream_only_hit_iterators are advanced.
/// In the event that this increment beyond the incompatible upstream-only-hit combinations
/// advance the most-significant upstream-only geometric-constraint's iterator
/// to its list end, then this function sets the value of
/// last_upstream_only_geomcst_advanced to zero.
bool
Matcher::test_upstream_only_hit_incompatibility(
	match_dspos1 const & m1,
	utility::vector1< HitPtrListCOP > const & upstream_only_hits,
	utility::vector1< std::list< Hit const * >::const_iterator > & upstream_only_hit_iterators,
	Size & last_upstream_only_geomcst_advanced
) const
{
	/// Determine if any of the secondary hits are incompatible and therefore
	/// could not produce a match.
	/// If we do find an incompatibility, then "continue" at the most-significant
	/// dimension.
	bool incompatible( false ), outstanding_list_increment( false );
	for ( Size ii = 2; ii <= n_geometric_constraints_; ++ii ) {
		Size ii_bp( m1.upstream_hits[ ii ].scaffold_build_id() );
		for ( Size jj = 1; jj < ii; ++jj ) {
			Size jj_bp( m1.upstream_hits[ jj ].scaffold_build_id() );

			/// We have already checked for compatibility between non-upstream-only matches.
			if ( ! geomcst_is_upstream_only_[ ii ] && ! geomcst_is_upstream_only_[ jj ] ) continue;

			if ( ! all_build_points_[ ii_bp ]->compatible( *all_build_points_[ jj_bp ] )) {
				incompatible = true;
			}
			if ( ! incompatible && ! upstream_builders_[ ii ]->compatible(
					fake_hit( m1.upstream_hits[ ii ] ), *all_build_points_[ ii_bp ],
					*upstream_builders_[ jj ], fake_hit( m1.upstream_hits[ jj ] ), *all_build_points_[ jj_bp ] ))  {
				incompatible = true;
			}

			if ( incompatible ) {
				/// Increment the most-significant geom-cst to alleviate the incompatibility
				/// either ii or jj represents an upstream-only geom-cst.
				/// if it's not ii, then we start our incrementing at jj.
				for ( Size kk = ( ! geomcst_is_upstream_only_[ ii ] ? jj : ii ); kk >= 1; --kk ) {
					if ( geomcst_is_upstream_only_[ kk ] ) {
						++upstream_only_hit_iterators[ kk ];
						last_upstream_only_geomcst_advanced = kk;
						if ( upstream_only_hit_iterators[ kk ] == upstream_only_hits[ kk ]->val().end() ) {
							outstanding_list_increment = true;
						} else {
							outstanding_list_increment = false;
							break;
						}
					}
				}
				//std::cout << "Incompatible at 1222 " << ii << " " << jj << std::endl;
				if ( outstanding_list_increment ) {
					// indicate to the calling function that we've visited all of the upstream-only hit combinations.
					last_upstream_only_geomcst_advanced = 0;
				}
				return true;
			} // if incompatible
		}
	}
	//std::cout << "Compatible at 1231" << std::endl;
	return false;
}

/// @details returns true if more upstream-only hit combinations remain,
/// and false if there are no upstream-only hit cominbations remaining
bool
Matcher::increment_upstream_only_hit_combination(
	utility::vector1< HitPtrListCOP > const & upstream_only_hits,
	Size starting_point,
	utility::vector1< std::list< Hit const * >::const_iterator > & upstream_only_hit_iterators,
	Size & last_upstream_only_geomcst_advanced
) const
{
	for ( Size ii = starting_point; ii >= 1; --ii ) {
		if ( geomcst_is_upstream_only_[ ii ] ) {
			++upstream_only_hit_iterators[ ii ];
			last_upstream_only_geomcst_advanced = ii;
			if ( upstream_only_hit_iterators[ ii ] != upstream_only_hits[ ii ]->val().end() ) {
				return true; // more upstream-only-hit-combinations remain
			}
		}
	}
	return false; // we did not find an upstream-only geom-cst that was not at the end of its hit list.
}

void
Matcher::process_matches_main_loop_enumerating_all_hit_combos( output::MatchProcessor & processor ) const
{

	HitHasher hit_hasher;
	hit_hasher.set_bounding_box( occ_space_bounding_box_ );
	hit_hasher.set_xyz_bin_widths( euclidean_bin_widths_ );
	hit_hasher.set_euler_bin_widths( euler_bin_widths_ );

	hit_hasher.set_nhits_per_match( n_geometric_constraints_ );
	hit_hasher.initialize();

	//utility::vector1< std::map< Size, bool > > hit_scaff_build_pts( n_geometric_constraints_ );

	MatchOutputTracker tracker;
	match        m(     n_geometric_constraints_ );
	match_lite   mlite( n_geometric_constraints_ );
	match_dspos1 m1(    n_geometric_constraints_ ); // for finding upstream-only hits

	TR << "Begining match enumeration:" << std::endl;
	for ( Size ii = 1; ii <= n_geometric_constraints_; ++ii ) {
		TR << "Geometric constraint " << ii << " produced " << hits_[ ii ].size() << " hits" << std::endl;
	}

	TR << "Begining examination of each of the 64 definitions of the 6D-hash-grid origin:" << std::endl;
	//TR << "process_matches_main_loop_enumerating_all_hit_combos" << std::endl;

	for ( Size ii = 1; ii <= 64; ++ii ) {

		/// Initialize the hit-hasher's hash map with the ii'th definition of the origin.
		/// Significant memory savings by deleting the ii'th hash map at the conclusion
		/// of the ii'th iteration, instead of having all 64 hash maps in memory at the same time.
		for ( Size jj = 1; jj <= n_geometric_constraints_; ++jj ) {
			if ( ! geomcst_is_upstream_only_[ jj ] ) {
				for ( std::list< Hit >::const_iterator iter = hits_[ jj ].begin(), iter_end = hits_[ jj ].end();
						iter != iter_end; ++iter ) {
					hit_hasher.insert_hit( ii, jj, & (*iter) );
					//hit_scaff_build_pts[ jj ][ iter->first()[ 1 ] ] = true;
				}
			}
			/// else { noop }, do not hash upstream-only hits; such hits are enumerated separately
		}

		for ( HitHasher::HitHash::const_iterator
				iter = hit_hasher.hit_hash_begin( ii ),
				iter_end = hit_hasher.hit_hash_end( ii );
				iter != iter_end; ++iter ) {

			/// MATCHES!
			HitHasher::MatchSet const & match_set( iter->second );

			utility::vector1< Size > n_hits_per_geomcst( n_geometric_constraints_ );
 			utility::vector1< utility::vector1< Hit const * > > hit_vectors( n_geometric_constraints_ );
			// representative hits, if we select a subset of hits for each geometric constraint
			// enumerating all combinations of hits can be very expensive.
			utility::vector1< utility::vector1< Size > > reps( n_geometric_constraints_ );
			/// hits from upstream-only downstream algorithms
			utility::vector1< HitPtrListCOP > upstream_only_hits( n_geometric_constraints_ );
			utility::vector1< std::list< Hit const * >::const_iterator > upstream_only_hit_iterators( n_geometric_constraints_ );

			/// First check that there is at least one hit per geometric constraint;
			/// go ahead and initialize the hit_vectors array at the same time.
			bool any_size_zero( false );
			for ( Size jj = 1; jj <= n_geometric_constraints_; ++jj ) {
				if ( ! geomcst_is_upstream_only_[ jj ] ) {
					n_hits_per_geomcst[ jj ] = match_set[ jj ].size();
					any_size_zero |= n_hits_per_geomcst[ jj ] == 0;
					hit_vectors[ jj ].resize( n_hits_per_geomcst[ jj ] );
					std::copy( match_set[ jj ].begin(), match_set[ jj ].end(), hit_vectors[ jj ].begin());
				} else {
					n_hits_per_geomcst[ jj ] = 1; // for the lex, indicate there's only a single value for geomcst jj
				}
			}
			if ( any_size_zero ) continue; // no matches possible in this voxel.

			select_hit_representatives( hit_vectors, n_hits_per_geomcst, reps );

			/// Prepare to iterate across all combinations of hits.
			utility::LexicographicalIterator lex( n_hits_per_geomcst );

			while ( ! lex.at_end() ) {

				/// Assemble the (partial) match
				for ( Size jj = 1; jj <= n_geometric_constraints_; ++jj ) {
					if ( ! geomcst_is_upstream_only_[ jj ] ) {
						mlite[ jj ] =   hit_vectors[ jj ][ reps[jj][lex[ jj ]] ];
						m[ jj ]     = *(hit_vectors[ jj ][ reps[jj][lex[ jj ]] ] );
						m1.upstream_hits[ jj ].copy_hit( m[ jj ] );
					}
				}

				/// if any of the non-upstream-only hits are incompatible, increment the lex
				/// and proceed to the next combination of hits
				if ( check_non_upstream_only_hit_incompatibility( m1, lex ) ) continue;

				/// Now descend into the upstream-only hits (and if there are none, output the singular
				/// match combination from the set of non-upstream-only hits)
				// "inner lex" traversed here -- we're enumerating all the combinations of upstream-only hits
				Size last_upstream_only_geomcst_advanced( 0 );
				while ( true ) {
					if ( last_upstream_only_geomcst_advanced != 0 ) {
						Size const lgca = last_upstream_only_geomcst_advanced; // brevity
						mlite[ lgca ] =  *upstream_only_hit_iterators[ lgca ];
						m[ lgca ]     = **upstream_only_hit_iterators[ lgca ];
						m1.upstream_hits[ lgca ].copy_hit( m[ lgca ] );
					}
					Size empty_hitlist_id( 0 );
					for ( Size jj = last_upstream_only_geomcst_advanced + 1; jj <= n_geometric_constraints_; ++jj ) {
						if ( geomcst_is_upstream_only_[ jj ] ) {
							upstream_only_hits[ jj ] = representative_downstream_algorithm_[ jj ]->
								hits_to_include_with_partial_match( m1 );
							if ( upstream_only_hits[ jj ] == 0 ) {
								empty_hitlist_id = jj;
								break;
							}
							upstream_only_hit_iterators[ jj ] = upstream_only_hits[ jj ]->val().begin();
							mlite[ jj ] =  *upstream_only_hit_iterators[ jj ];
							m[ jj ]     = **upstream_only_hit_iterators[ jj ];
							m1.upstream_hits[ jj ].copy_hit( m[ jj ] );
							last_upstream_only_geomcst_advanced = jj;
						}
					}
					if ( empty_hitlist_id != 0 ) {
						// advance the upstream_only iterators;
						increment_upstream_only_hit_combination( upstream_only_hits, empty_hitlist_id - 1,
							upstream_only_hit_iterators, last_upstream_only_geomcst_advanced );
					}

					if ( test_upstream_only_hit_incompatibility(
							m1, upstream_only_hits, upstream_only_hit_iterators,
							last_upstream_only_geomcst_advanced ) ) {
						// we have an incompatibility; break if we're at the end of the upstream-only hit combos
						if ( last_upstream_only_geomcst_advanced == 0 ) break;
						continue;
					}

					/// If we've already seen this match for a previous value of ii (some alternate
					/// definition of the hasher origin) then avoid outputting it a second time.
					if ( ! tracker.match_has_been_output( mlite ) ) {
						/// Record that we have seen this combination of hits.
						tracker.note_output_match( mlite );
						processor.process_match( m );
					}

					if ( ! increment_upstream_only_hit_combination(
							upstream_only_hits,
							n_geometric_constraints_,
							upstream_only_hit_iterators,
							last_upstream_only_geomcst_advanced )) {
						break;
					}
				}
				++lex;
			}
		}

		hit_hasher.clear_hash_map( ii );
		std::cout << "." << std::flush;
		if ( ii % 20 == 0 ) std::cout << std::endl;
	}
	std::cout << std::endl;
}


void
Matcher::process_matches_where_one_geomcst_defines_downstream_location(
	output::MatchProcessor & processor
) const
{
	typedef utility::fixedsizearray1< Size, 2 > Size2;
	typedef utility::OrderedTuple< Size2 > Size2Tuple;
	typedef std::map< Size2Tuple, Hit const * > UpstreamRotamerRepresentativeMap;
	typedef UpstreamRotamerRepresentativeMap::const_iterator UpRotRepMapConstIterator;

	utility::vector1< UpstreamRotamerRepresentativeMap > upstream_rotamer_representatives( n_geometric_constraints_ );

	for ( Size ii = 1; ii <= n_geometric_constraints_; ++ii ) {
		if ( ! geomcst_is_upstream_only_[ ii ] ) {
			for ( std::list< Hit >::const_iterator hititer = hits_[ ii ].begin(),
					hititer_end = hits_[ ii ].end(); hititer != hititer_end; ++hititer ) {
				Size2 rotid;
				rotid[ 1 ] = hititer->scaffold_build_id();
				rotid[ 2 ] = hititer->upstream_conf_id();
				UpRotRepMapConstIterator representative = upstream_rotamer_representatives[ ii ].find( rotid );
				if ( representative == upstream_rotamer_representatives[ ii ].end() ) {
					upstream_rotamer_representatives[ ii ][ rotid ] = & (*hititer);
				}
			}
		}
	}

	HitHasher hit_hasher;
	hit_hasher.set_bounding_box( occ_space_bounding_box_ );
	hit_hasher.set_xyz_bin_widths( euclidean_bin_widths_ );
	hit_hasher.set_euler_bin_widths( euler_bin_widths_ );

	hit_hasher.set_nhits_per_match( n_geometric_constraints_ );
	hit_hasher.initialize();

	utility::vector1< std::map< Size, bool > > hit_scaff_build_pts( n_geometric_constraints_ );

	MatchOutputTracker tracker;
	//match          m( n_geometric_constraints_ );
	match_dspos1  m1( n_geometric_constraints_ );
	match_lite mlite( n_geometric_constraints_ );

	/// The match lite where each hit but one has been replaced by it's
	/// upstream-rotamer-representative pointer.  This match-lite prevents repitition
	/// of the match_dspos1 outputs.
	match_lite mliteprime( n_geometric_constraints_ );

	TR << "Begining match enumeration:" << std::endl;
	for ( Size ii = 1; ii <= n_geometric_constraints_; ++ii ) {
		TR << "Geometric constraint " << ii << " produced " << hits_[ ii ].size() << " hits" << std::endl;
	}

	TR << "Begining examination of each of the 64 definitions of the 6D-hash-grid origin:" << std::endl;
	//TR << "process_matches_where_one_geomcst_defines_downstream_location" << std::endl;

	for ( Size ii = 1; ii <= 64; ++ii ) {
		for ( Size jj = 1; jj <= n_geometric_constraints_; ++jj ) {
			if ( ! geomcst_is_upstream_only_[ jj ] ) {
				for ( std::list< Hit >::const_iterator iter = hits_[ jj ].begin(), iter_end = hits_[ jj ].end();
						iter != iter_end; ++iter ) {
					hit_hasher.insert_hit( ii, jj, & (*iter) );
					hit_scaff_build_pts[ jj ][ iter->first()[ 1 ] ] = true;
				}
			} /// else noop, do not hash upstream-only hits; such hits are enumerated separately
		}

		for ( HitHasher::HitHash::const_iterator
				iter = hit_hasher.hit_hash_begin( ii ),
				iter_end = hit_hasher.hit_hash_end( ii );
				iter != iter_end; ++iter ) {

			/// MATCHES!
			HitHasher::MatchSet const & match_set( iter->second );

			utility::vector1< Size > n_hits_per_geomcst( n_geometric_constraints_ );
			utility::vector1< utility::vector1< Hit const * > > hit_vectors( n_geometric_constraints_ );
			//utility::vector1< utility::vector1< Size > > reps( n_geometric_constraints_ ); //representatives

			utility::vector1< Size > n_unique_upstream_rots( n_geometric_constraints_ );
			utility::vector1< utility::vector1< Hit const * > > unique_hit_representatives( n_geometric_constraints_ );
			/// hits from upstream-only downstream algorithms
			utility::vector1< HitPtrListCOP > upstream_only_hits( n_geometric_constraints_ );
			utility::vector1< std::list< Hit const * >::const_iterator > upstream_only_hit_iterators( n_geometric_constraints_ );

			bool any_size_zero( false );
			for ( Size jj = 1; jj <= n_geometric_constraints_; ++jj ) {
				if ( ! geomcst_is_upstream_only_[ jj ] ) {
					n_hits_per_geomcst[ jj ] = match_set[ jj ].size();
					if ( n_hits_per_geomcst[ jj ] == 0 ) {
						any_size_zero = true;
						break;
					}
					hit_vectors[ jj ].resize( n_hits_per_geomcst[ jj ] );
					std::copy( match_set[ jj ].begin(), match_set[ jj ].end(), hit_vectors[ jj ].begin());

					std::list< Hit const * > hit_representatives;
					for ( Size kk = 1; kk <= match_set[ jj ].size(); ++kk ) {
						Size2 upstream_rotamer;
						upstream_rotamer[ 1 ] = hit_vectors[ jj ][ kk ]->scaffold_build_id();
						upstream_rotamer[ 2 ] = hit_vectors[ jj ][ kk ]->upstream_conf_id();
						hit_representatives.push_back( upstream_rotamer_representatives[ jj ][ upstream_rotamer ] );
					}
					hit_representatives.sort();
					hit_representatives.unique();
					n_unique_upstream_rots[ jj ] = hit_representatives.size();
					unique_hit_representatives[ jj ].resize( n_unique_upstream_rots[ jj ] );
					std::copy( hit_representatives.begin(), hit_representatives.end(), unique_hit_representatives[ jj ].begin() );
				} else {
					n_unique_upstream_rots[ jj ] = 1; // to molify the lex
				}
			}
			if ( any_size_zero ) continue;

			utility::vector1< Size > n_hits_to_enumerate( n_geometric_constraints_ );

			for ( Size jj = 1; jj <= n_geometric_constraints_; ++jj ) {
				if ( ! output_match_dspos1_for_geomcst_[ jj ] ) continue;
				runtime_assert( ! geomcst_is_upstream_only_[ jj ] ); // THIS DOESN'T BELONG HERE. FIND IT A HOME!
				m1.originating_geom_cst_for_dspos = jj;

				/// For the geometric constraint being examined, construct the counts
				/// of unique matches where the location of the downstream partner is
				/// irrelevant for all other geometric constraints -- only the rotamer id of the
				/// upstream residue is relevant.
				for ( Size kk = 1; kk <= n_geometric_constraints_; ++kk ) {
					if ( jj == kk ) {
						n_hits_to_enumerate[ kk ] = n_hits_per_geomcst[ kk ];
					} else {
						n_hits_to_enumerate[ kk ] = n_unique_upstream_rots[ kk ];
					}
				}

				utility::LexicographicalIterator lex( n_hits_to_enumerate );

				while ( ! lex.at_end() ) {
					/// Assemble the match
					for ( Size kk = 1; kk <= n_geometric_constraints_; ++kk ) {
						if ( ! geomcst_is_upstream_only_[ kk ] ) {
							if ( kk == jj ) {
								mlite[ kk ] =   hit_vectors[ kk ][ lex[kk] ];
								//m[ kk ]     = *(hit_vectors[ kk ][ lex[kk] ] );
								m1.upstream_hits[ kk ] = upstream_hit( *(hit_vectors[ kk ][ lex[kk] ] ));
								m1.downstream_conf_id  = hit_vectors[ kk ][ lex[kk] ]->downstream_conf_id();
								m1.dspos               = hit_vectors[ kk ][ lex[kk] ]->second();
							} else {
								mlite[ kk ] =   unique_hit_representatives[ kk ][ lex[kk] ];
								//m[ kk ]     = *(unique_hit_representatives[ kk ][ lex[kk] ]);
								m1.upstream_hits[ kk ].copy_hit( *(unique_hit_representatives[ kk ][ lex[kk] ]) );
							}
						}
					}

					/// if any of the non-upstream-only hits are incompatible, increment the lex
					/// and proceed to the next combination of hits
					if ( check_non_upstream_only_hit_incompatibility( m1, lex ) ) continue;

					/// Now descend into the upstream-only hits (and if there are none, output the singular
					/// match combination from the set of non-upstream-only hits)
					// "inner lex" traversed here -- we're enumerating all the combinations of upstream-only hits
					Size last_upstream_only_geomcst_advanced( 0 );
					while ( true ) {
						if ( last_upstream_only_geomcst_advanced != 0 ) {
							Size const lgca = last_upstream_only_geomcst_advanced; // brevity
							mlite[ lgca ] =  *upstream_only_hit_iterators[ lgca ];
							//m[ lgca ]     = **upstream_only_hit_iterators[ lgca ];
							m1.upstream_hits[ lgca ].copy_hit( *mlite[ lgca ] );
						}
						Size empty_hitlist_id( 0 );
						for ( Size jj = last_upstream_only_geomcst_advanced + 1; jj <= n_geometric_constraints_; ++jj ) {
							if ( geomcst_is_upstream_only_[ jj ] ) {
								upstream_only_hits[ jj ] = representative_downstream_algorithm_[ jj ]->
									hits_to_include_with_partial_match( m1 );
								if ( upstream_only_hits[ jj ] == 0 ) {
									empty_hitlist_id = jj;
									break;
								}
								upstream_only_hit_iterators[ jj ] = upstream_only_hits[ jj ]->val().begin();
								mlite[ jj ] =  *upstream_only_hit_iterators[ jj ];
								//m[ jj ]     = **upstream_only_hit_iterators[ jj ];
								m1.upstream_hits[ jj ].copy_hit( *mlite[ jj ] );
								last_upstream_only_geomcst_advanced = jj;
							}
						}
						if ( empty_hitlist_id != 0 ) {
							// advance the upstream_only iterators;
							increment_upstream_only_hit_combination( upstream_only_hits, empty_hitlist_id - 1,
								upstream_only_hit_iterators, last_upstream_only_geomcst_advanced );
						}

						if ( test_upstream_only_hit_incompatibility(
								m1, upstream_only_hits, upstream_only_hit_iterators,
								last_upstream_only_geomcst_advanced ) ) {
							// we have an incompatibility; break if we're at the end of the upstream-only hit combos
							if ( last_upstream_only_geomcst_advanced == 0 ) break;
							continue;
						}

						/// If we've already seen this match for a previous value of ii (some alternate
						/// definition of the hasher origin) then avoid outputting it a second time.
						if ( ! tracker.match_has_been_output( mlite ) ) {
							/// Record that we have seen this combination of hits.
							tracker.note_output_match( mlite );
							processor.process_match( m1 );
						}

						if ( ! increment_upstream_only_hit_combination(
								upstream_only_hits,
								n_geometric_constraints_,
								upstream_only_hit_iterators,
								last_upstream_only_geomcst_advanced )) {
							break;
						}
					}
					++lex;
				}
			}
		}

		hit_hasher.clear_hash_map( ii );
		std::cout << "." << std::flush;
		if ( ii % 20 == 0 ) std::cout << std::endl;
	}
	std::cout << std::endl;
}


void
Matcher::note_primary_change_to_geom_csts_hitlist( Size geomcst_id )
{
	if ( ! geom_cst_has_primary_modification_[ geomcst_id ] ) {
		geom_csts_with_primary_hitlist_modificiations_.push_back( geomcst_id );
		geom_cst_has_primary_modification_[ geomcst_id ] = true;
	}
}

/*
	core::pose::PoseOP upstream_pose_;
	core::pose::PoseOP downstream_pose_;

	utility::vector1< Size > pose_build_resids_;
	utility::vector1< ScaffoldBuildPointOP > all_build_points_;
	utility::vector1< utility::vector1< ScaffoldBuildPointOP > > per_constraint_build_points_;

	utility::vector1< std::list< Hit > > hits_;

	Size n_geometric_constraints_;
	utility::vector1< UpstreamBuilderOP   >                       upstream_builders_;
	utility::vector1< std::map< std::string, Size > >             build_set_id_for_restype_;
	utility::vector1< utility::vector1< DownstreamAlgorithmOP > > downstream_algorithms_;

	utility::vector1< DownstreamBuilderOP >   all_downstream_builders_;
	utility::vector1< DownstreamAlgorithmOP > all_downstream_algorithms_;

	BumpGridOP bb_grid_;
	utility::vector1< BumpGridOP > original_scaffold_residue_bump_grids_;

	OccupiedSpaceHashOP occ_space_hash_;

*/

}
}
