// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
// :noTabs=false:tabSize=4:indentSize=4:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   protocols/match/SixDHasher.cc
/// @brief  Implementation of 6D hasher classes
/// @author Alex Zanghellini (zanghell@u.washington.edu)
/// @author Andrew Leaver-Fay (aleaverfay@gmail.com), porting to mini


// Unit headers
#include <protocols/match/SixDHasher.hh>


namespace protocols {
namespace match {


SixDCoordinateBinner::SixDCoordinateBinner()
{}

SixDCoordinateBinner::SixDCoordinateBinner(
	BoundingBox const & bounding_box,
	Size3 const & euler_offsets,
	utility::fixedsizearray1< Real, 6 > bin_widths
) :
	bounding_box_( bounding_box ),
	bin_widths_( bin_widths )
{
	Vector span = bounding_box_.upper() - bounding_box_.lower();
	Vector new_upper = bounding_box_.upper();

	for ( Size ii = 1; ii <= 3; ++ii ) {
		dimsizes_[ ii ] = static_cast< Size > ( span( ii ) / bin_widths_[ ii ] );
		if ( dimsizes_[ ii ] * bin_widths_[ ii ] < span( ii ) ) {
			dimsizes_[ ii ] += 1;
			new_upper( ii ) = bounding_box_.lower()( ii ) + dimsizes_[ ii ] * bin_widths_[ ii ];
		}
	}
	bounding_box_.set_upper( new_upper );

	for ( Size ii = 4; ii <= 5; ++ii ) {
		dimsizes_[ ii ] = static_cast< Size > ( 360.0 / bin_widths_[ ii ] );
	}
	dimsizes_[ 6 ] = static_cast< Size > ( 180.0 / bin_widths_[ 6 ] );

	/// Add an extra bin so that values near 180 ( wi binwidth/2) and near 0 ( wi binwidth/2 ) can be joined.
	if ( euler_offsets[ 3 ] != 0 ) {
		++dimsizes_[ 6 ];
	}

	dimprods_[ 6 ] = 1;
	for ( Size ii = 5; ii >= 1; --ii ) {
		dimprods_[ ii ] = dimprods_[ ii + 1 ] * dimsizes_[ ii + 1 ];
	}

	for ( Size ii = 1; ii <= 3; ++ii ) {
		if ( euler_offsets[ ii ] != 0 ) {
			euler_offsets_[ ii ] = bin_widths_[ ii + 3 ] / 2;
		} else {
			euler_offsets_[ ii ] = 0.0;
		}
	}
}

/// @details When floating point comparison breaks down, it is possible to have a point
/// hash outside of the bounding volume:  359.999999 / 10.0 -> 36 instead of 35.
/// For this reason, it's important to mod the resulting bin index by the number of bins.
Bin6D
SixDCoordinateBinner::bin6( Real6 const & values ) const {

	core::Vector xyzcoord( values[ 1 ], values[ 2 ], values[ 3 ] );

	assert( bounding_box_.contains( xyzcoord ) );
	assert( values[ 4 ] >= 0.0 && values[ 4 ] < 360.0 );
	assert( values[ 5 ] >= 0.0 && values[ 5 ] < 360.0 );
	assert( values[ 6 ] >= 0 && values[ 6 ] <= 180.0 );

	core::Vector from_corner = xyzcoord - bounding_box_.lower();
	Bin6D bins;

	bins[ 1 ] = static_cast< Size > ( from_corner.x() / bin_widths_[ 1 ] );
	if ( bins[ 1 ] == dimsizes_[ 1 ] ) --bins[ 1 ];

	bins[ 2 ] = static_cast< Size > ( from_corner.y() / bin_widths_[ 2 ] );
	if ( bins[ 2 ] == dimsizes_[ 2 ] ) --bins[ 2 ];

	bins[ 3 ] = static_cast< Size > ( from_corner.z() / bin_widths_[ 3 ] );
	if ( bins[ 3 ] == dimsizes_[ 3 ] ) --bins[ 3 ];

	Real3 euler;

	if ( values[ 6 ] > euler_offsets_[ 3 ] ) {
		euler[ 3 ] = values[ 6 ] + euler_offsets_[ 3 ];

	} else {
		euler[ 3 ] = values[ 6 ];
	}

	if (( values[ 6 ] < euler_offsets_[ 3 ] || values[ 6 ] >= 180.0 - euler_offsets_[ 3 ] ) &&
			( values[ 4 ] - euler_offsets_[ 1 ] > 180.0 )) {
		/// WRAP if phi > 180 to the region of negative theta rotation.
		euler[ 1 ] = values[ 4 ] - euler_offsets_[ 1 ] - 180.0;
		euler[ 2 ] = values[ 5 ] - euler_offsets_[ 2 ] - 180.0;
		if ( euler[ 2 ] < 0 ) euler[ 2 ] += 360;
	} else {
		/// Leave phi/psi in their usual positions
		euler[ 1 ] = values[ 4 ] - euler_offsets_[ 1 ];
		if ( euler[ 1 ] < 0 ) euler[ 1 ] += 360;

		euler[ 2 ] = values[ 5 ] - euler_offsets_[ 2 ];
		if ( euler[ 2 ] < 0 ) euler[ 2 ] += 360;
	}


	bins[ 4 ] = static_cast< Size > ( euler[ 1 ] / bin_widths_[ 4 ] ) % dimsizes_[ 4 ];
	bins[ 5 ] = static_cast< Size > ( euler[ 2 ] / bin_widths_[ 5 ] ) % dimsizes_[ 5 ];
	bins[ 6 ] = static_cast< Size > ( euler[ 3 ] / bin_widths_[ 6 ] ) % dimsizes_[ 6 ];

	return bins;
}



/*

/// @brief A struct to convert a discrete voxel in a bound volume into an integer.
/// This struct is inspired by the r++ version: struct hasher, which hashed 6 dofs.
/// however, it does not store grids.
/// An instance of this struct may be passed to the constructor of a boost::unordered_map
/// to define a hash map for a volume of space of particular size.  The index for each
/// dimension for a particular voxel should be less than the size of that dimension:
/// voxel[ i ] >= 0 && voxel[ i ] < dimsizes_[ i ].
/// CAUTION:  This is "0-based indexing" for the voxels,
/// instead of "1-based indexing" which is used in the rest of rosetta.
/// The functor returns a value in the range from [ 0, prod( 1<=i<=3 ) dimsizes_[ i ] ].
///
/// @details This struct does not contain static data, so separate boost::unordered_map
/// objects may be initialized with distinct instances of this class.  This would be
/// unremarkable except that the previous novozyme hash_map implementation used static
/// data in the hash object.
struct Bin3D_hash
{
public:
	typedef core::Size Size;

public:
	/// @brief default ctor sets all the dim sizes to 0 equivalent to a volume of 0.
	/// default ctor should not be used when creating a boost::unordered_hash
	/// object, but boost requires this constructor.
	Bin3D_hash() {
		dimsizes_[ 1 ] = dimsizes_[ 2 ] = dimsizes_[ 3 ] = 0;
		dimprods_[ 1 ] = dimprods_[ 2 ] = dimprods_[ 3 ] = 0;
	}

public:

	/// @brief Argument-constructor indicates the sizes of the three dimensions being
	/// hashed; Use this constructor and not the default ctor.
	Bin3D_hash( Size size1, Size size2, Size size3 )
	{
		dimsizes_[ 1 ] = size1;
		dimsizes_[ 2 ] = size2;
		dimsizes_[ 3 ] = size3;
		dimprods_[ 1 ] = 1;
		dimprods_[ 2 ] = size1;
		dimprods_[ 3 ] = size1 * size2;

		/// ensure that the volume of space being discretized is not too large to be hashed.
		Size na= size1 * size2 * size3;
		na = (Size) ((std::log( (core::Real) na ) / std::log( 2.0 )) + 1);
		Size const nb = sizeof( Size ) * 8;
		if ( na > nb ) {
			// return( A % P );
			std::cerr << "ERROR: hash key generation extends Size encoding size (" << 8 * sizeof( Size ) << " bits)" << std::endl;
			std::cerr << "ERROR: size1 = " << size1 << " size2 = " << size2 << " size3 = " << size3 << std::endl;
			std::cerr << "ERROR: unsupported code!!!" << std::endl;
			utility_exit_with_message( "ERROR: Bin3D_hash allocated with too large a grid size" );
		}
	}

	/// @brief functor used by boost (and sgi's stl) hash classes.
	core::Size
	operator() ( Bin3D const & bin ) const {
		Size const P = 999987;   // largest prime  number less than max int value coded by size_t int

		///A += k.i1+k.i2*ref->M+k.i3*ref->M*ref->N;
		assert( bin[ 1 ] < dimsizes_[ 1 ] );
		assert( bin[ 2 ] < dimsizes_[ 2 ] );
		assert( bin[ 3 ] < dimsizes_[ 3 ] );
		Size const A = bin[ 1 ] * dimprods_[ 1 ] + bin[ 2 ] * dimprods_[ 2 ] + bin[ 3 ] * dimprods_[ 3 ];

		return  A % P;

	}

	/// @brief Get the dimension size for a particular dimension; 1-based indexing.
	/// x=1, y=2, z=3.
	Size
	dimsize( Size dim ) const {
		return dimsizes_[ dim ];
	}

	/// @brief Return the number of cubes in the volume of space being represented;
	/// the number of cubes; trick: dimprods_[ 3 ] == dimsize_[ 1 ] * dimsize_[ 2 ]
	Size
	nboxes() const {
		return dimprods_[ 3 ] * dimsizes_[ 3 ];
	}

private:

	/// Array containing the sizes of the three dimensions that bound the volume
	/// of space being hashed.
	utility::fixedsizearray1< Size, 3 >  dimsizes_;

	/// "column major" so that (xbin, ybin, zbin) is next to (xbin+1, ybin, zbin )
	/// therefore, dimprods[ 0 ] = 1; dimprods[ 1 ] = dimsizes_[ 0 ] &&
	/// dimprods[ 2 ] == dimsizes_[ 0 ] * dimsizes_[ 1 ];
	utility::fixedsizearray1< Size, 3 > dimprods_;

};



class SixDHasher : public utility::pointer::ReferenceCount
{
public:
	typedef core::Vector Vector;

private:

	Vector      CoM_lower_boundary_;
	Vector      CoM_upper_boundary_;

	xyzbin_hash curr_CoM_hasher_;   // discrete representation of center-of-mass hashable volume
	euler_hash  curr_euler_hasher_; // discrete representation of euler-angle hashable volume


	//SixDHashOP hash_table_;

};

/// @brief Simple comparison functor struct for use in the boost hash table.
struct Bin6D_equals {
public:
	bool operator() ( Bin6D const & b1, Bin6D const & b2 ) {
		return
			b1[ 1 ] == b2[ 1 ] &&
			b1[ 2 ] == b2[ 2 ] &&
			b1[ 3 ] == b2[ 3 ] &&
			b1[ 4 ] == b2[ 4 ] &&
			b1[ 5 ] == b2[ 5 ] &&
			b1[ 6 ] == b2[ 6 ];
	}

};

struct Bin6D_hash {
public:
	typedef core::Size Size;

public:
	/// @brief default ctor sets all the dim sizes to 0 equivalent to a volume of 0.
	/// default ctor should not be used when creating a boost::unordered_hash
	/// object, but boost requires this constructor.
	Bin6D_hash() {
		dimsizes_[ 1 ] = dimsizes_[ 2 ] = dimsizes_[ 3 ] = dimsizes_[ 4 ] = dimsizes_[ 5 ] = dimsizes_[ 6 ] = 0;
		dimprods_[ 1 ] = dimprods_[ 2 ] = dimprods_[ 3 ] = dimprods_[ 4 ] = dimprods_[ 5 ] = dimprods_[ 6 ] = 0;
	}

public:

	/// @brief Argument-constructor indicates the sizes of the six dimensions being
	/// hashed; Use this constructor and not the default ctor.
	Bin6D_hash(
		Size size1,
		Size size2,
		Size size3,
		Size size4,
		Size size5,
		Size size6
	)
	{
		dimsizes_[ 1 ] = size1;
		dimsizes_[ 2 ] = size2;
		dimsizes_[ 3 ] = size3;
		dimsizes_[ 4 ] = size4;
		dimsizes_[ 5 ] = size5;
		dimsizes_[ 6 ] = size6;

		dimprods_[ 1 ] = 1;
		for ( Size ii = 2; ii <= 6; ++ii ) {
			dimprods_[ ii ] = dimsizes_[ ii - 1 ] * dimprods_[ ii - 1 ];
		}

		/// ensure that the volume of space being discretized is not too large to be hashed.
		Size log_nbins = 0;
		for ( Size ii = 1; ii <= 6; ++ii ) log_nbins += log2( dimsizes_[ ii ] );
		Size const nbin_limit = sizeof( boost::uint64_t ) * 8;
		if ( log_nbins > nbin_limit ) {
			std::cerr << "ERROR: hash key generation extends boost::uint64_t encoding size (" << 8 * sizeof( boost::uint64_t ) << " bits)" << std::endl;
			std::cerr << "ERROR: size1 = " << size1 << " size2 = " << size2 << " size3 = "
				<< size3 << " size4= " << size4 << " size5= " << size5 << " size6= " << size6 << std::endl;
			std::cerr << "ERROR: Discritization level is too fine or voxel span too large" << std::endl;
			utility_exit_with_message( "ERROR: Bin6D_hash allocated with too large a grid size" );
		}
	}

	/// @brief functor used by boost (and sgi's stl) hash classes.
	boost::uint64_t
	operator() ( Bin6D const & bin ) const {
		boost::uint64_t const P = 999987;   // largest prime  number less than max int value coded by size_t int (32 bits)

		assert( bin[ 1 ] >= 0 && bin_[ 1 ] < dimsizes_[ 1 ] );
		assert( bin[ 2 ] >= 0 && bin_[ 2 ] < dimsizes_[ 2 ] );
		assert( bin[ 3 ] >= 0 && bin_[ 3 ] < dimsizes_[ 3 ] );
		assert( bin[ 4 ] >= 0 && bin_[ 4 ] < dimsizes_[ 4 ] );
		assert( bin[ 5 ] >= 0 && bin_[ 5 ] < dimsizes_[ 5 ] );
		assert( bin[ 6 ] >= 0 && bin_[ 6 ] < dimsizes_[ 6 ] );
		boost::uint64_t const A =
			bin[ 1 ] * dimprods_[ 1 ] +
			bin[ 2 ] * dimprods_[ 2 ] +
			bin[ 3 ] * dimprods_[ 3 ] +
			bin[ 4 ] * dimprods_[ 4 ] +
			bin[ 5 ] * dimprods_[ 5 ] +
			bin[ 6 ] * dimprods_[ 6 ];

		return  A % P;

	}

	/// @brief Get the dimension size for a particular dimension; 1-based indexing.
	Size
	dimsize( Size dim ) const {
		return dimsizes_[ dim ];
	}

	/// @brief Return the number of cubes in the volume of space being represented;
	/// the number of cubes; trick: dimprods_[ 6 ] == prod_i dimsize_[ i ]
	Size
	nboxes() const {
		return dimprods_[ 6 ] * dimsizes_[ 6 ];
	}

private:

	/// Array containing the sizes of the three dimensions that bound the volume
	/// of space being hashed.
	utility::fixedsizearray1< Size, 6 >  dimsizes_;

	/// "column major" so that (xbin, ybin, zbin, phi, psi, theta) is next to (xbin+1, ybin, zbin, phi, psi, theta )
	/// therefore, dimprods[ 0 ] = 1; dimprods[ 1 ] = dimsizes_[ 0 ] etc.
	utility::fixedsizearray1< Size, 6 > dimprods_;

};
*/



}
}

