// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
// :noTabs=false:tabSize=4:indentSize=4:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   protocols/match/downstream/DownstreamAlgorithm.cc
/// @brief
/// @author Alex Zanghellini (zanghell@u.washington.edu)
/// @author Andrew Leaver-Fay (aleaverfay@gmail.com), porting to mini

// Unit headers
#include <protocols/match/downstream/DownstreamAlgorithm.hh>

// Package headers
#include <protocols/match/BumpGrid.hh>
#include <protocols/match/Matcher.hh>
#include <protocols/match/downstream/ActiveSiteGrid.hh>
#include <protocols/match/upstream/UpstreamBuilder.hh>
#include <protocols/match/upstream/ScaffoldBuildPoint.hh>

// Utility headers
#include <utility/pointer/ReferenceCount.hh>

// C++ headers
#include <list>

namespace protocols {
namespace match {
namespace downstream {

DownstreamAlgorithm::DownstreamAlgorithm( Size geom_cst_id ) : geom_cst_id_( geom_cst_id ) {}
DownstreamAlgorithm::DownstreamAlgorithm( DownstreamAlgorithm const & other ) :
	utility::pointer::ReferenceCount(),
	geom_cst_id_( other.geom_cst_id_ ),
	bbgrid_( other.bbgrid_ ),
	active_site_grid_( other.active_site_grid_ )
{}

DownstreamAlgorithm const &
DownstreamAlgorithm::operator = ( DownstreamAlgorithm const & rhs )
{
	if ( this != & rhs ) {
		geom_cst_id_ = rhs.geom_cst_id_;
		bbgrid_ = rhs.bbgrid_;
		active_site_grid_ = rhs.active_site_grid_;
	}
	return *this;
}


DownstreamAlgorithm::~DownstreamAlgorithm() {}

/// @details By initializing local std::list< Hit > variables inside the loop
/// over all of the build points, and then splicing them into a central vector
/// of hit lists, I can avoid expensive list-copy operations while guaranteeing
/// OpenMP thread saftey.
std::list< Hit >
DownstreamAlgorithm::build_hits_at_all_positions(
	Matcher & matcher
)
{
	return default_build_hits_at_all_positions( matcher );
}

/// @details no-op
void
DownstreamAlgorithm::prepare_for_match_enumeration( Matcher const & ) {}



/// @details Noop in base class.
void
DownstreamAlgorithm::respond_to_primary_hitlist_change( Matcher &, Size )
{}

/// @details Noop in base class.
void
DownstreamAlgorithm::respond_to_peripheral_hitlist_change( Matcher & )
{}

DownstreamAlgorithm::Size
DownstreamAlgorithm::geom_cst_id() const {
	return geom_cst_id_;
}

void
DownstreamAlgorithm::set_bb_grid(
	BumpGridCOP bbgrid
)
{
	bbgrid_ = bbgrid;
}

void
DownstreamAlgorithm::set_active_site_grid(
	ActiveSiteGridCOP active_site_grid
)
{
	active_site_grid_ = active_site_grid;
}


std::list< Hit >
DownstreamAlgorithm::default_build_hits_at_all_positions(
	Matcher const & matcher
) const
{
	utility::vector1< upstream::ScaffoldBuildPointCOP > const & launch_points
		( matcher.per_constraint_build_points( geom_cst_id_ ) );
	Size n_build_points = launch_points.size();

	std::list< Hit > all_hits;
	utility::vector1< std::list< Hit > > hits( n_build_points );

	/// Generate conformations for the upstream and downstream partners for each of the
	/// possible scaffold build points for this geometric constraint.
	/// This loop will be parallelized.  Everything down stream of this call is const,
	/// in spite of the fact that the matcher is handed as a non-const reference.
	for ( Size ii = 1; ii <= n_build_points; ++ii ) {
		std::list< Hit > iihits = matcher.upstream_builder( geom_cst_id_ )->build( * launch_points[ ii ] );
		hits[ ii ].splice( hits[ ii ].end(), iihits );
	}

	for ( Size ii = 1; ii <= n_build_points; ++ii ) {
		all_hits.splice( all_hits.end(), hits[ ii ] );
	}

	return all_hits;
}


}
}
}
