// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
// :noTabs=false:tabSize=4:indentSize=4:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   protocols/match/output/PDBWriter.cc
/// @brief  Forward declaration of class to write output matches that have (presumably) passed the output filters.
/// @author Florian Richter (floric@u.washington.edu), june 09

// Unit headers
#include <protocols/match/output/PDBWriter.hh>

// Package Headers
#include <protocols/match/Hit.hh>
#include <protocols/match/MatcherTask.hh>
#include <protocols/match/output/UpstreamHitCacher.hh>

// Project Headers
#include <core/conformation/Residue.hh>
#include <core/io/pdb/pose_io.hh>
#include <core/io/pdb/file_data.hh>
#include <core/pose/Pose.hh>
#include <core/pose/PDBInfo.hh>

#include <protocols/enzdes/EnzConstraintIO.hh>
#include <protocols/enzdes/enzdes_util.hh>

// Utility headers
#include <utility/string_util.hh>
#include <utility/pointer/ReferenceCount.hh>

#include <fstream>

namespace protocols {
namespace match {
namespace output {


PDBWriter::PDBWriter() :
	coordinate_cacher_(NULL),
	write_matchres_only_(false),
	scaf_name_(""),
	cstfile_name_(""),
	prefix_("UM"),
	num_written_(0)
{
	signature_map_.clear();
}

PDBWriter::~PDBWriter() {}


void
PDBWriter::prepare_for_output_writing()
{
	signature_map_.clear();
	num_written_ = 0;
}


void
PDBWriter::record_match( match const & m )
{
	utility::vector1< core::conformation::ResidueCOP > upstream_matchres;
	utility::vector1< core::pose::PoseCOP > downstream_poses;
	utility::vector1< core::Size > ex_geom_ids_for_upstream_res;

	std::map< core::Size, core::Size > redundant_upstream_res;
	determine_redundant_upstream_matchres( match_dspos1( m, 1 ), redundant_upstream_res );

	for ( Size ii = 1; ii <= m.size(); ++ii ) {
		core::conformation::ResidueCOP conf =
			coordinate_cacher_->upstream_conformation_for_hit( ii, m[ ii ] );
		upstream_matchres.push_back( conf );
		ex_geom_ids_for_upstream_res.push_back( m[ ii ].external_geom_id() );
		if ( ! output_dsgeom_for_geomcst_[ ii ] ) continue;
		if ( dsbuilders_[ ii ] ) {
			downstream_poses.push_back( dsbuilders_[ ii ]->downstream_pose_from_hit( m[ii] ) );
		} else {
			utility_exit_with_message( "Cannot output downstream pose for geomcst id " +
				utility::to_string( ii ) + " which does not have a downstream builder!" );
		}
	}

	utility::vector1< core::Size > matchres_seqpos;
	core::pose::PoseCOP up_outpose = create_output_upstream_pose(
		upstream_matchres, redundant_upstream_res, matchres_seqpos, *(downstream_poses[1]), ex_geom_ids_for_upstream_res );
	std::string outtag = assemble_outtag( upstream_matchres );
	core::Size outcounter(1);

	for ( Size ii = 1; ii <= m.size(); ++ii ) {
		if ( ! output_dsgeom_for_geomcst_[ ii ] ) continue;
		std::string this_tag = outtag + "_" + utility::to_string( ii ) + ".pdb";

		//every match in it's own file for now
		std::ofstream file_out( this_tag.c_str() );
		//Size atom_counter(0);

		up_outpose->dump_pdb( file_out );

		downstream_poses[ outcounter ]->dump_pdb( file_out );
		file_out.close();
		outcounter++;
	}

	num_written_++;

	//if( num_written_ % 100 == 0 ) std::cout << "recored match function called " << num_written_ << "times" <<std::endl;
}

void
PDBWriter::record_match( match_dspos1 const & m )
{

	/// What are we doing here?  The Matcher should never have sent us this match.
	if ( ! output_dsgeom_for_geomcst_[ m.originating_geom_cst_for_dspos ] ) return;

	if ( ! dsbuilders_[ m.originating_geom_cst_for_dspos ] ) {
		utility_exit_with_message( "Cannot output downstream pose for geomcst id " +
			utility::to_string( m.originating_geom_cst_for_dspos ) +
			" which does not have a downstream builder!" );
	}

	utility::vector1< core::conformation::ResidueCOP > upstream_matchres;
	utility::vector1< core::Size > ex_geom_ids_for_upstream_res;

	Hit fullhit = full_hit( m );

	core::pose::PoseCOP downstream_pose = dsbuilders_[ m.originating_geom_cst_for_dspos ]->
		downstream_pose_from_hit( fullhit );

	/// Does any single residue do double-duty? e.g. is there a residue whose
	/// sidechain is used in satisfying one geometric constraint, while it's
	/// backbone is used in another?
	std::map< core::Size, core::Size > redundant_upstream_res;
	determine_redundant_upstream_matchres( m, redundant_upstream_res );

	for ( Size ii = 1; ii <= m.upstream_hits.size(); ++ii ) {
		core::conformation::ResidueCOP conf =
			coordinate_cacher_->upstream_conformation_for_hit( ii, fake_hit( m.upstream_hits[ ii ] ) );
		upstream_matchres.push_back( conf );
		ex_geom_ids_for_upstream_res.push_back( fullhit.external_geom_id() );
	}

	utility::vector1< core::Size > matchres_seqpos;

	core::pose::PoseCOP up_outpose = create_output_upstream_pose(
		upstream_matchres, redundant_upstream_res, matchres_seqpos, *downstream_pose, ex_geom_ids_for_upstream_res );

	std::string outtag = assemble_outtag( upstream_matchres );

	std::string this_tag = outtag + "_" + utility::to_string( m.originating_geom_cst_for_dspos ) + ".pdb";
	std::ofstream file_out( this_tag.c_str() );
	up_outpose->dump_pdb( file_out );
	downstream_pose->dump_pdb( file_out );

	num_written_++;

}


void
PDBWriter::set_coordinate_cacher( UpstreamHitCacherOP cacher )
{
	coordinate_cacher_ = cacher;
}

void
PDBWriter::set_prefix( std::string const & prefix )
{
	prefix_ = prefix;
}

void
PDBWriter::initialize_from_matcher_task(
	MatcherTaskCOP mtask
){
	Parent::initialize_from_matcher_task( mtask );

	orig_upstream_pose_ = mtask->upstream_pose();
	write_matchres_only_ = mtask->output_matchres_only();
	output_dsgeom_for_geomcst_.resize( mtask->enz_input_data()->mcfi_lists_size() );
	std::fill( output_dsgeom_for_geomcst_.begin(), output_dsgeom_for_geomcst_.end(), false );
	dsbuilders_.resize( output_dsgeom_for_geomcst_.size() );
	for ( core::Size i = 1; i <= mtask->geom_csts_downstream_output().size(); ++i ) {
		output_dsgeom_for_geomcst_[ mtask->geom_csts_downstream_output()[i] ] = true;
	}

}

void
PDBWriter::set_downstream_builder(
	Size geomcst_id,
	downstream::DownstreamBuilderCOP dsbuilder
){
	dsbuilders_[ geomcst_id ] = dsbuilder;
}


void
PDBWriter::assemble_remark_lines(
	core::pose::Pose & outpose,
	utility::vector1< core::conformation::ResidueCOP > const & upstream_matchres,
	std::map< core::Size, core::Size > const & redundant_upstream_res,
	core::pose::Pose const & downstream_pose,
	utility::vector1< core::Size > const & ex_geom_ids_for_upstream_res
) const
{

	using namespace core::io::pdb;

	Remarks rems;

	std::string ds_resname = downstream_pose.residue(1).name3();

	for ( core::Size i = 1; i <= upstream_matchres.size(); ++i ) {

		RemarkInfo ri;
		ri.num = 666; /// really now?
		std::string upname3( upstream_matchres[ i ]->name3() );
		std::map< core::Size, core::Size >::const_iterator red_it = redundant_upstream_res.find( i );
		if( red_it != redundant_upstream_res.end() ){
			upname3 = upstream_matchres[ red_it->second ]->name3();
		}
		ri.value =  enzdes::enzutil::assemble_remark_line(
			"X", ds_resname, 0, "A", upname3, upstream_matchres[ i ]->seqpos(), i , ex_geom_ids_for_upstream_res[ i ] );
		rems.push_back( ri );

	}

	outpose.pdb_info()->remarks( rems );

}


core::pose::PoseCOP
PDBWriter::create_output_upstream_pose(
	utility::vector1< core::conformation::ResidueCOP > const & upstream_matchres,
	std::map< core::Size, core::Size > const & redundant_upstream_res,
	utility::vector1< core::Size > & matchres_seqpos,
	core::pose::Pose const & one_ds_pose,
	utility::vector1< core::Size > const & ex_geom_ids_for_upstream_res
)
{
	runtime_assert( ex_geom_ids_for_upstream_res.size() == upstream_matchres.size() );
	matchres_seqpos.clear();

	core::pose::PoseOP outpose;

	if( write_matchres_only_ ){
		outpose = new core::pose::Pose();

		core::pose::PDBInfoOP pdbinf = new core::pose::PDBInfo( *outpose );
		outpose->pdb_info( pdbinf );
	}

	else outpose = new core::pose::Pose( *orig_upstream_pose_ );

	for ( core::Size i = 1; i <= upstream_matchres.size(); ++i ) {
		if( redundant_upstream_res.find( i ) != redundant_upstream_res.end() ) continue;
		matchres_seqpos.push_back( upstream_matchres[i]->seqpos() );
		if( write_matchres_only_ ){
			if( i == 1 ) {
				outpose->append_residue_by_jump( *(upstream_matchres[i]), 1 );
			} else {
				outpose->append_residue_by_bond( *(upstream_matchres[i]) );
			}
			outpose->pdb_info()->chain( i, orig_upstream_pose_->pdb_info()->chain( upstream_matchres[i]->seqpos() ) );
			outpose->pdb_info()->number( i, orig_upstream_pose_->pdb_info()->number( upstream_matchres[i]->seqpos() ) );
		} else {
			outpose->replace_residue( upstream_matchres[i]->seqpos(), *(upstream_matchres[i]), false );
		}

	}

	assemble_remark_lines( *outpose, upstream_matchres, redundant_upstream_res, one_ds_pose, ex_geom_ids_for_upstream_res );

	//looks like we have to set the pdbinfo obsolete to false
	outpose->pdb_info()->obsolete( false );

	return outpose;
}


std::string
PDBWriter::assemble_outtag(
	utility::vector1< core::conformation::ResidueCOP > const & upstream_matchres
)
{

	std::string signature("");
	for ( core::Size i =1; i <= upstream_matchres.size(); ++i ){
		signature = signature + upstream_matchres[i]->name1() +
			utility::to_string( upstream_matchres[i]->seqpos() );
	}

	std::map< std::string, SizePair >::iterator map_it = signature_map_.find( signature );
	if ( map_it == signature_map_.end() ) {
		core::Size num_this_um = signature_map_.size() + 1;
		signature_map_[ signature ] = SizePair( num_this_um, 1 );
		map_it = signature_map_.find( signature );
	} else {
		map_it->second.second++;
	}

	std::string unique_string( utility::to_string( map_it->second.first ) + "_" +
		signature + "_" + utility::to_string( map_it->second.second ) );

	return prefix_ + "_" + unique_string + "_" + scaf_name_ + "_" + cstfile_name_;
}

}
}
}
