// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
// :noTabs=false:tabSize=4:indentSize=4:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   protocols/match/output/UpstreamCollisionFilter.hh
/// @brief  Implementation for class to filter matches where the upstream residues collide.
/// @author Alex Zanghellini (zanghell@u.washington.edu)
/// @author Andrew Leaver-Fay (aleaverfay@gmail.com), porting to mini

// Unit headers
#include <protocols/match/output/UpstreamCollisionFilter.hh>

// Package headers
#include <protocols/match/BumpGrid.hh>
#include <protocols/match/output/UpstreamHitCacher.hh>

// Project headers
#include <core/conformation/Residue.hh>
#include	<core/pose/Pose.hh>
#include <core/scoring/ScoreFunction.hh>
#include <core/scoring/ScoringManager.hh>
#include <core/scoring/etable/EtableEnergy.hh>
#include <core/scoring/methods/EnergyMethodOptions.hh>

// Utility headers
#include <utility/pointer/ReferenceCount.hh>

// C++ headers
#include <iostream>

namespace protocols {
namespace match {
namespace output {

UpstreamCollisionFilter::UpstreamCollisionFilter(
	UpstreamHitCacherOP coordinate_cacher
) :
	filter_by_lj_( false ),
	wfa_atr_( 0.8 ),
	wfa_rep_( 0.44 ),
	wfa_sol_( 0.6 ),
	lj_cutoff_( 10 ),
	tolerated_overlap_( 0.0 ),
	empty_pose_( new core::pose::Pose ),
	empty_sfxn_( new core::scoring::ScoreFunction ),
	cacher_( coordinate_cacher ),
	etable_energy_( 0 ),
	bump_grid_( new BumpGrid )
{
	std::cout << "Created UpstreamCollisionFilter" << std::endl;
}

UpstreamCollisionFilter::~UpstreamCollisionFilter()
{}

/// @brief Either sphere-overlap checks the atom pairs in the match residues, or
/// evaluates the Etable energies.  Returns false if any atom pair collides more than
/// the cutoff threshold or if the residue pair energy exceeds the energy cutoff.
/// Returns true otherwise.
bool
UpstreamCollisionFilter::passes_filter(
	match const & m
) const
{
	return passes_filter( match_dspos1( m, 1 ) );
}

bool
UpstreamCollisionFilter::passes_filter(
	match_dspos1 const & m
) const
{
	if ( filter_by_lj_ ) {
		using namespace core::scoring;
		EnergyMap emap;
		for ( Size ii = 1; ii < m.upstream_hits.size(); ++ii ) {
			for ( Size jj = ii + 1; jj <= m.upstream_hits.size(); ++jj ) {
				emap[ fa_atr ] = 0; emap[ fa_rep ] = 0; emap[ fa_sol ] = 0;
				etable_energy_->residue_pair_energy(
					*( cacher_->upstream_conformation_for_hit( ii, fake_hit( m.upstream_hits[ ii ] )) ),
					*( cacher_->upstream_conformation_for_hit( jj, fake_hit( m.upstream_hits[ jj ] )) ),
					*empty_pose_, *empty_sfxn_,
					emap );
				Real energy = wfa_atr_ * emap[ fa_atr ] + wfa_rep_ * emap[ fa_rep ] + wfa_sol_ * emap[ fa_sol ];
				if ( energy > lj_cutoff_ ) return false;
			}
		}
		return true;
	} else {
		for ( Size ii = 1; ii < m.upstream_hits.size(); ++ii ) {
			core::conformation::ResidueCOP iires = cacher_->upstream_conformation_for_hit( ii, fake_hit( m.upstream_hits[ ii ] ) );
			Size ii_first_sc = iires->first_sidechain_atom();
			for ( Size jj = ii + 1; jj <= m.upstream_hits.size(); ++jj ) {
				core::conformation::ResidueCOP jjres = cacher_->upstream_conformation_for_hit( jj, fake_hit( m.upstream_hits[ jj ] ) );
				Size jj_first_sc = jjres->first_sidechain_atom();
				for ( Size kk = ii_first_sc; kk <= iires->nheavyatoms(); ++kk ) {
					ProbeRadius kk_rad = probe_radius_for_atom_type( iires->atom_type_index( jj ) );
					for ( Size ll = jj_first_sc; ll <= jjres->nheavyatoms(); ++ll ) {
						ProbeRadius ll_rad = probe_radius_for_atom_type( jjres->atom_type_index( ll ) );
						Real minsep = bump_grid_->required_separation_distance( kk_rad, ll_rad );
						if ( iires->xyz( kk ).distance_squared( jjres->xyz( ll )) < minsep * minsep ) {
							return false;
						}
					}
				}
			}
		}
		return true;
	}
}

void UpstreamCollisionFilter::set_filter_by_lj( bool setting )
{
	filter_by_lj_ = setting;
	if ( filter_by_lj_ ) {
		using namespace core::scoring;
		using namespace core::scoring::etable;
		using namespace core::scoring::methods;
		EnergyMethodOptions eopts;
		etable_energy_ = new EtableEnergy(
			*(ScoringManager::get_instance()->etable( eopts.etable_type() )), eopts );
	}
}

void UpstreamCollisionFilter::set_lj_cutoff( Real setting )
{
	lj_cutoff_ = setting;
}

void UpstreamCollisionFilter::set_lj_atr_weight( Real setting )
{
	wfa_atr_ = setting;
}

void UpstreamCollisionFilter::set_lj_rep_weight( Real setting )
{
	wfa_rep_ = setting;
}

void UpstreamCollisionFilter::set_lj_sol_weight( Real setting )
{
	wfa_sol_ = setting;
}

void UpstreamCollisionFilter::set_tolerated_overlap( Real setting )
{
	tolerated_overlap_ = setting;
	bump_grid_->set_general_overlap_tolerance( tolerated_overlap_ );
}

/*
private:
	bool filter_by_lj_;
	Real wfa_atr_;
	Real wfa_rep_;
	Real wfa_sol_;
	Real lj_cutoff_;
	Real tolerated_overlap_;
	UpstreamCoordinateCacherCOP cacher_;
	core::scoring::methods::ShortRangeTwoBodyEnergyOP   etable_energy_;
};
*/

}
}
}
