// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
// :noTabs=false:tabSize=4:indentSize=4:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   protocols/match/output/UpstreamCollisionFilter.hh
/// @brief  Declaration for class to filter matches where the upstream residues collide.
/// @author Alex Zanghellini (zanghell@u.washington.edu)
/// @author Andrew Leaver-Fay (aleaverfay@gmail.com), porting to mini

#ifndef INCLUDED_protocols_match_output_UpstreamCollisionFilter_HH
#define INCLUDED_protocols_match_output_UpstreamCollisionFilter_HH

// Unit headers
#include <protocols/match/output/UpstreamCollisionFilter.fwd.hh>

// Package headers
#include <protocols/match/BumpGrid.fwd.hh>
#include <protocols/match/output/MatchFilter.hh>
#include <protocols/match/output/UpstreamHitCacher.fwd.hh>

// Project headers
#include	<core/pose/Pose.fwd.hh>
#include <core/scoring/ScoreFunction.fwd.hh>
#include <core/scoring/methods/ShortRangeTwoBodyEnergy.fwd.hh>


// Utility headers
#include <utility/pointer/ReferenceCount.hh>

namespace protocols {
namespace match {
namespace output {

/// @brief This class is used to detect collisions between the upstream residues
/// and filter out matches that have too much collision.  It can perform either
/// hard-sphere collision detection, or score-function (Etable) driven collision
/// detection.  Four command-line flags are read by the MatcherTask to initialize
/// this class:
/// match::filter_colliding_upstream_residues
/// match::upstream_residue_collision_tolerance
/// match::upstream_residue_collision_score_cutoff
/// match::upstream_residue_collision_Wfa_atr
/// match::upstream_residue_collision_Wfa_rep
/// match::upstream_residue_collision_Wfa_sol
class UpstreamCollisionFilter : public MatchFilter {
public:
	typedef core::Real Real;

public:
	UpstreamCollisionFilter();
	UpstreamCollisionFilter( UpstreamHitCacherOP coordinate_cacher );

	virtual
	~UpstreamCollisionFilter();

	/// @brief Returns true if the given match does not contain too much residue-pair collision.
	virtual
	bool
	passes_filter(
		match const & m
	) const;

	/// @brief Returns true if the given match does not contain too much residue-pair collision.
	virtual
	bool
	passes_filter(
		match_dspos1 const & m
	) const;

	void set_filter_by_lj( bool setting );
	void set_lj_cutoff( Real setting );
	void set_lj_atr_weight( Real setting );
	void set_lj_rep_weight( Real setting );
	void set_lj_sol_weight( Real setting );

	void set_tolerated_overlap( Real setting );

private:
	bool filter_by_lj_;
	Real wfa_atr_;
	Real wfa_rep_;
	Real wfa_sol_;
	Real lj_cutoff_;
	Real tolerated_overlap_;

	core::pose::PoseOP empty_pose_;
	core::scoring::ScoreFunctionOP empty_sfxn_;

	UpstreamHitCacherOP cacher_;
	core::scoring::methods::ShortRangeTwoBodyEnergyOP   etable_energy_;
	BumpGridOP bump_grid_;
};


}
}
}

#endif
