// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file protocols/motifs/Motif.cc
/// @brief Implmentation of interaction motifs

#include <core/pose/Pose.hh>
#include <core/conformation/Atom.hh>
#include <core/conformation/Residue.hh>

#include <core/kinematics/Jump.hh>

#include <protocols/motifs/Motif.hh>

#include <utility/string_util.hh>

namespace protocols {
namespace motifs {

// Base class methods

Motif::Motif(
	std::string & resname1,
	std::string & res1_atom1,
	std::string & res1_atom2,
	std::string & res1_atom3,
	std::string & resname2,
	std::string & res2_atom1,
	std::string & res2_atom2,
	std::string & res2_atom3,
	core::kinematics::Jump const & orientation
) :
	restype_name1_( resname1 ),
	res1_atom1_name_( res1_atom1 ),
	res1_atom2_name_( res1_atom2 ),
	res1_atom3_name_( res1_atom3 ),
	restype_name2_( resname2 ),
	res2_atom1_name_( res2_atom1 ),
	res2_atom2_name_( res2_atom2 ),
	res2_atom3_name_( res2_atom3 ),
	forward_jump_( orientation ),
	backward_jump_( orientation.reversed() )
{}

	bool Motif::apply_check(
		core::pose::Pose const & pose,
		core::Size pos
	)
	{
		return( utility::sloppy_same( pose.residue( pos ).name3(), restype_name1_ ) ||
						utility::sloppy_same( pose.residue( pos ).name3(), restype_name2_ ) );
	}

	core::kinematics::Jump const & Motif::get_jump()
	{
		return forward_jump_;
	}

  void Motif::place_residue(
		core::conformation::Residue const & fixed,
		core::conformation::Residue & mobile
	)
	{
		if( utility::sloppy_same( fixed.name3(), restype_name1_ ) &&
				utility::sloppy_same( mobile.name3(), restype_name2_ ) ) {
			std::cout << "Forward transform!\n";
			return place_residue_( fixed, mobile, true );

		} else if ( utility::sloppy_same( fixed.name3(), restype_name2_ ) &&
				utility::sloppy_same( mobile.name3(), restype_name1_ ) ) {
			std::cout << "Reverse transform!\n";
			return place_residue_( fixed, mobile, false );

		} else {
			std::cout << "Bad Mojo! call to Motif::place_residue() with wrong residue(s)!\n";
			std::cout << "Motif wants: " << restype_name1_ << " and " << restype_name2_ << "\n";
			std::cout << "arguments are: " << fixed.name3() << " and " << mobile.name3() << "\n";
			std::cout << "Neither order matches!\n";
		}

		return;
	}

  void Motif::place_residue_(
		core::conformation::Residue const & fixed,
		core::conformation::Residue & mobile,
		bool forward
	)
	{
		core::kinematics::Stub end_stub;
		core::kinematics::Stub mobile_stub;

		if( forward ) {
			// Extract a stub from the fixed residue
			core::kinematics::Stub start_stub(
				fixed.atom( res1_atom2_name_ ).xyz(),
				fixed.atom( res1_atom1_name_ ).xyz(),
				fixed.atom( res1_atom2_name_ ).xyz(),
				fixed.atom( res1_atom3_name_ ).xyz()
			);

			mobile_stub.from_four_points(
				mobile.atom( res2_atom2_name_ ).xyz(),
				mobile.atom( res2_atom1_name_ ).xyz(),
				mobile.atom( res2_atom2_name_ ).xyz(),
				mobile.atom( res2_atom3_name_ ).xyz()
			);

			// Get the stub at the other end of the jump
			forward_jump_.make_jump( start_stub, end_stub );

		} else {
			// Extract a stub from the fixed residue
			core::kinematics::Stub start_stub(
				fixed.atom( res2_atom2_name_ ).xyz(),
				fixed.atom( res2_atom1_name_ ).xyz(),
				fixed.atom( res2_atom2_name_ ).xyz(),
				fixed.atom( res2_atom3_name_ ).xyz()
			);

			mobile_stub.from_four_points(
				mobile.atom( res1_atom2_name_ ).xyz(),
				mobile.atom( res1_atom1_name_ ).xyz(),
				mobile.atom( res1_atom2_name_ ).xyz(),
				mobile.atom( res1_atom3_name_ ).xyz()
			);

			// Get the stub at the other end of the jump
			backward_jump_.make_jump( start_stub, end_stub );
		}

		// Apply to the mobile residue
		for( core::Size i=1, end_i = mobile.natoms() ; i <= end_i ; ++i ) {
			mobile.set_xyz( i, end_stub.local2global( mobile_stub.global2local( mobile.xyz( i ) ) ) );
		}

		return;
	}



// SingleMotif methods

SingleMotif::SingleMotif(
	std::string & resname1,
	std::string & res1_atom1,
	std::string & res1_atom2,
	std::string & res1_atom3,
	std::string & resname2,
	std::string & res2_atom1,
	std::string & res2_atom2,
	std::string & res2_atom3,
	core::kinematics::Jump const & orientation
) :
	Motif( resname1, res1_atom1, res1_atom2, res1_atom3,
					resname2, res2_atom1, res2_atom2, res2_atom3, orientation )
{}

}
}
