// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file
/// @brief
/// @author


#ifndef INCLUDED_protocols_moves_BackboneMover_HH
#define INCLUDED_protocols_moves_BackboneMover_HH

// Unit headers
#include <protocols/moves/BackboneMover.fwd.hh>

// Package headers
#include <protocols/moves/Mover.hh>

#include <core/types.hh>

#include <core/pose/Pose.fwd.hh>

#include <core/kinematics/MoveMap.hh>

#include <core/scoring/ScoreFunction.hh>

#include <core/pack/pack_rotamers.hh>
#include <core/pack/task/PackerTask.hh>

// ObjexxFCL Headers

// C++ Headers
#include <map>
#include <string>

// Utility Headers
#include <utility/pointer/ReferenceCount.hh>

namespace protocols {
namespace moves {

///////////////////////////////////////////////////////////////////////////////
/// BackboneMover class has elements of the MC temperature to do repetitions
/// of bb moves (small, shear, wobble, etc.).
/// @todo change this to some kind of 'protocol' so the MC is managed separately from
///       conformational moves
class BackboneMover : public Mover {

public:
	typedef core::Real Real;

public:

	// empty constructor fills the mover with default values
	// default values from smallmoves.cc of Rosetta++ (small_move_param)
	BackboneMover() : Mover(), temperature_( 0.5 ), nmoves_( 1 ) {
		Mover::type( "BackboneMoverBase" );
		core::kinematics::MoveMapOP movemap = new core::kinematics::MoveMap;
		movemap->set_bb( true ); // allow all backbone residues to move
		movemap_=movemap; // and make this a constant object
		angle_max( 'H', 0.0 ); // helix
		angle_max( 'L', 6.0 ); // other
		angle_max( 'E', 5.0 ); // strand
	}

	BackboneMover(
		core::kinematics::MoveMapCOP movemap_in,
		core::Real temperature_in,
		core::Size nmoves_in
	) : Mover(), movemap_( movemap_in ), temperature_( temperature_in), nmoves_( nmoves_in )
	{
		Mover::type( "BackboneMoverBase" );
		// set default values for angle_max since it is not passed in
		angle_max( 'H', 0.0 ); // helix
		angle_max( 'L', 6.0 ); // other
		angle_max( 'E', 5.0 ); // strand
	}

	/// virtual functions that get overloaded or called from the inheriting classes
	virtual void apply( core::pose::Pose & );

	virtual void setup_list( core::pose::Pose & ) = 0;

	virtual void set_angles( core::Real ) = 0;

	virtual bool make_move( core::pose::Pose & ) = 0;

	void clear();

	bool check_rama();

/// Properties set/get functions
	void temperature( core::Real const temperature_in ) { temperature_ = temperature_in; }
	void nmoves( core::Size const nmoves_in ) { nmoves_ = nmoves_in; }
	core::kinematics::MoveMapCOP movemap() { return movemap_; };

	// Because this function is in a .hh and not a .cc, we must #include <MoveMap.hh> and
	// are not able to #include <MoveMap.fwd.hh> alone.  As a general rule, do not put
	// function definitions in .hh files.  Break that rule only if you're able to demonstrate a
	// genuine inlining performance boost.
	void movemap(core::kinematics::MoveMapCOP new_movemap) { movemap_=new_movemap; };
	void angle_max( char const type, core::Real const angle ) { angle_max_[ type ] = angle; }
	// note pass in by value for one direction assignment.
	void angle_max( std::map< char, core::Real > angle_max_in ) { angle_max_.swap( angle_max_in ); }

	core::Real new_phi() { return new_phi_; }
	core::Real new_psi() { return new_psi_; }

protected:

	core::kinematics::MoveMapCOP movemap_;

	/// controls bias w/which uphill moves are accepted
	core::Real temperature_;

	/// number of positions at which to make moves
	Size nmoves_;

	/// max allowed angle-change as a function of ss type
	std::map< char, core::Real > angle_max_;

	// variables for the apply
	int num_, resnum_, tries_;
	core::Real big_angle_, small_angle_;
	utility::vector1< std::pair< int, core::Real > > pos_list_;
	utility::vector1< int > already_moved_;

	core::Real old_phi_, new_phi_, old_psi_, new_psi_;
	core::Real old_rama_score_, new_rama_score_;
};

///////////////////////////////////////////////////////////////////////////////
/// @brief this mover makes independent random perturbations of phi and psi of residue i.
/// It selects residue i at random among movable residues, and the final torsion angle
/// is subject to a metropolis criterion using the rama score to ensure that
/// only favorable backbone torsion angles are being selected.  The number of perturbations,
/// and the magnitude of perturbations, and the temperature in the rama check, can all
/// be modified.

class SmallMover : public BackboneMover {

public:

	// default constructor
	SmallMover() : BackboneMover() { Mover::type( "SmallMover" ); }

	// constructor with arguments
	SmallMover(
		core::kinematics::MoveMapCOP movemap_in,
		core::Real temperature_in,
		core::Size nmoves_in
	) : BackboneMover( movemap_in, temperature_in, nmoves_in )
	{
		Mover::type( "SmallMover" );
	}

	virtual void setup_list( core::pose::Pose & pose );
	virtual void set_angles( core::Real angle_in );
	virtual bool make_move( core::pose::Pose & pose );

	virtual void test_move( core::pose::Pose & );
};


///////////////////////////////////////////////////////////////////////////////
/// @brief this mover perturbs the phi of residue i and the psi of residue -1 in
/// such a way that they create a 'shearing' effect that minimizes the downstream
/// consequences of this torsional perturbation. The final torsion angle
/// is subject to a metropolis criterion using the rama score to ensure that
/// only favorable backbone torsion angles are being selected.  The number of
/// perturbations, and the magnitude of perturbations, and the temperature in the
/// rama check, can all be modified.
class ShearMover : public BackboneMover {

public:

	// default constructor
	ShearMover() : BackboneMover() { Mover::type( "ShearMover" );	}

	// constructor with arguments
	ShearMover(
		core::kinematics::MoveMapCOP movemap_in,
		core::Real temperature_in,
		core::Size nmoves_in
	) : BackboneMover( movemap_in, temperature_in, nmoves_in )
	{
		Mover::type( "ShearMover" );
	}

	virtual void setup_list( core::pose::Pose & pose );
	virtual void set_angles( core::Real angle_in );
	virtual bool make_move( core::pose::Pose & pose );

	virtual void test_move( core::pose::Pose & );
};
} // moves
} // protocols


#endif //INCLUDED_protocols_moves_BackboneMover_HH

