// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file protocols/moves/BackrubMover.hh
/// @brief definition of BackrubMover class and functions
/// @author Colin A. Smith (colin.smith@ucsf.edu)


#ifndef INCLUDED_protocols_moves_BackrubMover_HH
#define INCLUDED_protocols_moves_BackrubMover_HH

// Core Headers
#include <core/pose/Pose.fwd.hh>
#include <core/kinematics/tree/Atom.fwd.hh>

// Protocols Headers
#include <protocols/branch_angle/BranchAngleOptimizer.hh>
#include <protocols/moves/Mover.hh>
#include <protocols/moves/BackrubSegment.hh>

// Utility Headers
#include <utility/vector0.hh>
#include <utility/vector1.hh>
#include <numeric/conversions.hh>
#include <utility/keys/Key3Vector.hh>

// Numeric Headers
#include <numeric/IntervalSet.fwd.hh>

// C++ Headers
#include <map>

namespace protocols {
namespace moves {


/// @brief class for applying backrub moves to arbitrary protein segments
class BackrubMover : public Mover {

public:

	BackrubMover():
		max_bond_angle_(numeric::conversions::radians(10.)),
		max_angle_disp_4_(numeric::conversions::radians(40.)),
		max_angle_disp_7_(numeric::conversions::radians(20.)),
		max_angle_disp_slope_(numeric::conversions::radians(-1./3.))
	{}

	virtual MoverOP clone() const;
	virtual MoverOP fresh_instance() const;

	// @brief apply the backrub move to a Pose object
	void
	apply(
		Pose & pose
	);

	/// @brief get the segment ID for a given starting and ending atom
	/// @detailed
	/// if the segment ID does not exist, 0 is returned
	Size
	segment_id(
		core::id::AtomID start_atomid,
		core::id::AtomID end_atomid
	)
	{
		for (Size i = 1; i <= segments_.size(); ++i) {
			if (segments_[i].start_atomid() == start_atomid && segments_[i].end_atomid() == end_atomid) return i;
		}

		return 0;
	}

	/// @brief determine whether a given segment exists
	bool
	has_segment(
		core::id::AtomID start_atomid,
		core::id::AtomID end_atomid
	)
	{
		return segment_id(start_atomid, end_atomid);
	}

	/// @brief remove all segements
	void
	clear_segments()
	{
		segments_.clear();
	}

	/// @brief get the number of segments
	core::Size
	num_segments()
	{
		return segments_.size();
	}

	/// @brief add a segment to the mover
	Size
	add_segment(
		core::id::AtomID start_atomid,
		core::id::AtomID end_atomid,
		core::Real max_angle_disp = 0
	);

	/// @brief add segments between a list of mainchain atomids
	core::Size
	add_mainchain_segments(
		utility::vector1<core::id::AtomID> atomids,
		core::Size min_atoms,
		core::Size max_atoms
	);

	/// @brief add segments between mainchain atoms in contiguous stretches of residues
	core::Size
	add_mainchain_segments(
		utility::vector1<core::Size> resnums,
		utility::vector1<std::string> atomnames,
		core::Size min_atoms,
		core::Size max_atoms
	);

	/// @brief add segments between mainchain atoms using command line options
	core::Size
	add_mainchain_segments_from_options();

	/// @breif optimize branching atoms for all segment pivot atoms
	void
	optimize_branch_angles(
		Pose & pose
	);

	/// @brief do a manual rotation about the given segment
	void
	rotate_segment(
		Pose & pose,
		Size segment_id,
		core::Real angle,
		utility::vector0<core::Real> & start_constants,
		utility::vector0<core::Real> & end_constants
	);

	/// @brief get a random angle for a segment subject to bond angle and rotation constraints
	core::Real
	random_angle(
		Pose & pose,
		Size segment_id,
		utility::vector0<core::Real> & start_constants,
		utility::vector0<core::Real> & end_constants
	);

	/// @brief get the branch angle optimizer stored in the mover
	protocols::branch_angle::BranchAngleOptimizer &
	branchopt()
	{
		return branchopt_;
	}

	/// @brief get id the last backrub segment moved
	core::Size
	last_segment_id() const;

	/// @brief get the name of the atom at the start of the last segment
	std::string
	last_start_atom_name() const;

	/// @brief get the name of the atom at the end of the last segment
	std::string
	last_end_atom_name() const;

	/// @brief update string describing the move type
	void
	update_type();

private:

	utility::vector1<BackrubSegment> segments_;
	protocols::branch_angle::BranchAngleOptimizer branchopt_;
	std::map<BackrubSegment::BondAngleKey, core::Size> bond_angle_map_;
	core::Real max_bond_angle_;
	core::Real max_angle_disp_4_;
	core::Real max_angle_disp_7_;
	core::Real max_angle_disp_slope_;
	core::Size last_segment_id_;
	std::string last_start_atom_name_;
	std::string last_end_atom_name_;
};

/// @brief calculate constants necessary for calculating internal angles/derivatives
void
backrub_rotation_constants(
	core::kinematics::tree::Atom const * PM2_atom,
	core::kinematics::tree::Atom const * PM1_atom,
	core::kinematics::tree::Atom const * P_atom,
	core::kinematics::tree::Atom const * PP1_atom,
	core::kinematics::tree::Atom const * PP2_atom,
	core::kinematics::tree::Atom const * REF_atom,
	utility::vector0<double> & constants,
	core::Real const alpha_min = 0,
	core::Real const alpha_max = numeric::NumericTraits<core::Real>::pi(),
	numeric::IntervalSet<core::Real> *tau_intervals = NULL
);

/// @brief calculate internal coordinate values for any tau value
void
backrub_rotation_angles(
	utility::vector0<core::Real> const & constants,
	core::Real const tau,
	core::Real & bondange,
	core::Real & torsion1,
	core::Real & torsion2
);

} // moves
} // protocols

#endif
