// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file moves/CDRH3Modeler.cc
/// @brief models CDR H3 loop using loop modeling
/// @detailed
/// @author Aroop Sircar (aroopsircar@yahoo.com)

// Rosetta Headers
#include <core/chemical/ChemicalManager.fwd.hh>
#include <core/conformation/Conformation.hh>
#include <core/fragment/BBTorsionSRFD.hh>
#include <core/fragment/ConstantLengthFragSet.hh>
#include <core/fragment/FragData.hh>
#include <core/id/AtomID_Map.Pose.hh>
#include <core/id/types.hh>
#include <core/io/pdb/pose_io.hh>
#include <core/kinematics/FoldTree.hh>
#include <core/options/util.hh>
#include <core/pack/task/PackerTask.hh>
#include <core/pack/task/TaskFactory.hh>
#include <core/pose/Pose.hh>
#include <core/pose/util.hh>
#include <core/scoring/rms_util.hh>
#include <core/scoring/ScoreFunction.hh>
#include <core/scoring/ScoreFunctionFactory.hh>
#include <core/util/Tracer.hh>

#include <numeric/random/random.hh>
#include <numeric/xyz.functions.hh>
#include <numeric/xyz.io.hh>

#include <protocols/loops/loops_main.hh>
#include <protocols/loops/LoopClass.hh>
#include <protocols/loops/LoopMover.fwd.hh>
#include <protocols/loops/LoopMover.hh>
#include <protocols/loops/LoopRelaxMover.hh>
#include <protocols/moves/CDRH3Modeler.hh>
#include <protocols/moves/MoverContainer.hh>
//#include <protocols/moves/ResidueMover.hh>
#include <protocols/moves/ReturnSidechainMover.hh>
#include <protocols/moves/SwitchResidueTypeSetMover.hh>


#include <utility/exit.hh>
#include <utility/io/izstream.hh>
#include <utility/pointer/owning_ptr.hh>
#include <utility/vector1.functions.hh>

// option key includes
#include <core/options/keys/antibody.OptionKeys.gen.hh>
#include <core/options/keys/loops.OptionKeys.gen.hh>

static numeric::random::RandomGenerator RG(11141980);

static core::util::Tracer TR("protocols.moves.CDRH3Modeler");

namespace protocols {
	namespace moves {

		CDRH3Modeler::CDRH3Modeler():Mover( "CDRH3Modeler" ){
			set_default();
		} // CDRH3Modeler default constructor

		void CDRH3Modeler::set_default() {
			using namespace core::options;
			using namespace OptionKeys;
			do_h3_modeling_ = false;
			do_h3_modeling_ = option[ OptionKeys::antibody::model_h3 ]();
			if( do_h3_modeling_ )
				TR << "Setting up default settings for CDR H3 Modeler" << std::endl;
			else
				TR << "Not modeling CDR H3 loop" << std::endl;
			base_ = 1;
			c_ter_stem_ = 3;
		} // CDRH3Modeler set_default

		void CDRH3Modeler::apply( core::pose::Pose & pose_in )
		{
			if( !do_h3_modeling_ )
				return;

			pose_in.dump_pdb( "grafted.pdb" );
			TR <<  "Modeling CDR H3 loop" << std::endl;

			core::pose::Pose start_pose = pose_in;
			antibody::Antibody antibody_in( pose_in );

			Size framework_loop_begin( antibody_in.cdr_h3_start );
			Size frmrk_loop_end_plus_one( antibody_in.cdr_h3_end + 1 );
			Size framework_loop_size = (frmrk_loop_end_plus_one-framework_loop_begin) + 1;
			Size cutpoint = framework_loop_begin + 1;
			loops::Loop cdr_h3( framework_loop_begin, frmrk_loop_end_plus_one,
													cutpoint,	0, true );
			simple_one_loop_fold_tree( antibody_in.Fv, cdr_h3 );

			// silly hack to make extended loops to work
			loops::Loops cdr_h3_loop_list;
			cdr_h3_loop_list.add_loop( cdr_h3 );
			loops::LoopMoverOP my_loop_move( new loops::LoopMover(cdr_h3_loop_list));
			my_loop_move->set_extended_torsions( antibody_in.Fv, cdr_h3 );
			my_loop_move->apply( antibody_in.Fv );

			Size unaligned_cdr_loop_begin(0), unaligned_cdr_loop_end(0);
			core::io::pdb::pose_from_pdb( template_pose_, "hfr.pdb" );
			antibody::Antibody hfr_template( template_pose_, "h3" );
			unaligned_cdr_loop_begin = hfr_template.current_start;
			unaligned_cdr_loop_end = hfr_template.current_end;

			antibody_in.Fv.set_psi(framework_loop_begin-1,
				template_pose_.psi( unaligned_cdr_loop_begin-1));
			antibody_in.Fv.set_omega(framework_loop_begin-1,
				template_pose_.omega( unaligned_cdr_loop_begin-1));

			Size modified_framework_loop_end = frmrk_loop_end_plus_one -
                                         ( c_ter_stem_ );
			loops::Loop trimmed_cdr_h3( framework_loop_begin,
				modified_framework_loop_end, cutpoint, 0, true );
			loops::Loops trimmed_h3_loop_list;
			trimmed_h3_loop_list.add_loop( trimmed_cdr_h3 );

			antibody::Antibody starting_antibody;
			starting_antibody = antibody_in;
			bool closed_cutpoints( false );
			// saving sidechains for use of include_current
			core::pose::Pose saved_sidechains( antibody_in.Fv );

			// switching to centroid mode
			SwitchResidueTypeSetMover to_centroid( core::chemical::CENTROID );
			SwitchResidueTypeSetMover to_full_atom( core::chemical::FA_STANDARD );
			to_centroid.apply( antibody_in.Fv );

			while( !closed_cutpoints) {
				antibody_in = starting_antibody;
				if( framework_loop_size > 5 )
					antibody_modeling_insert_ter( antibody_in );
				{
					using namespace core::options;
					std::string remodel ( option[ OptionKeys::loops::remodel ]() );
					std::string refine ( option[ OptionKeys::loops::refine ]() );

					protocols::loops::LoopRelaxMover centroid_mover;
					// centroid_mover.frag_libs( frag_libs_ );
					centroid_mover.loops( trimmed_h3_loop_list );
					centroid_mover.remodel( remodel );
					centroid_mover.refine( "no" );
					centroid_mover.apply( antibody_in.Fv );

					antibody_in.Fv.dump_pdb( "intermediate.pdb" );

					// switching to full atom
					SwitchResidueTypeSetMover to_full_atom(core::chemical::FA_STANDARD );
					to_full_atom.apply( antibody_in.Fv );

					//recover sidechains from starting structures
					protocols::moves::ReturnSidechainMover recover_sidechains(
						saved_sidechains );
					recover_sidechains.apply( antibody_in.Fv );

					// High resolution scores
					core::scoring::ScoreFunctionOP antibody_score(
					  core::scoring::ScoreFunctionFactory::create_score_function(
							core::scoring::STANDARD_WTS ) );

					//set up packer
					bool include_current( true );
					// Packer
					moves::PackRotamersMoverOP packer;
					core::pack::task::PackerTaskOP task;
					task = core::pack::task::TaskFactory::create_packer_task(
            antibody_in.Fv );
					task->restrict_to_repacking();
					task->or_include_current( include_current );
					packer = new moves::PackRotamersMover( antibody_score, task );
					packer->apply( antibody_in.Fv );

					protocols::loops::LoopRelaxMover fullatom_mover;
					fullatom_mover.remodel( "no" );
					fullatom_mover.loops( cdr_h3_loop_list );
					fullatom_mover.refine( refine );
					fullatom_mover.apply( antibody_in.Fv );
				}
				closed_cutpoints = cutpoints_separation( antibody_in );
			} // while( ( cut_separation > 1.9 )

			pose_in = antibody_in.Fv;

			// compute full atom score
			core::Real final_score;
			fa_scorefxn_=core::scoring::ScoreFunctionFactory::create_score_function(
				core::scoring::STANDARD_WTS,	core::scoring::SCORE12_PATCH);
			final_score = (*fa_scorefxn_)(pose_in);  // may include constraint score
			core::pose::setPoseExtraScores(
				pose_in, std::string("final_looprelax_score"), final_score	);

			// compute CDR H3 global rms
			core::pose::Pose native_pose;
			if( get_native_pose() )
				native_pose = *get_native_pose();
			else
				native_pose = start_pose;
			setPoseExtraScores(	pose_in, "looprms",
													loops::loop_rmsd( native_pose, pose_in,
																						cdr_h3_loop_list ) );
		} // CDRH3Modeler::apply()

		void CDRH3Modeler::store_H3_cter_fragment(
			utility::vector1< core::fragment::FragData > & base_library_in ) {
			H3_base_library = base_library_in;
			return;
		}

		void CDRH3Modeler::store_frags(
      utility::vector1< core::fragment::FragSetOP > new_libs ) {
			frag_libs_ = new_libs;
		}

		void simple_one_loop_fold_tree(
			core::pose::Pose & pose_in,
			loops::Loop const & loop	) {
			using namespace core::kinematics;
			//setup fold tree for this loop
			FoldTree f;
			f.clear();
			core::Size nres = pose_in.total_residue();
			core::Size jumppoint1 = loop.start() - 1;
			core::Size jumppoint2 = loop.stop() + 1;

			if( jumppoint1 < 1 )   jumppoint1 = 1;
			if( jumppoint2 > nres) jumppoint2 = nres;

			f.add_edge( 1, jumppoint1, Edge::PEPTIDE );
			f.add_edge( jumppoint1, loop.cut(), Edge::PEPTIDE );
			f.add_edge( loop.cut() + 1, jumppoint2, Edge::PEPTIDE );
			f.add_edge( jumppoint2, nres, Edge::PEPTIDE );
			f.add_edge( jumppoint1, jumppoint2, 1 );
			f.reorder( 1 );

			pose_in.fold_tree( f );
			return;
		}

		void read_H3_cter_fragment(
			antibody::Antibody & antibody_in,
			utility::vector1< core::fragment::FragData > & H3_base_library ) {
			using namespace core::fragment;

			bool is_kinked( antibody_in.kinked_ );
			bool is_extended( antibody_in.extended_ );

			// extract single letter aa codes for the chopped loop residues
			core::Size cdr_h3_size = ( antibody_in.cdr_h3_end -
																 antibody_in.cdr_h3_start ) + 1;
			utility::vector1< char > aa_1name;
			for( core::Size ii = antibody_in.cdr_h3_start - 2;
					 ii <= ( antibody_in.cdr_h3_start - 2 ) + cdr_h3_size + 3; ++ii )
				aa_1name.push_back( antibody_in.Fv_sequence_[ii] );

			// used only when no length & kink match are found
			utility::vector1< FragData > H3_base_library_seq_kink;

			// used only when no (length & kink) or (length & seq) are found
			utility::vector1< FragData > H3_base_library_kink;

			std::string H3_ter_library_filename;
			// file is read in from where other contraints are supposed to exist
			H3_ter_library_filename = "H3_CTERM";

			// Read the file defined by command line option
			utility::io::izstream H3_ter_library_stream( H3_ter_library_filename );

			// Check to see if file exists
			if ( !H3_ter_library_stream ) {
				TR << "[Error]: Could not open H3 base library file: "
					 << H3_ter_library_filename << std::endl
					 << "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX" << std::endl;
				std::exit( EXIT_FAILURE );
			}

			std::string pdb_name;
			std::string res_no;
			char res_name;
			core::Real phi(0.0);
			core::Real psi(0.0);
			core::Real omega(0.0);
			core::Size H3_length(0);
			core::Real resolution(0.0);
			std::string base_type;

			core::Size pdb_H3_length = cdr_h3_size;
			core::Size h3_base_frag_size(4);
			bool end_not_reached(true);
			while(end_not_reached){
				bool seq_match( true );
				bool kink_match( false );

				FragData f;
				f.set_valid( true );

				for ( core::Size i = 1; i <= h3_base_frag_size; ++i ) {
					H3_ter_library_stream >> pdb_name
																>> res_no
																>> res_name
																>> omega
																>> phi
																>> psi
																>> H3_length
																>> resolution
																>> base_type
																>> std::skipws;
					if(H3_ter_library_stream.eof()) {
						end_not_reached = false;
						break;
					}
					if( res_name != aa_1name[aa_1name.size() - 5 + i] )
							seq_match = false;

					utility::pointer::owning_ptr< BBTorsionSRFD > res_torsions(
						new BBTorsionSRFD( 3, 'L', res_name ) ); // 3 protein torsions
					// ugly numbers 1-3, but pose.set_phi also uses explicit numbers
					res_torsions->set_torsion   ( 1, phi   );
					res_torsions->set_torsion   ( 2, psi   );
					res_torsions->set_torsion   ( 3, omega );
					res_torsions->set_secstruct ( 'L' );
					f.add_residue( res_torsions );
				}

				if( is_kinked && base_type == "KINK" )
					kink_match = true;
				else if( is_extended && base_type == "EXTENDED" )
					kink_match = true;
				else if( !is_kinked && !is_extended && base_type == "NEUTRAL" )
					kink_match = true;
				if(end_not_reached && (H3_length == pdb_H3_length)
					 && kink_match == true)
					H3_base_library.push_back( f );
				if(end_not_reached && (seq_match == true) && kink_match == true)
					H3_base_library_seq_kink.push_back( f );
				if(end_not_reached && kink_match == true)
					H3_base_library_kink.push_back( f );
			}

			H3_ter_library_stream.close();
			H3_ter_library_stream.clear();

			// if no match found based on sequence and kink match criterion
			// then choose based on size and kink match criterion
			// if still no match, then choose based only on kink
			if( H3_base_library.size() == 0 )
				H3_base_library = H3_base_library_seq_kink;
			if( H3_base_library.size() == 0 )
				H3_base_library = H3_base_library_kink;

			return;
		}

		void CDRH3Modeler::antibody_modeling_insert_ter(
			antibody::Antibody & antibody_in ) {

			Size loop_begin(0), loop_end(0), cutpoint(0), random_H3_ter(0);
			utility::vector1< core::fragment::FragData >::const_iterator H3_ter;

			loop_begin = antibody_in.cdr_h3_start;
			cutpoint = antibody_in.cdr_h3_start + 1;
			random_H3_ter = RG.random_range( 1, H3_base_library.size() );
			H3_ter = H3_base_library.begin();

			loop_end = antibody_in.cdr_h3_end + 1;

			// Storing source status
			// FArray1D_bool bb_move_old,chi_move_old,jump_move_old;
			// framework_pose.retrieve_allow_move(bb_move_old, chi_move_old, jump_move_old);
			// Initial fold tree
			// Fold_tree const input_tree( antibody_in.Fv.fold_tree() );

			// allow backbone moves
			//for( Size i = loop_begin; i <= loop_end; i++ )
			//	framework_pose.set_allow_bb_move( i, true );

			loops::Loop cdr_h3( loop_begin, loop_end, cutpoint,	0, true );
			simple_one_loop_fold_tree( antibody_in.Fv, cdr_h3 );

			//framework_pose.one_jump_tree( nres, loop_begin-1, loop_end+1, cutpoint, 1);
			//framework_pose.set_allow_jump_move(1,false);

			// choosing a base randomly
			H3_ter = H3_ter + random_H3_ter;

			//inserting base dihedrals
			if( (antibody_in.cdr_h3_end - 2) <= antibody_in.cdr_h3_start )
				TR << "H3 LOOP IS TOO SHORT: CAN NOT USE N-TERM INFORMATION"
					 << std::endl;
			else
				H3_ter->apply( antibody_in.Fv, antibody_in.cdr_h3_end - 2,
											 antibody_in.cdr_h3_end + 1 );

			// Restoring pose stuff
			//framework_pose.set_fold_tree( input_tree ); // initial tree
			//framework_pose.set_allow_jump_move( jump_move_old );//restore the jump move
			//framework_pose.set_allow_bb_move( bb_move_old );
			//framework_pose.set_allow_chi_move( chi_move_old );
			return;
		}

		bool CDRH3Modeler::cutpoints_separation(
  		antibody::Antibody & antibody_in ) {

			bool closed_cutpoints = true;

			for( loops::Loops::const_iterator it=antibody_in.all_cdr_loops.begin(),
						 it_end=antibody_in.all_cdr_loops.end(),
						 it_next; it != it_end; ++it ) {
				Size cutpoint   = it->cut();
				core::Real separation = 10.00; // an unlikely high number
				separation = cutpoint_separation( antibody_in.Fv, cutpoint );

				if( separation > 1.9 ) {
					closed_cutpoints = false;
					break;
				}
			}
			return( closed_cutpoints );
		} // cutpoints_separation

		core::Real CDRH3Modeler::cutpoint_separation(
  		core::pose::Pose & pose_in,
			Size cutpoint ) {

			Size const N ( 1 ); // N atom
			Size const C ( 3 ); // C atom

			// Coordinates of the C atom of cutpoint res and N atom of res cutpoint+1
			numeric::xyzVector_float peptide_C(pose_in.residue( cutpoint ).xyz( C )),
				peptide_N( pose_in.residue( cutpoint + 1 ).xyz( N ) );
			core::Real cutpoint_separation=distance(peptide_C, peptide_N);

			return( cutpoint_separation );
		} // cutpoint_separation


	}  // namespace moves
}  // namespace protocols
