// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file moves/GraftMover.cc
/// @brief grafts a cdr onto the template of an antibody framework
/// @detailed
/// @author Aroop Sircar (aroopsircar@yahoo.com)

// Rosetta Headers
#include <core/chemical/ChemicalManager.fwd.hh>
#include <core/conformation/Conformation.hh>
#include <core/id/AtomID_Map.Pose.hh>
#include <core/id/types.hh>
#include <core/io/pdb/pose_io.hh>
#include <core/options/util.hh>
#include <core/pack/task/PackerTask.hh>
#include <core/pack/task/TaskFactory.hh>
#include <core/pose/Pose.hh>
#include <core/scoring/rms_util.hh>
#include <core/scoring/ScoreFunction.hh>
#include <core/scoring/ScoreFunctionFactory.hh>
#include <core/util/Tracer.hh>

#include <numeric/xyz.functions.hh>
#include <numeric/xyz.io.hh>

#include <protocols/antibody/AntibodyClass.hh>
#include <protocols/loops/loops_main.hh>
#include <protocols/loops/LoopMover.fwd.hh>
#include <protocols/loops/LoopMover.hh>
#include <protocols/loops/util.hh>
#include <protocols/moves/CDRH3Modeler.hh>
#include <protocols/moves/GraftMover.hh>
#include <protocols/moves/MoverContainer.hh>
#include <protocols/moves/SwitchResidueTypeSetMover.hh>
#include <protocols/moves/ReturnSidechainMover.hh>


#include <utility/exit.hh>
#include <utility/vector1.functions.hh>

// option key includes

#include <core/options/keys/out.OptionKeys.gen.hh>
#include <core/options/keys/cluster.OptionKeys.gen.hh>
#include <core/options/keys/run.OptionKeys.gen.hh>
#include <core/options/keys/antibody.OptionKeys.gen.hh>

static core::util::Tracer TR("protocols.moves.GraftMover");
static core::util::Tracer TRO("protocols.moves.GraftOneMover");

namespace protocols {
	namespace moves {

		GraftMover::GraftMover():Mover( "GraftMover" ){
			set_default();
		} // GraftMover default constructor

		void GraftMover::set_default() {
			using namespace core::options;
			using namespace OptionKeys;
			TR <<  "Setting up default settings" << std::endl;
			graft_l1_ = option[ OptionKeys::antibody::graft_l1 ]();
			graft_l2_ = option[ OptionKeys::antibody::graft_l2 ]();
			graft_l3_ = option[ OptionKeys::antibody::graft_l3 ]();
			graft_h1_ = option[ OptionKeys::antibody::graft_h1 ]();
			graft_h2_ = option[ OptionKeys::antibody::graft_h2 ]();
			graft_h3_ = option[ OptionKeys::antibody::graft_h3 ]();
		} // GraftMover set_default

		void GraftMover::apply( core::pose::Pose & pose_in )
		{
			TR <<  "Grafting designated CDRs" << std::endl;

			antibody::Antibody antibody_in( pose_in );
			SequenceMoverOP graft_sequence ( new SequenceMover() );
			Size nres = pose_in.total_residue();

			// Storing secondary structure
			utility::vector1<char> secondary_struct_storage;
			for( Size i = 1; i <= nres; i++ )
				secondary_struct_storage.push_back( pose_in.secstruct( i ) );

			if ( graft_l1_ ) {
				GraftOneMoverOP graftone_l1( new GraftOneMover(
        antibody_in.cdr_l1_start,antibody_in.cdr_l1_end,"l1" ));
				graft_sequence->add_mover( graftone_l1 );
			}
			if ( graft_l2_ ) {
				GraftOneMoverOP graftone_l2( new GraftOneMover(
        antibody_in.cdr_l2_start,antibody_in.cdr_l2_end,"l2" ));
				graft_sequence->add_mover( graftone_l2 );
			}
			if ( graft_l3_ ) {
				GraftOneMoverOP graftone_l3( new GraftOneMover(
			  antibody_in.cdr_l3_start,antibody_in.cdr_l3_end,"l3" ));
				graft_sequence->add_mover( graftone_l3 );
			}
			if ( graft_h1_ ) {
				GraftOneMoverOP graftone_h1( new GraftOneMover(
        antibody_in.cdr_h1_start,antibody_in.cdr_h1_end,"h1" ));
				graft_sequence->add_mover( graftone_h1 );
			}
			if ( graft_h2_ ) {
				GraftOneMoverOP graftone_h2( new GraftOneMover(
        antibody_in.cdr_h2_start,antibody_in.cdr_h2_end,"h2" ));
				graft_sequence->add_mover( graftone_h2 );
			}
			if ( graft_h3_ ) {
				GraftOneMoverOP graftone_h3( new GraftOneMover(
        antibody_in.cdr_h3_start,antibody_in.cdr_h3_end,"h3" ));
				graft_sequence->add_mover( graftone_h3 );
			}

			graft_sequence->apply(pose_in);

			if ( !graft_h3_ ) {
				TR << "Extending CDR H3" << std::endl;

				Size framework_loop_begin( antibody_in.cdr_h3_start - 1 );
				Size frmrk_loop_end_plus_one( antibody_in.cdr_h3_end + 1 );
				Size cutpoint = framework_loop_begin + 1;
				loops::Loop cdr_h3( framework_loop_begin, frmrk_loop_end_plus_one,
														cutpoint,	0, false );
				protocols::moves::simple_one_loop_fold_tree( pose_in, cdr_h3);

				// silly hack to make extended loops work
				loops::Loops loop_list;
				loop_list.add_loop( cdr_h3 );

				loops::LoopMoverOP my_loop_move( new loops::LoopMover( loop_list ) );
				my_loop_move->set_extended_torsions( pose_in, cdr_h3 );
			}

			if( graft_l1_ || graft_l2_ || graft_l3_ ||
					graft_h1_ || graft_h2_ || graft_h3_ ) {
				// disallowing bogus sidechains while repacking using include_current
				utility::vector1<bool> allow_repack( nres, false );
				for( Size i = 1; i <= nres; i++ ) {
					if( pose_in.secstruct(i) == 'X' )
						allow_repack[i] = true;
					if ( !graft_h2_ && ( i >= antibody_in.cdr_h2_start ) &&
							 ( i <= antibody_in.cdr_h2_end ) )
						allow_repack[i] = true;
					if ( !graft_h3_ && ( i >= antibody_in.cdr_h3_start ) &&
							 ( i <= antibody_in.cdr_h3_end ) )
						allow_repack[i] = true;
				}

				// Recover secondary structures
				for( Size i = 1; i <= nres; i++ )
					pose_in.set_secstruct( i, secondary_struct_storage[ i ] );
				// saving sidechains for use of include_current
				core::pose::Pose saved_sidechains( pose_in );

				// generating centroids for residues devoid of sidechains
				SwitchResidueTypeSetMover to_centroid( core::chemical::CENTROID );
				SwitchResidueTypeSetMover to_full_atom( core::chemical::FA_STANDARD );
				to_centroid.apply( pose_in );
				to_full_atom.apply( pose_in );
				//recover sidechains from starting structures
				protocols::moves::ReturnSidechainMover recover_sidechains(
					saved_sidechains );
				recover_sidechains.apply( pose_in );
				bool include_current( false );
				// High resolution scores
				core::scoring::ScoreFunctionOP antibody_score(
					core::scoring::ScoreFunctionFactory::create_score_function(
            core::scoring::STANDARD_WTS ) );
				set_packer_default( pose_in, antibody_score, include_current );
				packer_->apply( pose_in );

				// Retain coordinates for which full atomic coordinates were
				// available in the input structure. Even for template residues
				// which match corresponding target residues, keep the template
				// sidechians. For the residues which do not have any sidechain
				// coordinates, grab them from a repacked version of the pose
				// generated without using include_current
				for( Size i = 1; i <= nres; i++ ) {
					core::conformation::Residue const & original_rsd(
						saved_sidechains.residue( i ) );
					core::conformation::Residue const & packed_rsd(
						pose_in.residue( i ) );
					if( !allow_repack[i] ) {
						for( Size j=1; j <= packed_rsd.natoms(); ++j ) {
							std::string const & atom_name( packed_rsd.atom_name(j) );
							if( original_rsd.type().has_atom_name( atom_name ) )
								pose_in.set_xyz( core::id::AtomID( packed_rsd.atom_index(
									atom_name),i), original_rsd.xyz( atom_name ) );
						}
					}
				}
				include_current = true;
				set_packer_default( pose_in, antibody_score, include_current );
				packer_->apply( pose_in );
			}

			// align Fv to native.Fv
			core::pose::Pose native_pose;
			if( get_native_pose() )
				native_pose = *get_native_pose();
			else
				native_pose = pose_in;
			antibody::Antibody native_ab( native_pose );
			antibody::Antibody antibody_out( pose_in );
			antibody_out.align_to_native( native_ab );
			pose_in = antibody_out.Fv;
		} // GraftMover::apply()

		void GraftMover::set_packer_default(
			core::pose::Pose & pose_in,
			core::scoring::ScoreFunctionOP scorefxn,
			bool include_current) {

			//set up packer
			core::pack::task::PackerTaskOP task;
			task = core::pack::task::TaskFactory::create_packer_task( pose_in );
			task->restrict_to_repacking();
			task->or_include_current( include_current );
			GraftMover::packer_ = new moves::PackRotamersMover( scorefxn, task );

		} // GraftMover set_packer_default

		GraftOneMover::GraftOneMover(
			Size query_start,
			Size query_end,
			std::string template_name ) : Mover( "GraftOneMover" ){
			query_start_ = query_start;
			query_end_ = query_end;
			template_name_ = template_name;
			set_default();
		} // GraftOneMover default constructor

		void GraftOneMover::set_default() {
			TRO << "Reading in template: " << template_name_ << ".pdb "
					<< std::endl;
			core::io::pdb::pose_from_pdb( template_pose_, template_name_ + ".pdb" );
			antibody::Antibody antibody( template_pose_, template_name_ );
			template_start_ = antibody.current_start;
			template_end_ = antibody.current_end;
		} // GraftOneMover::set_default

		void GraftOneMover::apply( core::pose::Pose & pose_in ) {
			Size const nres( pose_in.total_residue() ); // Total residues
			Size query_size = ( query_end_ - query_start_ ) + 1;

			core::pose::Pose truncated_pose;
			truncated_pose = pose_in;

			// create a sub pose with  5 flanking residues on either side of CDR loop
			truncated_pose.conformation().delete_residue_range_slow(
        query_end_ + 5, nres);
			truncated_pose.conformation().delete_residue_range_slow(
        1, query_start_ - 5);
			truncated_pose.conformation().delete_residue_range_slow(
        5, ( query_size - 1 ) + 5 );

			// create atom map for superimposing 2 flanking resiudes
			core::id::AtomID_Map< core::id::AtomID > atom_map;
			core::id::initialize( atom_map, template_pose_,
														core::id::BOGUS_ATOM_ID );

			for( Size start_stem = 3; start_stem <= 4; start_stem++ ) {
				for( Size j=1; j <= 4; j++ ) {
					core::id::AtomID const id1( j, start_stem );
					core::id::AtomID const id2( j, start_stem );
					atom_map[ id1 ] = id2;
				}
			}
			for( Size end_stem = 4 + query_size + 1; end_stem <=  4 + query_size + 2;
					 end_stem++ ) {
				for( Size j=1; j <= 4; j++ ) {
					core::id::AtomID const id1( j, end_stem );
					core::id::AtomID const id2( j, end_stem - query_size );
					atom_map[ id1 ] = id2;
				}
			}
			core::scoring::superimpose_pose( template_pose_, truncated_pose,
																			 atom_map );


			// copy coordinates of properly oriented template to framework
			for( Size i = query_start_; i <= query_end_; ++i ) {
				core::conformation::Residue const & orientable_rsd(pose_in.residue(i));
				core::conformation::Residue const & template_rsd( template_pose_.
          residue( i - (query_start_ - 5) ) );
				// keeping track of missing rotamers
				if( template_rsd.name() != orientable_rsd.name() )
					pose_in.set_secstruct( i, 'X' );
				for( Size j=1; j <= template_rsd.natoms(); ++j ) {
					std::string const & atom_name( template_rsd.atom_name(j) );
					if( orientable_rsd.type().has_atom_name( atom_name ) )
						pose_in.set_xyz( core::id::AtomID( orientable_rsd.atom_index(
																							 atom_name),i), template_rsd.xyz(
                                               atom_name ) );
					// hack for generating coordinates of amide hydrogen for target
					// residues whose template was a Proline residue (Proline residues
					// do not have the amide hydrogen found ubiquitously in all other
					// amino acids)
					if( template_rsd.name() == "PRO" && orientable_rsd.name() != "PRO"
							&& atom_name == " CD " ){
						core::id::AtomID_Mask missing( false );
						// dimension the missing-atom mask
						core::id::initialize( missing, pose_in );
						missing[ core::id::AtomID(
							pose_in.residue_type(i).atom_index("H"), i ) ] = true;
						pose_in.conformation().fill_missing_atoms( missing );
					}
				}
			}
		} // GraftOneMover::apply
	}  // namespace moves
}  // namespace protocols
