// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file protocols/moves/KinematicWrapper.cc
/// @brief KinematicWrapper methods implemented - this is a mover which simplifies use of KinematicMover loop modeling
/// @author Steven Lewis

// Unit Headers
#include <protocols/moves/KinematicWrapper.hh>
#include <protocols/moves/KinematicMover.hh>

// Package Headers

// Project Headers
#include <core/pose/Pose.hh>

#include <protocols/loops/LoopClass.hh>

// Utility Headers
#include <core/options/option.hh>
#include <core/util/Tracer.hh>
#include <core/types.hh>
#include <utility/exit.hh>
#include <numeric/random/random.hh>

// option key includes

#include <core/options/keys/loops.OptionKeys.gen.hh>



// C++ Headers

using core::util::T;
using core::util::Error;
using core::util::Warning;

static core::util::Tracer TR( "protocols.moves.KinematicWrapper" );
static numeric::random::RandomGenerator RG(171528);

namespace protocols {
namespace moves {

/// @details the responsiblity of apply() is to use the underlying
/// KinematicMover to close the loop, taking care of moving the KinematicMover's
/// begin/middle/end and ensuring that closure does occur.
void KinematicWrapper::apply( core::pose::Pose & pose ){

	core::Size counter(1);
	core::Size alc_start(0), alc_middle(0), alc_end(0); // three pivot residues for kinematic loop closure

	for( /*counter*/; counter<=limit_; ++counter){
		//pick new points for kinematic closure
		alc_start = RG.random_range(loop_begin_, loop_end_-2);
		alc_end = RG.random_range(alc_start+2, loop_end_);
		core::Size middle_offset = (alc_end - alc_start) / 2;
		alc_middle = alc_start + middle_offset;
		//alc_middle = RG.random_range(alc_start+1, alc_end-1); //??what would this do??

		//TR << "cycle " << counter << " pivots begin/middle/end "
		//	 << alc_start << '/' << alc_middle << '/' << alc_end << std::endl;

		kinmover_->set_pivots(alc_start, alc_middle, alc_end);
		kinmover_->apply(pose);
		if (kinmover_->last_move_succeeded() ) break; //we hope this will be the exit point
	}//end for many cycles

	if( counter <= limit_ ){
		TR << "KinematicMover closed in " << counter << " cycles; loop was begin/middle/end "
			 << alc_start << '/' << alc_middle << '/' << alc_end << std::endl;
	}
	else if( counter > limit_ ) TR << "KinematicMover failed to close in " << limit_ << " cycles" << std::endl;
	else {utility_exit_with_message("How did we get here? - KinematicWrapper");}

}//apply

using namespace core::options;
///@brief ctor with Loop
KinematicWrapper::KinematicWrapper(
                                   protocols::moves::KinematicMoverOP kinmover_in,
                                   protocols::loops::Loop loop_in,
																	 core::Size cycles
) : Mover(), kinmover_(kinmover_in), loop_begin_(loop_in.start()), loop_end_(loop_in.stop()),
		limit_( (cycles == 0) ? option[OptionKeys::loops::kinematic_wrapper_cycles].value() : cycles) //option or parameter
{
	Mover::type( "KinematicWrapper" );
	runtime_assert((loop_end_ - loop_begin_ +1) >=3 ); // loop must be long enough for start/cut/stop
	//should we clone (deep copy?) kinmover instead of just copying the OP?
	//NO, because that would complicate simulated annealing
}

///@brief ctor with explicit loop begin/end
KinematicWrapper::KinematicWrapper(
                                   protocols::moves::KinematicMoverOP kinmover_in,
																	 core::Size loop_begin,
																	 core::Size loop_end,
																	 core::Size cycles
) : Mover(), kinmover_(kinmover_in), loop_begin_(loop_begin), loop_end_(loop_end),
		limit_( (cycles == 0) ? option[OptionKeys::loops::kinematic_wrapper_cycles].value() : cycles) //option or parameter
{
	Mover::type( "KinematicWrapper" );
	runtime_assert((loop_end_ - loop_begin_ +1) >=3 );
	//should we clone (deep copy?) kinmover instead of just copying the OP?
	//NO, because that would complicate simulated annealing
}

KinematicWrapper::~KinematicWrapper(){}

}//moves
}//protocols

