// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file MinMover.cc
/// @brief
/// @author ashworth

// Unit headers
#include <protocols/moves/MinMover.hh>

#include <protocols/moves/DataMap.hh>

// Package headers

#include <core/kinematics/MoveMap.hh>
#include <core/optimization/AtomTreeMinimizer.hh>
#include <core/optimization/MinimizerOptions.hh>
#include <core/scoring/ScoreFunction.hh>
#include <core/scoring/ScoreFunctionFactory.hh> // getScoreFunction
#include <core/pose/Pose.fwd.hh>
#include <core/pose/Pose.hh>
#include <core/util/prof.hh>
#include <utility/Tag/Tag.hh>

// ObjexxFCL Headers

// C++ Headers

// Utility Headers

namespace protocols {
namespace moves {

using namespace core;
  using namespace kinematics;
  using namespace optimization;
  using namespace scoring;

// default constructor
// proper lightweight default constructor
MinMover::MinMover()
	: Mover("MinMover"),
		movemap_(0),
		scorefxn_(0),
		min_options_(0),
		threshold_(1000000.0)
{
	min_options_ = new MinimizerOptions( "linmin", 0.01, true, false, false );
}

MinMover::MinMover( std::string const & name )
	: Mover(name),
		movemap_(0),
		scorefxn_(0),
		min_options_(0),
		threshold_(1000000.0)
{
	min_options_ = new MinimizerOptions( "linmin", 0.01, true, false, false );
}

MinMover::~MinMover(){}

// constructor with arguments
MinMover::MinMover(
	MoveMapOP movemap_in,
	ScoreFunctionCOP scorefxn_in,
	std::string const & min_type_in,
	Real tolerance_in,
	bool use_nb_list_in,
	bool deriv_check_in /* = false */,
	bool deriv_check_verbose_in /* = false */
) : Mover("MinMover"),
		movemap_( movemap_in ),
		scorefxn_( scorefxn_in ),
		min_options_(0),
		threshold_(1000000.0)
{
	min_options_ = new MinimizerOptions(
		min_type_in, tolerance_in, use_nb_list_in, deriv_check_in, deriv_check_verbose_in );
}

/// @brief allow non-const access to the internal minimizer options object
MinimizerOptionsOP
MinMover::min_options() {
	return min_options_;
}

/// @brief allow const access to the internal minimizer options object
MinimizerOptionsCOP
MinMover::min_options() const {
	return min_options_;
}

void
MinMover::movemap( MoveMapCOP movemap_in )
{
	runtime_assert( movemap_in );
	movemap_ = new MoveMap( *movemap_in );
}

MoveMapCOP
MinMover::movemap()
{
	return movemap_;
}

void
MinMover::score_function( ScoreFunctionCOP scorefxn_in )
{
	runtime_assert( scorefxn_in );
	scorefxn_ = scorefxn_in;
}

void
MinMover::score_function( ScoreFunction const & scorefxn_in )
{
	scorefxn_ = scorefxn_in.clone();
}

ScoreFunctionCOP
MinMover::score_function() const
{
	return scorefxn_;
}

void
MinMover::min_type( std::string min_type_in ) { min_options_->min_type( min_type_in ); }

void
MinMover::tolerance( Real tolerance_in ) { min_options_->minimize_tolerance( tolerance_in ); }

void
MinMover::nb_list( bool nb_list_in ) { min_options_->use_nblist( nb_list_in ); }

void
MinMover::deriv_check( bool deriv_check_in ) { min_options_->deriv_check( deriv_check_in ); }

void
MinMover::apply( pose::Pose & pose_ )
{
	// lazy default initialization
	if ( ! movemap_ ) movemap_ = new MoveMap;
	if ( ! scorefxn_ ) scorefxn_ = getScoreFunction(); // get a default (INITIALIZED!) ScoreFunction

	PROF_START( util::MINMOVER_APPLY );
	AtomTreeMinimizer minimizer;
	(*scorefxn_)(pose_);
	minimizer.run( pose_, *movemap_, *scorefxn_, *min_options_ );
	PROF_STOP( util::MINMOVER_APPLY );
}

MoverOP MinMover::clone() const { return new MinMover( *this ); }
MoverOP MinMover::fresh_instance() const { return new MinMover; }

void MinMover::parse_my_tag(
	TagPtr const tag,
	DataMap & data,
	Filters_map const & ,
	Movers_map const & ,
	Pose const & )
{
	if ( tag->hasOption("scorefxn") ) {
		std::string const key( tag->getOption<std::string>("scorefxn") );
		if ( data.has("scorefxns",key) ) {
			score_function( data.get< ScoreFunction* >("scorefxns",key) );
		} else {
			utility_exit_with_message("ScoreFunction " + key + " not found in DataMap.");
		}
	}
	if ( tag->hasOption("chi") ) {
		bool const value( tag->getOption<bool>("chi") );
		if ( ! movemap_ ) movemap_ = new MoveMap;
		movemap_->set_chi(value);
	}
	if ( tag->hasOption("bb") ) {
		bool const value( tag->getOption<bool>("bb") );
		if ( ! movemap_ ) movemap_ = new MoveMap;
		movemap_->set_bb(value);
	}
  if ( tag->hasOption("type") ) {
    std::string const type( tag->getOption<std::string>("type") );
		min_type( type );
	}
	if ( tag->hasOption("tolerance") ) {
    Real const tol( tag->getOption<Real>("tolerance") );
  	tolerance( tol );
  }
}

} // moves
} // protocols
