// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file
/// @brief
/// @author


#ifndef INCLUDED_protocols_moves_MonteCarlo_HH
#define INCLUDED_protocols_moves_MonteCarlo_HH


// type headers
#include <core/types.hh>

// unit headers
#include <protocols/moves/MonteCarlo.fwd.hh>
#include <protocols/moves/mc_convergence_checks/ConvergenceCheck.hh>

// package headers
#include <core/pose/Pose.fwd.hh>
#include <core/scoring/ScoreFunction.fwd.hh>

// utility headers
#include <utility/pointer/ReferenceCount.hh>
// #include "utility/basic_sys_util.h"
#include <utility/vector1.hh>

// C++ headers
#include <map>
#include <string>

// Forward declarations

namespace protocols {
namespace moves {


// 	mc_accepted
// 		3 = accepted:score beat low score and last_accepted score
// 		2 = accepted:score beat last_accepted score
// 		1 = thermally accepted: score worse than last_accepted score
// 		0 = not accepted
typedef enum {
	MCA_accepted_score_beat_low=3,
	MCA_accepted_score_beat_last=2,
	MCA_accepted_thermally=1,
	MCA_rejected=0
} MCA; // mc_accepted state

/// @brief This object is responsible for all of the major functions needed
/// in a Monte Carlo simulation.  Its main purpose is to apply the Metropolis
/// Criterion on an pose, based on a ScoreFunction and temperature and the previously
/// accepted pose.  It stores the lowest-energy pose and last-accepted pose.
///It also keeps track of a variety of statistics.
class MonteCarlo : public utility::pointer::ReferenceCount {
public:
	typedef core::scoring::ScoreFunction ScoreFunction;
	typedef core::scoring::ScoreFunctionOP ScoreFunctionOP;
	typedef core::scoring::ScoreFunctionCOP ScoreFunctionCOP;
	typedef core::pose::Pose Pose;
	typedef core::pose::PoseOP PoseOP;
	typedef core::pose::PoseCOP PoseCOP;
	typedef core::Real Real;


public:

	MonteCarlo(
		Pose const & init_pose, // PoseCOP init_pose,
		ScoreFunction const & scorefxn, // ScoreFunctionCOP scorefxn,
		Real const temperature
	);

	/// @brief constructor without Pose -- call reset(pose) before first use
	MonteCarlo(
		ScoreFunction const & scorefxn, // ScoreFunctionCOP scorefxn,
		Real const temperature
	);

	void
	reset_scorefxn(
		Pose const & init_pose,
		ScoreFunction const & scorefxn
	);

	/// @brief sets the temperature that is used in the Metropolis Criterion
	void
	set_temperature( Real const temp );

	Real
	temperature() {
		return temperature_;
	}

	void
	set_autotemp(
		bool const setting,
		core::Real const quench_temp
	);


	/// @brief applies the Metropolis Criterion on the inputted pose based on the
	/// last accepted pose, the temperature, and the ScoreFunction
	bool
	boltzmann(
		Pose & pose,//PoseOP pose,
		std::string const & move_type = "unk"
	);

	/// @brief sets lowest and last_accepted poses to pose -- (does not reset
	/// counters)
	void
	reset( Pose const & pose );

	/// @brief returns the last-accepted pose
	Pose const &
	last_accepted_pose() const
	{
		return *last_accepted_pose_;
	}


	/// @brief returns the lowest-scoring pose
	Pose const &
	lowest_score_pose() const
	{
		return *lowest_score_pose_;
	}

	/// @brief for recovering from a checkpoint
	void set_last_accepted_pose( Pose const & pose );

	/// @brief for recovering from a checkpoint
	void set_lowest_score_pose( Pose const & pose );

	/// @brief attach observer to last accepted conformation
	/// @tparam ConformationObserver any class implementing <tt> void attach_to( Conformation & ) </tt>
	template< typename ConformationObserver >
	void
	attach_observer_to_last_accepted_conformation( ConformationObserver & obs );

	/// @brief attach observer to lowest score conformation
	/// @tparam ConformationObserver any class implementing <tt> void attach_to( Conformation & ) </tt>
	template< typename ConformationObserver >
	void
	attach_observer_to_lowest_score_conformation( ConformationObserver & obs );

	/// @brief attach observer to last accepted pose
	/// @tparam PoseObserver any class implementing <tt> void attach_to( Pose & ) </tt>
	template< typename PoseObserver >
	void
	attach_observer_to_last_accepted_pose( PoseObserver & obs );

	/// @brief attach observer to lowest score pose
	/// @tparam PoseObserver any class implementing <tt> void attach_to( Pose & ) </tt>
	template< typename PoseObserver >
	void
	attach_observer_to_lowest_score_pose( PoseObserver & obs );

	/// @brief Copies the lowest_accepted_ pose into the input pose, as well as into
	/// last_accepted_pose_.
	void
	recover_low( Pose & pose );


	/// @brief set the scorefxn,  re-scores last-accepted and lowest-score pose
	void
	score_function( ScoreFunction const & scorefxn ); // ScoreFunctionCOP scorefxn )


	ScoreFunction const & score_function() const;

	void
	show_scores() const;


	void
	reset_counters();

	void show_state() const;

	void
	show_counters() const;

	/// @brief returns number in all trial-counters
	Size total_trials() const;


	Real
	last_accepted_score() const;

	Real
	lowest_score() const;


	MCA
	mc_accepted() const;

	void clear_poses();// remove last_accepted and lowest_score

	void set_update_boinc( bool setting ){ update_boinc_ = setting; }


	Real total_score_of_last_considered_pose() const { return total_score_of_last_considered_pose_; }

	///@brief trials since last acceptance
	core::Size last_accept() const {
		return last_accept_;
	}

	core::Size heat_after_cycles() const {
		return heat_after_cycles_;
	}

	void set_heat_after_cycles( core::Size setting ) {
		heat_after_cycles_ = setting;
	}

	void push_back( mc_convergence_checks::ConvergenceCheckOP );
	/////////////////////////////////////////////////////////////////////////////
	// private methods
private:

	/// @brief for managing the temperature, if we need to do so
	void
	autotemp_reject();

	void
	autotemp_accept();

	void
	evaluate_convergence_checks( core::pose::Pose const& pose, bool reject, bool final );

public:
	Size
	check_frequency() const {
		return check_frequency_;
	}
	/////////////////////////////////////////////////////////////////////////////
	// data
private:

	/// @brief accepted structure
	PoseOP last_accepted_pose_;

	/// @brief lowest energy structure we've seen
	PoseOP lowest_score_pose_;

	/// @brief acceptance criterion temperature
	core::Real temperature_;

	/// @brief our own internal scoring function
	ScoreFunctionOP score_function_;

	/// @brief for abinitio-style increasing the temperature after a large number of rejects:
	bool autotemp_;
	core::Real quench_temp_;
	int last_accept_;

	/// @brief result of the last call to boltzmann
	MCA mc_accepted_;

	/// @brief diagnostics
	std::map< std::string, int > trial_counter;
	std::map< std::string, int > accept_counter;
	std::map< std::string, core::Real > energy_drop_counter;

	bool update_boinc_;

	Real total_score_of_last_considered_pose_;

	Size heat_after_cycles_;

	utility::vector1< mc_convergence_checks::ConvergenceCheckOP > convergence_checks_;
	Size last_check_;
	Size check_frequency_;
};


} // moves
} // rosetta

#endif
