// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   protocols/moves/MoverFactory.cc
/// @brief
/// @author ashworth

#include <protocols/moves/MoverFactory.hh>
#include <protocols/moves/Mover.hh>

// required for passing to Mover::parse_my_tag
#include <protocols/moves/DataMap.hh>
#include <protocols/filters/Filter.hh>

#include <utility/exit.hh> // runtime_assert, utility_exit_with_message
#include <utility/Tag/Tag.hh>

namespace protocols {
namespace moves {

MoverFactory::MoverFactory()
	: utility::pointer::ReferenceCount()
{
	// no Movers are registered by default
	// they must be registered using the add_type method
}

MoverFactory::~MoverFactory(){}

///@brief add a Mover prototype, using its default type name as the map key
void
MoverFactory::add_type( MoverOP mover )
{
	runtime_assert( mover );
	std::string const type( mover->get_name() );
	if ( type == "UNDEFINED NAME" ) {
		utility_exit_with_message("Can't map derived Mover with undefined type name.");
	}
	mover_prototype_map_[ type ] = mover;
}

///@brief add a Mover prototype, using an arbitrary type name as the map key
void
MoverFactory::add_type( std::string const & type, MoverOP mover )
{
	runtime_assert( mover );
	mover_prototype_map_[ type ] = mover;
}

///@brief return new Mover by key lookup in mover_prototype_map_ (new Mover parses Tag if provided)
MoverOP
MoverFactory::newMover(	std::string const & type )
{
	MoverMap::const_iterator iter( mover_prototype_map_.find( type ) );
	if ( iter != mover_prototype_map_.end() ) {
		if ( ! iter->second ) {
			utility_exit_with_message( "Error: MoverOP prototype for " + type + " is NULL!" );
		}
		// use of cloning method would be faithful to pre-initialized prototypes
		//return iter->second->clone();
		// fresh_instance prevents propagation of pre-initialized prototypes, which may be safer(?)
		return iter->second->fresh_instance();
	}
	else {
		utility_exit_with_message( type + " is not known to the MoverFactory. Was it registered via the add_type method?" );
		return NULL;
	}
}

///@brief return new Mover by Tag parsing
MoverOP
MoverFactory::newMover(
	TagPtr const tag,
	moves::DataMap & data,
	protocols::filters::Filters_map const & filters,
	moves::Movers_map const & movers,
	Pose const & pose )
{
	MoverOP mover( newMover( tag->getName() ) );
	runtime_assert( mover );
	mover->parse_my_tag( tag, data, filters, movers, pose );
	return mover;
}

} //namespace moves
} //namespace protocols
