// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file OutputMovers.hh
/// @brief File to contain classes that deal with output and pdb dumping
/// @detailed
/// @author Monica Berrondo


#ifndef INCLUDED_protocols_moves_OutputMovers_HH
#define INCLUDED_protocols_moves_OutputMovers_HH

// Unit headers
#include <protocols/moves/OutputMovers.fwd.hh>

#include <protocols/moves/Mover.hh>
#include <protocols/moves/MonteCarlo.hh>

#include <core/io/pdb/pose_io.hh>
#include <core/scoring/rms_util.hh>
#include <core/scoring/Energies.hh>
#include <core/scoring/ScoreFunction.hh>

#include <core/util/Tracer.hh>
#include <core/util/prof.hh>

// ObjexxFCL Headers
#include <ObjexxFCL/formatted.o.hh>

// C++ Headers
#include <map>
#include <string>

// Utility Headers

namespace protocols {
namespace moves {

using core::util::T;
using core::util::Error;
using core::util::Warning;

// a mover to dump pdbs within cycles using movers (see DockingHighRes and DockingLowRes for examples on usage
class PDBDumpMover : public Mover
{
public:
	PDBDumpMover(std::string name_in, core::util::Tracer const & tr=core::io::pdb::TR_dump_pdb_dummy)
	 : Mover(), name_(name_in), num_(0)
	{
		tracer_.init(tr);

		Mover::type("PDBDump");
	}

	virtual void apply( core::pose::Pose & pose )
	{
		num_+=1;
		std::string filename ( name_+ObjexxFCL::right_string_of(num_,2,'0')+".pdb" );
		core::io::pdb::traced_dump_pdb( tracer_, pose, filename );
	}

	void name( std::string name_in )
	{
		name_ = name_in;
		clear();
	}

	void clear() { num_=0; }

private:
	std::string name_;
	int num_;

	core::util::Tracer tracer_;

}; // class PDBDumpMover

// allows something profiler output to be printed to the screen during a move cycle
class ProfilerMover : public Mover
{
public:
	ProfilerMover() :
	Mover()
	{
		Mover::type( "Profiler" );
	}

	virtual void apply( core::pose::Pose & /*pose*/ )
	{
		core::util::prof_show();
	}
};

// allows mc.show_scores to be used inside a cycle of movers
class MCShowMover : public Mover
{
public:
	MCShowMover( MonteCarloOP mc_in ) :
	Mover(), mc_( mc_in )
	{
		Mover::type( "MCShow" );
	}

	virtual void apply( core::pose::Pose & pose )
	{
		using namespace ObjexxFCL::fmt;
		mc_->show_scores();
		mc_->score_function()( pose );
		/// Now handled automatically.  mc_->score_function().accumulate_residue_total_energies( pose );
		for ( core::scoring::EnergyMap::const_iterator it=pose.energies().total_energies().begin(),
						it_end = pose.energies().total_energies().end(); it != it_end; ++it ) {
			if ( *it != 0.0 ) {
				std::cout << "total_energy " << core::scoring::ScoreType( it - pose.energies().total_energies().begin() + 1 ) << ' ' << F(12,3,*it) << std::endl;
			}
		}
	}

private:
	// the monte carlo that needs to be shown
	MonteCarloOP mc_;
};
} // moves
} // protocols

#endif //INCLUDED_protocols_moves_OutputMovers_HH
