// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file
/// @brief
/// @author Monica Berrondo
/// @author Modified by Sergey Lyskov

#ifndef INCLUDED_protocols_moves_PackRotamersMover_HH
#define INCLUDED_protocols_moves_PackRotamersMover_HH

// Unit headers
#include <protocols/moves/PackRotamersMover.fwd.hh>
#include <protocols/moves/Mover.hh>

// Project headers
#include <core/types.hh>
#include <core/pack/types.hh>
#include <core/pack/interaction_graph/InteractionGraphBase.fwd.hh>
#include <core/pack/rotamer_set/RotamerSets.fwd.hh>
#include <core/pack/task/PackerTask.fwd.hh>
#include <core/pack/task/TaskFactory.fwd.hh>
#include <core/pose/Pose.fwd.hh>
#include <core/scoring/ScoreFunction.fwd.hh>
#include <protocols/filters/Filter.fwd.hh>

#include <utility/Tag/Tag.fwd.hh>

#include <vector> // for rot_to_pack

namespace protocols {
namespace moves {

/// @brief a mover that packs the side-chains using a rotamer library.  It uses
/// a ScoreFunction for packing and either a PackerTask, or a TaskFactory that generates
/// a PackerTask for instructions on what rotamer sets are allowed at each residue
/// position during packing.
class PackRotamersMover : public Mover {
/// @brief please derive from PackRotamersMover instead of attempting to add protocol-specific stuff here!
/// @author ashworth (current form)
public:
	typedef core::pack::interaction_graph::InteractionGraphBaseOP InteractionGraphBaseOP;
	typedef core::pack::interaction_graph::InteractionGraphBaseCOP InteractionGraphBaseCOP;
	typedef core::pack::rotamer_set::RotamerSetsOP RotamerSetsOP;
	typedef core::pack::rotamer_set::RotamerSetsCOP RotamerSetsCOP;
	typedef core::pack::task::PackerTaskCOP PackerTaskCOP;
	typedef core::pack::task::TaskFactoryCOP TaskFactoryCOP;
	typedef core::scoring::ScoreFunctionCOP ScoreFunctionCOP;

public:
	///@brief default constructor
	PackRotamersMover();
	///@brief constructor with typename
	PackRotamersMover( std::string const & );

	PackRotamersMover(
		ScoreFunctionCOP scorefxn,
		PackerTaskCOP task = 0,
		core::Size nloop = 1
	);

	// destructor (important for properly forward-declaring smart-pointer members)
	virtual ~PackRotamersMover();

	// copy constructor
	PackRotamersMover( PackRotamersMover const & other );

	// methods
	virtual void apply( Pose & pose );
	bool task_is_valid( Pose const & pose ) const; // should this be virtual?

	///@brief parse XML (specifically in the context of the parser/scripting scheme)
	virtual void parse_my_tag(
		TagPtr const,
		DataMap &,
		Filters_map const &,
		Movers_map const &,
		Pose const & );

	///@brief parse "scorefxn" XML option (can be employed virtually by derived Packing movers)
	virtual void parse_score_function(
		TagPtr const,
		DataMap const &,
		Filters_map const &,
		Movers_map const &,
		Pose const & );

	///@brief parse "task_operations" XML option (can be employed virtually by derived Packing movers)
	virtual void parse_task_operations(
		TagPtr const,
		DataMap const &,
		Filters_map const &,
		Movers_map const &,
		Pose const & );

	///@brief required in the context of the parser/scripting scheme
	virtual MoverOP fresh_instance() const;
	///@brief required in the context of the parser/scripting scheme
	virtual MoverOP clone() const;

	// setters
	void score_function( ScoreFunctionCOP sf );
	void task_factory( TaskFactoryCOP tf );

	// accessors
	ScoreFunctionCOP score_function() const;
	PackerTaskCOP task() const;
	core::Size nloop() const { return nloop_; }
	TaskFactoryCOP task_factory() const;
	RotamerSetsCOP rotamer_sets() const;
	InteractionGraphBaseCOP ig() const;

protected:
	///@brief get rotamers, energies. Also performs lazy initialization of ScoreFunction, PackerTask.
	virtual void setup( Pose & pose );
	// need a more elegant rot_to_pack implementation than this
	virtual core::pack::PackerEnergy run(
		Pose & pose,
		std::vector< int > rot_to_pack = std::vector< int >()
	) const;
	virtual void note_packertask_settings( Pose const & );

private:
	// pointers to data that are passed in
	ScoreFunctionCOP scorefxn_;
	PackerTaskCOP task_;
	core::Size nloop_;
	TaskFactoryCOP task_factory_;

	// 'really private:' packer data, actually created and owned by this class
	RotamerSetsOP rotamer_sets_;
	InteractionGraphBaseOP ig_;

};

// note: it is better to create new files, instead of adding additional classes here

} // moves
} // protocols

#endif
