// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file
/// @brief
/// @author Monica Berrondo
/// @author Ian Davis
/// @author Ingemar Andre


#ifndef INCLUDED_protocols_moves_RigidBodyMover_HH
#define INCLUDED_protocols_moves_RigidBodyMover_HH

// Unit headers
#include <protocols/moves/RigidBodyMover.fwd.hh>

// Package headers
#include <protocols/moves/Mover.hh>
#include <protocols/geometry/RB_geometry.hh>

#include <core/types.hh>

#include <core/pose/Pose.fwd.hh>

#include <core/kinematics/MoveMap.fwd.hh>

#include <core/chemical/ResidueTypeSet.hh>

#include <core/scoring/ScoreFunction.hh>

#include<core/conformation/symmetry/SymDof.hh>

// ObjexxFCL Headers

// C++ Headers
#include <map>

// Utility Headers
#include <numeric/xyzVector.io.hh>
#include <numeric/random/random.hh>
#include <utility/pointer/ReferenceCount.hh>

namespace protocols {
namespace moves {

/// @brief Partner, which partner gets moved
typedef enum {
	partner_upstream = 1,
	partner_downstream //=2
} Partner;

/// @brief Direction, which direction
typedef enum {
	c2n=-1,
	random=0,
	n2c=1
} Direction;

///////////////////////////////////////////////////////////////////////////////
/// @brief Rigid-body random translate/rotate around centroid of downstream side of a jump.
/// @details We operate on a single jump rather than e.g. a randomly selected
/// jump from the mobile jumps in a MoveMap.
/// If you want a random choice among jumps, put multiple RigidBodyMovers
/// into a RandomMover (which will give you more control, anyway).
class
RigidBodyMover : public Mover {

public:

	// default constructor
	RigidBodyMover() : Mover(), rb_jump_( 1 ), dir_( n2c ), rot_center_( 0.0 )
	{
		Mover::type( "RigidBodyBase" );
	}

	// constructor with arguments
	RigidBodyMover(
		int const rb_jump_in,
		Direction dir_in=n2c
	):
		Mover(),
		rb_jump_( rb_jump_in ), dir_( dir_in ), rot_center_( 0.0 )
	{
		Mover::type( "RigidBodyBase" );
		if ( dir_ == random ) {
			dir_ = ( numeric::random::uniform() < 0.5 ? c2n : n2c );
		} else {
			runtime_assert( dir_ == n2c || dir_ == c2n );
		}
	}


	/// @brief Manual override of rotation center.
	void rot_center( core::Vector const rot_center_in )	{	rot_center_ = rot_center_in; }

	// Can't change jump number after creation b/c it determines center of rotation!
	//void rb_jump_( int const rb_jump_in ) { rb_jump_ = rb_jump_in; }
	int rb_jump() const { return rb_jump_; }

	virtual void apply( core::pose::Pose & pose ) = 0;

	// data
protected:
	int rb_jump_;
	/// direction of folding (n-term to c-term)
	Direction dir_;

	/// center of rotation
	core::Vector rot_center_;
};

// does a perturbation defined by the rotational and translational magnitudes
class
RigidBodyPerturbMover : public RigidBodyMover {

public:
	// default constructor
	RigidBodyPerturbMover() : RigidBodyMover(), rot_mag_( 3.0 ), trans_mag_( 8.0 )
	{
		Mover::type( "RigidBodyPerturb" );
	}

	// constructor with arguments (rb_jump defined)
	RigidBodyPerturbMover(
		int const rb_jump_in,
		core::Real const rot_mag_in,
		core::Real const trans_mag_in,
		Partner const partner_in=partner_downstream,
		bool interface_in=false //rot_center calculated at interface
	);

	// constructor with arguments (overloaded for default jump, but defined rot/mag)
	RigidBodyPerturbMover(
		core::Real const rot_mag_in,
		core::Real const trans_mag_in,
		Partner const partner_in=partner_downstream,
		bool interface_in=false //rot_center calculated at interface
	);

	virtual void apply( core::pose::Pose & pose );

	void rot_magnitude( core::Real const magnitude ) { rot_mag_ = magnitude; }

	void trans_magnitude( core::Real const magnitude ) { trans_mag_ = magnitude; }

	/// @brief Manual override of rotation center.
	void rot_center( core::Vector const /*rot_center_in*/ ) { utility_exit_with_message("Rotation point is automatically determined ONLY"); }

protected:
	/// perturbation magnitudes (rotational and translational)
	core::Real rot_mag_;
	core::Real trans_mag_;
private:
	Partner partner_;
	bool interface_;
};

///@brief does a perturbation defined by the rotational and translational magnitudes
/// 	without setting up the center
///		Can be defined through a move map or with rb_jump
class
RigidBodyPerturbNoCenterMover : public RigidBodyMover{

public:
	// default constructor
	RigidBodyPerturbNoCenterMover() : RigidBodyMover(), rot_mag_( 3.0 ), trans_mag_( 8.0 )
	{
		Mover::type( "RigidBodyPerturbNoCenter" );
	}

	// constructor with arguments (rb_jump defined)
	RigidBodyPerturbNoCenterMover(
		int const rb_jump_in,
		core::Real const rot_mag_in,
		core::Real const trans_mag_in
	);

	// constructor with arguments (rb_jump not defined)
	// movemap used instead
	RigidBodyPerturbNoCenterMover(
		core::pose::Pose const & pose_in,
		core::kinematics::MoveMap const & mm,
		core::Real const rot_mag_in,
		core::Real const trans_mag_in,
		Direction dir_in
	);

	virtual void apply( core::pose::Pose & pose );

	void rot_magnitude( core::Real const magnitude ) { rot_mag_ = magnitude; }

	void trans_magnitude( core::Real const magnitude ) { trans_mag_ = magnitude; }


private:
	/// perturbation magnitudes (rotational and translational)
	core::Real rot_mag_;
	core::Real trans_mag_;
};

class
RigidBodyRandomizeMover : public RigidBodyMover {

public:
	// default constructor
	RigidBodyRandomizeMover() : RigidBodyMover(), partner_( partner_downstream ), phi_angle_(360),psi_angle_(360)
	{
		Mover::type( "RigidBodyRandomize" );
	}

	// constructor with arguments
	RigidBodyRandomizeMover(
		core::pose::Pose const & pose_in,
		int const rb_jump_in=1,
		Partner const partner_in=partner_downstream,
		int phi_angle=360,
		int psi_angle=360
	):
		RigidBodyMover( rb_jump_in ),
		partner_( partner_in ),
		phi_angle_(phi_angle),
		psi_angle_(psi_angle)
	{
		Mover::type( "RigidBodyRandomize" );
		core::Vector upstream_dummy, downstream_dummy;
		protocols::geometry::centroids_by_jump(pose_in, rb_jump_in, upstream_dummy, downstream_dummy );
		if ( partner_in == partner_downstream ) rot_center_ = downstream_dummy;
		else rot_center_ = upstream_dummy;
	}

	virtual void apply( core::pose::Pose & pose );

private:
	Partner partner_; // which partner gets randomized
	core::Size phi_angle_;
	core::Size psi_angle_;

};

// spin about a random axis
class
RigidBodySpinMover : public RigidBodyMover {

public:
	// default constructor
	RigidBodySpinMover() : RigidBodyMover(), spin_axis_( 0.0 )
	{
		Mover::type( "RigidBodySpin" );
	}

	///@brief constructor with arguments
	///       spin axis is initialized to 0 and then calculated during apply()
	RigidBodySpinMover(
		int const rb_jump_in
	):
		RigidBodyMover( rb_jump_in ), spin_axis_( 0.0 ), update_spin_axis_( true )
	{
		Mover::type( "RigidBodySpin" );
	}

	void spin_axis ( core::Vector spin_axis_in )
		{
			spin_axis_ = spin_axis_in;
			update_spin_axis_ = false;
		}

	void rot_center ( core::Vector const rot_center_in )
		{
			rot_center_ = rot_center_in;
			update_spin_axis_ = false;
		}

	virtual void apply( core::pose::Pose & pose );

private:
	core::Vector spin_axis_;
	bool update_spin_axis_;

};

// translate down an axis
class
RigidBodyTransMover : public RigidBodyMover {

public:
	// default constructor
	RigidBodyTransMover() : RigidBodyMover()
	{
		Mover::type( "RigidBodyTrans" );
	}

	// constructor with arguments
	RigidBodyTransMover(
		core::pose::Pose const & pose_in,
		int const rb_jump_in=1
	):
		RigidBodyMover( rb_jump_in )
	{
		Mover::type( "RigidBodyTrans" );
		step_size_ = 1.0;
		core::Vector upstream_dummy, downstream_dummy;
		protocols::geometry::centroids_by_jump(pose_in, rb_jump_in, upstream_dummy, downstream_dummy );
		trans_axis_ = downstream_dummy - upstream_dummy;
	}

	core::Vector & trans_axis() { return trans_axis_; }
	core::Vector trans_axis() const { return trans_axis_; }
	void trans_axis( core::Vector trans_axis_in ) { trans_axis_ = trans_axis_in; }

	void step_size( core::Real step_size_in ) { step_size_ = step_size_in; }

	virtual void apply( core::pose::Pose & pose );

private:
	core::Real step_size_;
	core::Vector trans_axis_;

};


/// @brief Rigid-body move that evenly samples the space within a sphere
class
UniformSphereTransMover : public RigidBodyMover {

public:
	// default constructor
	UniformSphereTransMover() : RigidBodyMover(), step_size_(1)
	{
		Mover::type( "UniformSphereTrans" );
	}

	// constructor with arguments
	UniformSphereTransMover(
		int const rb_jump_in,
		core::Real step_size_in
	):
		RigidBodyMover( rb_jump_in ),
		step_size_( step_size_in )
	{
		Mover::type( "UniformSphereTrans" );
	}

	virtual void apply( core::pose::Pose & pose );

private:
	core::Real step_size_;

};

// Initialize all dofs in the system randomly. Start by rotation angles
// only.
class
RigidBodyDofRandomizeMover : public RigidBodyMover {

public:
  // default constructor
  RigidBodyDofRandomizeMover() :
		RigidBodyMover()
  {
    Mover::type( "RigidBodyDofRandomize" );
  }

  // constructor with arguments
  RigidBodyDofRandomizeMover(
    int const rb_jump_in,
		core::conformation::symmetry::SymDof
  );

  virtual void apply( core::pose::Pose & pose );

private:
	core::conformation::symmetry::SymDof dof_;
};

// Initialize all dofs in the system randomly. Start by rotation angles
// only.
class
RigidBodyDofSeqRandomizeMover : public RigidBodyMover {

public:
  // default constructor
  RigidBodyDofSeqRandomizeMover() :
		RigidBodyMover()
  {
    Mover::type( "RigidBodyDofRandomize" );
  }

	// constructor with arguments
  RigidBodyDofSeqRandomizeMover(
		std::map< Size, core::conformation::symmetry::SymDof > dofs
  );

  virtual void apply( core::pose::Pose & pose );

private:
	std::map< Size, core::conformation::symmetry::SymDof > dofs_;

};


// Translate down axis determined by the available dofs.
// Translations are made along all allowed directions (x,y or z)
// for a selected jump
class
RigidBodyDofTransMover : public RigidBodyMover {

public:
	// default constructor
	RigidBodyDofTransMover() : RigidBodyMover(), jump_dir_(n2c)
	{
		Mover::type( "RigidBodyDofTrans" );
	}

	// constructor with arguments
	RigidBodyDofTransMover(
		core::conformation::symmetry::SymDof dof,
		int const rb_jump_in,
		core::Real step_size
	);

	// constructor with arguments
  RigidBodyDofTransMover(
		std::map< Size, core::conformation::symmetry::SymDof > dofs
  );

	core::Vector & trans_axis() { return trans_axis_; }
	core::Vector trans_axis() const { return trans_axis_; }
	void trans_axis( core::Vector trans_axis_in ) { trans_axis_ = trans_axis_in; }

	void step_size( core::Real step_size_in ) { step_size_ = step_size_in; }

	virtual void apply( core::pose::Pose & pose );

private:
	int jump_dir_;
	core::Real step_size_;
	core::Vector trans_axis_;

};

// Translate down axis determined by the available dofs.
// Translation are made along all allowed directions (x,y or z)
// for all available jumps. Jumps are visited in random order
class
RigidBodyDofSeqTransMover : public RigidBodyMover {

public:
	// default constructor
	RigidBodyDofSeqTransMover() : RigidBodyMover()
	{
		Mover::type( "RigidBodyDofSeqTrans" );
	}

	// constructor with arguments
  RigidBodyDofSeqTransMover(
		std::map< Size, core::conformation::symmetry::SymDof > dofs
  );

	core::Vector & trans_axis() { return trans_axis_; }
  core::Vector trans_axis() const { return trans_axis_; }
	void trans_axis( core::Vector trans_axis_in ) { trans_axis_ = trans_axis_in; }
	void step_size( core::Real step_size_in ) { step_size_ = step_size_in; }

	virtual void apply( core::pose::Pose & pose );

private:
	 /// allowed dofs
  std::map< Size, core::conformation::symmetry::SymDof > dofs_;
	// allowed jumps
  utility::vector1 < int > rb_jumps_;
	core::Real step_size_;
	// silly, stores the direction only!!!
	core::Vector trans_axis_;
};

// Translate down axis determined by the available dofs.
// Translation are made along all allowed directions (x,y or z)
// for a randomly selected jump.
class
RigidBodyDofRandomTransMover : public RigidBodyMover {

public:
	// default constructor
	RigidBodyDofRandomTransMover() : RigidBodyMover()
	{
		Mover::type( "RigidBodyDofRandomTrans" );
	}

	// constructor with arguments
  RigidBodyDofRandomTransMover(
		std::map< Size, core::conformation::symmetry::SymDof > dofs
  );

	core::Vector & trans_axis() { return trans_axis_; }
  core::Vector trans_axis() const { return trans_axis_; }
	void trans_axis( core::Vector trans_axis_in ) { trans_axis_ = trans_axis_in; }
	void step_size( core::Real step_size_in ) { step_size_ = step_size_in; }

	virtual void apply( core::pose::Pose & pose );

private:
	 /// allowed dofs
  std::map< Size, core::conformation::symmetry::SymDof > dofs_;
	// allowed jumps
  utility::vector1 < int > rb_jumps_;
	core::Real step_size_;
	// silly, stores the direction only!!!
	core::Vector trans_axis_;
};




///@brief does a perturbation defined by the rotational and translational magnitudes
///   Allowed dofs are specified by a map
///   Can be defined through a move map or with rb_jump. A single jump is selected
class
RigidBodyDofPerturbMover : public RigidBodyMover{

public:
  // default constructor makes no sense at all!!!
//  RigidBodyDofPerturbMover() : RigidBodyMover(), rot_mag_( 1.0 ), trans_mag_( 3.0 )
//  {
//    Mover::type( "RigidBodyDofPerturbMover" );
//  }

  // constructor with arguments (rb_jump not defined)
  // movemap used instead
  RigidBodyDofPerturbMover(
		std::map< Size, core::conformation::symmetry::SymDof > dofs,
    core::Real const rot_mag_in = 1.0,
    core::Real const trans_mag_in = 3.0
  );

	RigidBodyDofPerturbMover(
    int const rb_jump_in,
    core::conformation::symmetry::SymDof dof,
    core::Real const rot_mag_in = 1.0,
    core::Real const trans_mag_in = 3.0
  );

  virtual void apply( core::pose::Pose & pose );

  void rot_magnitude( core::Real const magnitude ) { rot_mag_ = magnitude; }

  void trans_magnitude( core::Real const magnitude ) { trans_mag_ = magnitude; }

	void dof( core::conformation::symmetry::SymDof dof ) { dof_ = dof; }

private:
	/// allowed dofs
	core::conformation::symmetry::SymDof dof_;
  /// perturbation magnitudes (rotational and translational)
  core::Real rot_mag_;
  core::Real trans_mag_;
};

///@brief does a perturbation defined by the rotational and translational magnitudes
///   Allowed dofs are specified by a map
///   Can be defined through a move map or with rb_jump. All jumps are selected in random order
class
RigidBodyDofSeqPerturbMover : public RigidBodyMover{

public:
  // default constructor makes no sense at all!!!
//  RigidBodyDofSeqPerturbMover() : RigidBodyMover(), rot_mag_( 3.0 ), trans_mag_( 8.0 )
//  {
//    Mover::type( "RigidBodyDofSeqPerturbMover" );
//  }

  // constructor with arguments (rb_jump not defined)
  // movemap used instead
  RigidBodyDofSeqPerturbMover(
		std::map< Size, core::conformation::symmetry::SymDof > dofs,
    core::Real const rot_mag_in = 1.0,
    core::Real const trans_mag_in = 3.0
  );

  virtual void apply( core::pose::Pose & pose );

  void rot_magnitude( core::Real const magnitude ) { rot_mag_ = magnitude; }

  void trans_magnitude( core::Real const magnitude ) { trans_mag_ = magnitude; }

	void dofs( std::map< Size, core::conformation::symmetry::SymDof > dofs ) { dofs_ = dofs; }

private:
	/// allowed dofs
	std::map< Size, core::conformation::symmetry::SymDof > dofs_;
	// allowed jumps
	utility::vector1 < int > rb_jumps_;
  /// perturbation magnitudes (rotational and translational)
  core::Real rot_mag_;
  core::Real trans_mag_;
};

} // moves
} // protocols


#endif //INCLUDED_protocols_moves_RigidBodyMover_HH
