// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file RotamerTrialsMinMover.cc
/// @brief Mover for Rotamer-Trials with Minimization (based on RotamerTrialsMover)
/// @author Barak Raveh

// Unit headers
#include <protocols/moves/RotamerTrialsMinMover.hh>

// Project headers
#include <core/pose/Pose.hh>
#include <core/scoring/Energies.hh>
#include <core/scoring/ScoreFunction.hh>
#include <core/pack/task/PackerTask.hh>
#include <core/pack/task/TaskFactory.hh>

#include <core/pack/rtmin.hh>

#include <protocols/moves/MonteCarlo.hh>

// Utility Headers
#include <core/util/Tracer.hh>
using core::util::T;
using core::util::Error;
using core::util::Warning;

static core::util::Tracer TR("protocols.moves.RotamerTrialsMinMover");

namespace protocols {
namespace moves {

// default constructor
RotamerTrialsMinMover::RotamerTrialsMinMover() : Mover()
{
	Mover::type( "RotamerTrialsMin" );
}

// constructor with arguments
RotamerTrialsMinMover::RotamerTrialsMinMover(
	ScoreFunctionCOP scorefxn_in,
	PackerTask & task_in
) : Mover(), scorefxn_( scorefxn_in ), factory_( NULL )
{
	Mover::type( "RotamerTrialsMin" );
	task_ = task_in.clone();
}

// constructor with arguments
RotamerTrialsMinMover::RotamerTrialsMinMover(
	ScoreFunctionCOP scorefxn_in,
	TaskFactoryCOP factory_in
) : Mover(), scorefxn_( scorefxn_in ), task_( NULL ), factory_( factory_in )
{
	Mover::type( "RotamerTrialsMin" );
}

RotamerTrialsMinMover::~RotamerTrialsMinMover() {}

// setters
void RotamerTrialsMinMover::score_function( core::scoring::ScoreFunctionCOP sf ) { scorefxn_ = sf; }
void RotamerTrialsMinMover::task_factory( core::pack::task::TaskFactoryCOP tf ) { factory_ = tf; }

void
RotamerTrialsMinMover::apply( core::pose::Pose & pose )
{
	//task() contains the call to the TaskFactory
	//TR << *(task(pose)) << std::flush;
	core::pack::rtmin( pose, *scorefxn_, task(pose) );
}

/// @brief read access for derived classes
RotamerTrialsMinMover::ScoreFunctionCOP
RotamerTrialsMinMover::scorefxn() const
{
	return scorefxn_;
}

/// @brief read access for derived classes
RotamerTrialsMinMover::PackerTaskCOP
RotamerTrialsMinMover::task( core::pose::Pose const & pose ) const
{
	//if we have a factory, generate and return a new task
	if(factory_) return factory_->create_task_and_apply_taskoperations( pose );
	//else assert( task_is_valid( pose ) );

	//else return the unsafe one
	return task_;
}


// default constructor
EnergyCutRotamerTrialsMinMover::EnergyCutRotamerTrialsMinMover() :
	RotamerTrialsMinMover()
{
	Mover::type( "EnergyCutRotamerTrialsMin" );
}

// constructor with arguments
EnergyCutRotamerTrialsMinMover::EnergyCutRotamerTrialsMinMover(
	ScoreFunctionCOP scorefxn_in,
	PackerTask & task_in,
	MonteCarloOP mc_in,
	core::Real energycut_in
) : RotamerTrialsMinMover(scorefxn_in, task_in), mc_( mc_in ), energycut_( energycut_in )
{
	Mover::type( "EnergyCutRotamerTrialsMin" );
}

// constructor with arguments
EnergyCutRotamerTrialsMinMover::EnergyCutRotamerTrialsMinMover(
	ScoreFunctionCOP scorefxn_in,
	TaskFactoryCOP factory_in,
	MonteCarloOP mc_in,
	core::Real energycut_in
) : RotamerTrialsMinMover(scorefxn_in, factory_in), mc_( mc_in ), energycut_( energycut_in )
{
	Mover::type( "EnergyCutRotamerTrialsMin" );
}

EnergyCutRotamerTrialsMinMover::~EnergyCutRotamerTrialsMinMover() {}

void
EnergyCutRotamerTrialsMinMover::apply( core::pose::Pose & pose )
{
	PackerTaskOP rottrial_task( task(pose)->clone() );
	( *scorefxn() )(pose);
	setup_energycut_task( pose, *mc_, *rottrial_task );
	/// This call is dangerous.  If sequence or length has changed since task was created, it will crash.
	/// Not a problem if you used a TaskFactory
	core::pack::rtmin( pose, *scorefxn(), rottrial_task );
}


/// @details starting from a fresh task, it reduces the number of residues to be repacked to only
/// those whose energy has increased by energycut_ since the application of the last move.
void
EnergyCutRotamerTrialsMinMover::setup_energycut_task(
	core::pose::Pose const & pose,
	MonteCarlo const & mc,
	core::pack::task::PackerTask & task_in
) const
{
	using namespace core;
	using core::scoring::total_score;

	//Size count_fixed( 0 ), count_repacked( 0 );

	task_in.restrict_to_repacking();

	for ( int i=1, i_end = pose.total_residue(); i<= i_end; ++i ) {
		core::Real const resE ( pose.energies().residue_total_energies(i)[ total_score ] );
		core::Real const lowest_resE( mc.lowest_score_pose().energies().residue_total_energies(i)[ total_score ] );
		core::Real const deltaE ( resE - lowest_resE );
		if ( deltaE < energycut_ ) {
			task_in.nonconst_residue_task(i).prevent_repacking();
			//++count_fixed;
		} else {
			// let this residue be repacked
			//++count_repacked;
		}
	}
}

} // moves
} // protocols
