// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file SetReturningPackRotamersMover.cc
/// @brief A pack mover which returns not just the best packer solution, but all nloop solutions.
/// @author Ron Jacak

// Unit headers
#include <protocols/moves/SetReturningPackRotamersMover.hh>
#include <protocols/moves/PackRotamersMover.hh>

// Project headers
#include <core/pack/task/PackerTask.hh>
#include <core/scoring/ScoreFunction.hh>
#include <core/pose/Pose.hh> // need .hh because we create Pose object in apply()

#include <ObjexxFCL/formatted.o.hh>

#include <core/util/Tracer.hh>

static core::util::Tracer TR("protocols.moves.SetReturningPackRotamersMover");

// Utility Headers

namespace protocols {
namespace moves {

using namespace core;
using namespace scoring;
using namespace pack;


// custom c'tor
SetReturningPackRotamersMover::SetReturningPackRotamersMover(
	ScoreFunctionCOP scorefxn,
	task::PackerTaskCOP task,
	Size ndruns
) :
	PackRotamersMover( scorefxn, task, ndruns ),
	ndruns_(ndruns)
{
	//TR << "Resizing repacked_poses_ to size " << ndruns << std::endl;
	repacked_poses_.resize( ndruns );
}


//
// @begin apply
//
// @brief
// The apply method for SetReturningPackRotamersMover.  Assumes that a valid score function and packer task were passed in.
// Still makes the best packed pose the returned pose, but also puts all of the repacked poses created in the member variable vector.
//
void
SetReturningPackRotamersMover::apply( pose::Pose & pose ) {

	// jec update_residue_neighbors() required to update EnergyGraph (ensures graph_state == GOOD) when calling Interface.cc
	pose.update_residue_neighbors();

	// in case PackerTask was not generated locally, verify compatibility with pose
	assert( task_is_valid( pose ) );

	// get rotamers, energies
	this->setup( pose );

	PackerEnergy best_energy(0.);
	pose::Pose best_pose;
	best_pose = pose;

	for ( Size run(1); run <= ndruns_; ++run ) {
		// run SimAnnealer
		PackerEnergy packer_energy( this->run( pose ) );
		// Real const score( scorefxn_( pose ) ); another option for deciding which is the 'best' result
		if ( run == 1 || packer_energy < best_energy ) {
			best_pose = pose;
			best_energy = packer_energy;
		}
		TR << "run " << run << ", packer energy: " << packer_energy << std::endl;
		repacked_poses_[ run ] = pose;

	}
	if ( ndruns_ > 1 )
		pose = best_pose;
}


//
// @begin get_repacked_poses
//
// @brief
// Copies the poses in repacked_poses into the passed in vector reference.
//
void
SetReturningPackRotamersMover::get_repacked_poses( utility::vector1< core::pose::Pose > & v ) {
	v = repacked_poses_;
}

//
// @begin output_repacked_poses
//
// @brief
// Writes out all of the poses in the repacked poses vector to PDB files.
//
void
SetReturningPackRotamersMover::output_repacked_poses( std::string filename_prefix ) {

	for ( Size ii=1; ii <= repacked_poses_.size(); ++ii ) {
		// output repacked structure with the vector index in the name
		std::string filename = filename_prefix + "." + ObjexxFCL::fmt::I( 3, 3, ii ) + ".pdb";
		repacked_poses_[ ii ].dump_scored_pdb( filename, *(score_function()) );
	}

}


} // moves
} // protocols


