// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file protocols/moves/SidechainMover.hh
/// @brief definition of SidechainMover class and functions
/// @author Colin A. Smith (colin.smith@ucsf.edu)


#ifndef INCLUDED_protocols_moves_SidechainMover_HH
#define INCLUDED_protocols_moves_SidechainMover_HH

// Unit Headers
#include <protocols/moves/SidechainMover.fwd.hh>

// Protocols Headers
#include <protocols/moves/Mover.hh>

// Core Headers
#include <core/pack/task/TaskFactory.hh>
#include <core/pack/task/PackerTask.fwd.hh>
#include <core/pose/Pose.fwd.hh>
#include <core/scoring/dunbrack/RotamerLibrary.fwd.hh>
#include <core/scoring/dunbrack/RotamerLibraryScratchSpace.fwd.hh>

// Utility Headers
#include <utility/vector0.hh>
#include <utility/vector1.hh>
#include <numeric/conversions.hh>

namespace protocols {
namespace moves {


/// @brief class for non-discrete side chain sampling using Dunbrack rotamer probabilities/distributions
class SidechainMover : public Mover {

public:

	/// @brief default constructor
	SidechainMover();

	/// @brief constructor with user provided rotamer library
	SidechainMover(
		core::scoring::dunbrack::RotamerLibrary const & rotamer_library
	);

	~SidechainMover();

	/// @brief initialize the packer task if necessary
	void
	init_task(
		core::pose::Pose & pose
	);

	/// @brief apply a sidechain move to a Pose object
	void
	apply(
		core::pose::Pose & pose
	);

	/// @brief test the backrub move
	void
	test_move(
		core::pose::Pose &
	);

	/// @brief idealize sidechains that might be altered
	void
	idealize_sidechains(
		core::pose::Pose & pose
	);

	/// @brief get the rotamer library
	core::scoring::dunbrack::RotamerLibrary const &
	rotamer_library() const;

	/// @brief get the task factory
	core::pack::task::TaskFactoryCOP
	task_factory() const;

	/// @brief set the task factory
	void
	set_task_factory(
		core::pack::task::TaskFactoryCOP task_factory
	);

	/// @brief get the packer task
	core::pack::task::PackerTaskCOP
	task() const;

	/// @brief set the task
	void
	set_task(
		core::pack::task::PackerTaskCOP task
	);

	/// @brief get the probability of uniformly sampling chi angles
	core::Real
	prob_uniform() const;

	/// @brief set the probability of uniformly sampling chi angles
	void
	set_prob_uniform(
		core::Real prob_uniform
	);

	/// @brief get the number of chi angles sampled in the last move
	core::Size
	last_nchi() const;

	/// @brief get whether the last move mutated the residue
	bool
	last_mutation() const;

	/// @brief get whether the last move used uniform chi sampling
	bool
	last_uniform() const;

	/// @brief update string describing the move type
	void
	update_type();

private:

	core::scoring::dunbrack::RotamerLibrary const & rotamer_library_;
	core::pack::task::TaskFactoryCOP task_factory_;
	core::pack::task::PackerTaskCOP task_;
	utility::vector1<core::Size> packed_residues_;
	core::Real prob_uniform_;
	utility::vector1<core::Real> last_chi_angles_;
	core::Size last_nchi_;
	bool last_mutation_;
	bool last_uniform_;
	core::scoring::dunbrack::RotamerLibraryScratchSpaceOP scratch_;
};

} // moves
} // protocols

#endif
