// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file TrialMover
/// @brief performs a move and accepts it according to Monte Carlo accept/reject criterion.
/// @author Monica Berrondo

#include <protocols/moves/TrialMover.hh>

// Rosetta Headers
#include <core/pose/Pose.hh>
#include <core/conformation/Residue.hh>
#include <core/scoring/Energies.hh>
#include <core/util/basic.hh>
#include <core/util/Tracer.hh>

#include <protocols/moves/MonteCarlo.hh>
#include <protocols/moves/Mover.hh>

// ObjexxFCL Headers
#include <ObjexxFCL/formatted.o.hh>

// Random number generator
#include <numeric/random/random.hh>

core::util::Tracer tr("protocols.TrialMover");

//
#include <string>

namespace protocols {
namespace moves {

using namespace core;

/// @begin TrialMover::apply()
/// @brief:
/// 	the apply function for a trial
///	@detailed:
///		the trial object is created with an mc object
///		the mover is applied before doing an mc.boltzmann
///
///	@author: Monica Berrondo
void TrialMover::apply( pose::Pose & pose )
{
	using scoring::total_score;

	/// get the initial scores
	if ( keep_stats_type() == all_stats ) {
		stats_.add_score( mc_->last_accepted_score() ); ///< initial_last_accepted_score
		stats_.add_score( pose.energies().total_energy() ); ///< initial_pose_score
	}

	/// make the move
	mover_->apply( pose );

	// if ( keep_stats_type() == all_stats ) { //// score and get residue energies
	// Stupid and wasteful.  The structure will be scored inside mc_->boltzman.  mc_->score_function()( pose );
	// Unneccessary since revision 23846 --- mc_->score_function().accumulate_residue_total_energies( pose );
	// WAIT FOR IT. stats_.add_score( pose.energies().total_energy() ); ///< score_after_move
	// }

	/// test if MC accepts or rejects it
	bool accepted_move = mc_->boltzmann( pose, mover_->type() );

	if ( keep_stats_type() == all_stats ) {
		stats_.add_score( mc_->total_score_of_last_considered_pose() );
	}

	if ( stats_type_ <= accept_reject ) {
		stats_.accepted( accepted_move );
	}
	if ( keep_stats_type() > no_stats ) {
		stats_.print( mc_, mover_->type() );
	}
}

Real TrialMover::acceptance_rate() const
{
	tr << "Acceptance rate: " << stats_.acceptance_rate() << std::endl;
	return stats_.acceptance_rate();
}

int TrialMover::num_accepts() const
{
	return stats_.num_accepted();
}

// sets the input pose also for the contained mover (barak)
void TrialMover::set_input_pose( PoseCOP pose )
{
	this->Mover::set_input_pose( pose );
	if(mover_)
		mover_->set_input_pose( pose );
}

// sets the native pose also for the contained mover (barak)
void TrialMover::set_native_pose( PoseCOP pose )
{
	this->Mover::set_native_pose( pose );
	if(mover_)
		mover_->set_native_pose( pose );
}


}  // namespace moves
}  // namespace protocols
